/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2017
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: tree.js
 *  
 *  Description:
 *    Tree plugin
 *
 *  Author: Judy Evans
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  01-MAY-2017 EVANSJA Initial version
 *  08-AUG-2017 EVANSJA pr50345 - Add connect_id to the genlink URLs.
 *  18-SEP-2017 EVANSJA pr50432 - Move common javascript to util.js
 *  28-SEP-2017 EVANSJA pr50438a- Use uifcomponent for Tree instead of jstree.
 *  16-May-2019 FLANAGEJ pr51553:  Support HTML5 for Collection Editor
 *  24-MAY-2019 DOWMW    v930pr50376b - Change logic for End of Transmission
 *  10-Jun-2019 EVANSJA pr51508c: Add connect_id to xmljson to get proper task_idx.
 *  -
*/

// iptreevw.h
// Constants for TreeView $datatype
var TV_NONE = 0;   // Empty tree
var TV_MENU = 1;   // Menu selections
var TV_FILES = 2;  // Files
var TV_XML = 3;    // XML source
var TV_SYSVAR = 4; // System Variables
var TV_PRG_COL =  20; // Collections
var TV_PRG_JOB =  21; // Jobs
var TV_PRG_ALL =  22; // All programs
var TV_PRG_CALL = 23; // Call trees
var TV_PRG_PC =   24; // PC programs
var TV_PRG_XML =  25; // XML defined programs
var TV_PRG_PCVR = 26; // PC&VR programs
var myid = 0;

// Private functions 
// Create control only when PMON Server is ready
function tree_ServerReady(data) {
  // Initialize control data.
  tree_InitCtlData(data);

  // Create control.
  tree_CreateCtl(data);

  // Initialize events.
  tree_InitRobotEvent(true, data);

} // tree_ServerReady

// New instance of control.
function tree_NPP_New(data) {

  data.DataType = parseInt(data.DataType);

  SetCommonParams(data);
  if (data.IsDialog) {
    tree_ServerReady(data);
  }
  else { 
    top.rpcmc_getTaskIdx(data.fDeviceId, function(taskIdx) { 
      data.fTaskIdx = taskIdx;
      if (undefined != window.tview) {
        if (undefined != window.tview.PaneId) {
          window.tview.PaneId = taskIdx;
        }
      }

      // Complete after top.rpcmc_getTaskIdx completes.
      tree_ServerReady(data);
    });
  }

} // tree_NPP_New

// Destroy instance of control.
function tree_NPP_Destroy(data) {

  // Uninitialize events.
  tree_InitRobotEvent(false, data);

  // Delete control.
  tree_DeleteCtl(data);

} // tree_NPP_Destroy

// Private functions

// Set the border.
function tree_SetBorder(data) {

  var border = parseInt(data.Border);
  if (border > 0) {
    data.$this.css({"border-style":"solid", "border-width":"1px", "border-color":"gray"});
  }
  else {
    data.$this.css("border-style", "none");
  }

} // tree_SetBorder

// Update Control.
function tree_UpdateCtl(data) {


} // tree_UpdateCtl

// Initialize or uninitialize pipe event.
function tree_InitPipeEvent(init, data) {

  // Notify PMON to start/stop monitoring our pipe
  if (init) {
    top.jQuery.filelis.bind('PipeEvent', data, tree_HandlePipeEvent); // Attach handler for PipeEvent
    top.rpcmc_startPipeMonitor(data.Pipe, NOT_A_SHORT); // Start PMON monitor for our pipe
  }
  else {
    top.rpcmc_stopPipeMonitor(data.Pipe); // Stop PMON monitor for our pipe
    top.jQuery.filelis.unbind('PipeEvent', tree_HandlePipeEvent); // Detach handler for PipeEvent.
  }
} // tree_InitPipeEvent

// Replace any indirection with actual value.
function tree_IndirectReplace(data) {

  var l_ind;
  var pos;

 $.each(data, function( argn, argv ) { 
    if (typeof argv !== 'string') {
      return;
    }
    // Value contain !PaneId?
    if ((pos = argv.toLowerCase().indexOf("!paneid")) >= 0) {
      argv = argv.substring(0, pos) + data.fTaskIdx + argv.substring(pos+7);
      data[argn] = argv;
    }
  });
} // tree_IndirectReplace

// Initialize Control Data.
function tree_InitCtlData(data) {

  // Process parameters.
  // Name not supported
  // Border
  // Pipe
  // Scene
  // SubPane
  // Verbose not supported
  // ExecConnectId not required (only for ActiveX Controls)

  data.Pipe = data.Pipe.toUpperCase();
  tree_IndirectReplace(data);

  data.SubPane = parseInt(data.SubPane);

  if (data.BackColor == "") {
    data.BackColor = data.InvisibleColor;
  }
  else {
    data.BackColor = translateColor(data.BackColor);
  }
  if (data.ForeColor == "") {
    data.ForeColor = data.TextColor;
  }
  else {
    data.ForeColor = translateColor(data.ForeColor);
  }

} // tree_InitCtlData

// Initialize or uninitialize events for each type.
function tree_InitRobotEvent(init, data) {

  // Start/stop the Pipe Event.
  if (data.Pipe != "") {
    tree_InitPipeEvent(init, data);
  }

} // tree_InitRobotEvent

// Create Control.
function tree_CreateCtl(data) {

  data.$this.css("display", "inline-block");
  if (data.width.indexOf("%") >= 0) {
    data.$this.css("width", data.width);
  }
  else {
    data.$this.css("width", data.width + "px");
  }
  if (data.height.indexOf("%") >= 0) {
    data.$this.css("height", data.height);
  }
  else {
    data.$this.css("height", data.height + "px");
  }
  data.$this.css("vertical-align", "top");
  tree_SetBorder(data);
  SetColors(data);
  SetFont(data);

  if ((data.DataType == TV_XML) || (data.DataType >= TV_PRG_COL)) {
    tree_InitTreeFile(data);
  }
  else {
    data.$this.html('N/A');
  }

  // Attach handler for mousedown event.
  data.$this.bind("mousedown", data, tree_HandleMouseDownEvent);

  // Attach handler for mouseup event.
  data.$this.bind("mouseup", data, tree_HandleMouseUpEvent);

  // Attach handler for mouseout event.
  data.$this.bind("mouseout", data, tree_HandleMouseUpEvent);

} // tree_CreateCtl

// Handle Control events.
function tree_CtlEvent(data) {

  var sb_state;

  if (data.IsEcho) {
    return;  
  }

} // tree_CtlEvent

// Delete Control Data.
function tree_DeleteCtl(data) {

} // tree_DeleteCtl

// Private functions

function tree_HandlePipeEvent(event, file, buffer) {
  event.preventDefault();
  var data = event.data || event;
  if (file == data.Pipe) {
    if (buffer.length > 0) {
      /* Replace single quote with double quote */
      buffer = buffer.replace(/'/g, '"');
      /* remove whitespace before and after equal sign */
      buffer = buffer.replace(/\s*=\s*/g, '=');
      var lineData = buffer.match(/(?:[^\s"]+|"[^"]*")+/g)
      for (var i = 0; i < lineData.length; i++) {
        if (lineData[i]) {
          var tempLineData = lineData[i].trim();
          if (tempLineData.indexOf('=') > 0) {
            /* split the command into individual elements */
            var subData = tempLineData.split('=');
            /* if xml command in the pipe is "Refresh", refresh the tree view */
            if (subData[0].toLowerCase() === "refresh") {
              $.getJSON('/softpart/xmljson?type=' + data.DataType + '&id=uiftree_json&connect_id=' + top.g_connect_id, function(json) {
                document.getElementById(myid).refresh(json, "", false);
              });
            }
          }
        }
      }
    }
    //alert('Pipe Event ' + buffer);
  }
  return true;
} // tree_HandlePipeEvent

function tree_HandleMouseDownEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  tree_CtlEvent(data);
  return true;
} // tree_HandleMouseDownEvent

function tree_HandleMouseUpEvent(event) {
  event.preventDefault();
  return true;
} // tree_HandleMouseUpEvent

// Create a Tree from a File
function tree_InitTreeFile(data) {
  /* set a global myid for use in other functions */
  myid = data.$this[0].id;
  UIFComponents.treeSelect.init(document, myid, {});
  if (data.DataType == TV_XML) {
    if (data.XMLFile === '') {
      data.$this.html('data.XMLFile is not specified');
      return;
    }
  $.getJSON('/softpart/xmljson?file=' + data.XMLFile + '&id=uiftree&connect_id=' + top.g_connect_id, function(json) {
    document.getElementById(myid).refresh(json, "", false);
    document.getElementById(myid).config.onclick = nodeOnClick;
  });
  function nodeOnClick(root, props) {
    if (data.IsEcho) {
      return;  
    }
    if (typeof props.value === 'string') {
      if ((top.g_connect_id) && (pos = props.value.indexOf('/softpart/genlink?') == 0)) {
        var newstr;
        newstr = props.value.substring(0, 18) + 'connect_id=' + top.g_connect_id + '&' + props.value.substring(18);
        // No response is sent back from this request.
        $.get(newstr, function() { });
      }
      else {
        // No response is sent back from this request.
        $.get(props.value, function() { });
      }
    }
  }
  }
  else if (data.DataType >= TV_PRG_COL) {
    $.getJSON('/softpart/xmljson?type=' + data.DataType + '&id=uiftree_json&connect_id=' + top.g_connect_id, function(json) {
      document.getElementById(myid).refresh(json, "", false);
      document.getElementById(myid).config.onclick = selectOnClick;
    });

    function selectOnClick(root, props) {
      if (data.IsEcho) {
        return;  
      }
      if (typeof props.value === 'string') {
        if (undefined != window.tview) {
          if (undefined != window.tview.SelectedText) {
            window.tview.SelectedText = props.value;
          }
        }
      }
    }
  } 
} // tree_InitTreeFile

