/**********************************************************************
 * PC Editor script
 * for dialog.
 * Created: 2017/04/25
 * Modified:2018/12/13 for Operator Panel (Proting from PCEditor)
 */
/* === Dialog definition === */
/*
  = This table  : Definition data for dialog construction.
    = id: Dialog ID
    = titleId: Title ID of Dialog
    = titleRes: ID for title when referring to the language resource table.
    = dlgTextId: Element ID of body text.
    = textRes: ID for body message when referring to the language resource table.
    = append : before=append before msgForm, after=append after msgForm, no=no append.
    = msgForm: An HTML element when a form is required in the body of the dialog.
	= defaultId: The ID of the element that sets the default focus.
	= tabTable: Transition table when input tab key.
    = btnDef: Button element definition.
	  = type: ok = Positive choice. ng = Negative choice. cancel = Cancel.
	          Use for class name. (add 'btn')(ex: .btnok)
      = id: Button id.
	  = face: Button face resource id string.
	  = call: Callback function when button is selected.
*/
var dialogDefinitionTable = [
	{
		id: 'confcycledlg',		// Dialog ID
		titleId: 'dlgttlconf',  	// Dialog Title
		titleRes: 'dlgTtlConf',		// Dialog Title resource id
		dlgTextId: 'dlgmsgtext',	// Dialog Text ID
		textRes: '',	// Text resource id
		append: 'no',	// text append no/before/after
		msgForm: '',	// append data HTML
		defaultId: 'btnconfng',	// default element id.
		tabTable: ['btnconfok', 'btnconfng'],
		btnDef: [		// Button Definition. {type(ok/ng/cancel), id, face, callback}
			{ type: 'ok', bgcol: 'nocancel', id: 'btnconfok', face: 'dlgBtnYes', call: afterConfCycleStart },
			{ type: 'ng', bgcol: 'okyes', id: 'btnconfng', face: 'dlgBtnNo', call: null }
		]
	}
];

/* == Attention ID of Move Dialog == */
var MVDLG_ATTENTIONID = '#pagearea';

/* == Pickup message from resource.
      If nothing, resource id is output message text. == */
function pickupDialogMessage(resId, dlgDef) {
	var dlgMsg = "";
	if ((resId.length > 0) || (dlgDef.textRes.length > 0)) {
		dlgMsg = resources[(resId.length > 0) ? resId : dlgDef.textRes];
		if ((dlgMsg == undefined) || (dlgMsg.length == 0)) {
			dlgMsg = resId;
		}
	}
	return dlgMsg;
}
/* === Dialog configure === */
function configDialog(dlgId, textResId) {
	// Search dialog define
	var dlgDef = null;
	for (var i = 0; i < dialogDefinitionTable.length; i++) {
		dlgDef = dialogDefinitionTable[i];
		var idStr = '#' + dlgDef.id;
		if (idStr === dlgId) {	// target dialog?
			break;
		}
	}
	if (dlgDef == null) {
		console.log("Illega Dialog ID:" + dlgId);
		return null;
	}
	// Make dialog
	var dlgConf = $('<div>',
		{ id: dlgDef.id, 'class': 'modaldlg' });
	var dlgTtl = $('<div>',
		{ id: dlgDef.titleId, 'class': 'dlgtitlebar' });
	dlgTtl.html(resources[dlgDef.titleRes]);
	var ttlCancel = $('<div>', { id: 'dlgTtlCancel', 'class': 'dlgttlbarbtn' });
	var ttlBtn = $('<img>',
		{ id: 'ttlCancelBtn', 'src': 'softpanel/images/closeicon.png', 'class': 'dlgttlbtnimg' });
	ttlCancel.append(ttlBtn);
	dlgTtl.append(ttlCancel);
	dlgConf.append(dlgTtl);
	var dlgMsg = $('<p>');
	dlgMsg.attr('id', dlgDef.dlgTextId);
	dlgMsg.addClass('dlgcontent');
	var innerMsg = "";
	if ((dlgDef.append == 'no') || (dlgDef.append == 'before')) {
		innerMsg = pickupDialogMessage(textResId, dlgDef);
	}
	if ((dlgDef.msgForm != undefined) && (dlgDef.msgForm.length > 0)) {
		innerMsg += dlgDef.msgForm;
	}
	if (dlgDef.append == 'after') {
		innerMsg += pickupDialogMessage(textResId, dlgDef);
	}
	dlgMsg.html(innerMsg);
	dlgConf.append(dlgMsg);
	var dlgBtn = $('<div align="center" style="clear: both;">');
	var dlgForm = $('<form id="dlgbtnform">');
	for (var j = 0; j < dlgDef.btnDef.length; j++) {	// Yes/No/OK/Cancel
		var btnClass = "dlgbutton";
		btnClass += (' ' + dlgDef.btnDef[j].bgcol);
		btnClass += " btn" + dlgDef.btnDef[j].type;
		var btnObj = $('<button>',
			{ type: 'button', id: dlgDef.btnDef[j].id, 'class': btnClass, value: resources[dlgDef.btnDef[j].face] });
		btnObj.attr({ 'call': j, tabindex: (j + 1) });
		if (dlgDef.btnDef[j].disabled) {
			btnObj.prop('disabled', true);
		}
		btnObj.html(resources[dlgDef.btnDef[j].face]);
		dlgForm.append(btnObj);
	}
	dlgBtn.append(dlgForm);
	dlgConf.append(dlgBtn);
	$('body').append(dlgConf);
	return dlgDef;
}

var dragDialog = false;	// Dragging the header of the dialog.
function isDragDialog() {
	return dragDialog;
}
var prevDialogPos = { x: 0, y: 0 };	// Previous Dialog position.
var crntDialogPos = { x: 0, y: 0 };	// Current Dialog position.
// Mouse down on Dialog header.
function downDialogBox(evt) {
	var tempClient = getEventClient(evt);
	prevDialogPos.x = tempClient.x;
	prevDialogPos.y = tempClient.y;
	dragDialog = true;
}
// Mouse up on Dialog header.
function upDialogBox(evt) {
	dragDialog = false;
}
// Mouse move on Dialog header.
function moveDialogBox(evt) {
	var tempClient = getEventClient(evt);
	crntDialogPos.x = tempClient.x;
	crntDialogPos.y = tempClient.y;
	var diffX = crntDialogPos.x - prevDialogPos.x;
	var diffY = crntDialogPos.y - prevDialogPos.y;
	prevDialogPos.x = crntDialogPos.x;
	prevDialogPos.y = crntDialogPos.y;
	var crntT = evt.target.parentElement.offsetTop;
	var crntL = evt.target.parentElement.offsetLeft;
	var crntH = evt.target.parentElement.offsetHeight;
	var crntW = evt.target.parentElement.offsetWidth;
	var dlgIdStr = '#' + evt.target.parentElement.id;
	var maxT = $(MVDLG_ATTENTIONID).height();
	var maxL = $(MVDLG_ATTENTIONID).width();
	crntT += diffY;
	if (crntT < 0) {
		crntT = 0;
	} else if ((crntT + crntH) > maxT) {
		crntT = (maxT - crntH);
	}
	crntL += diffX;
	if (crntL < 0) {
		crntL = 0;
	} else if ((crntL + crntW) > maxL) {
		crntL = (maxL - crntW);
	}
	$(dlgIdStr).css({ "left": crntL + "px", "top": crntT + "px" });
}
// Close Dialog
function closeDialog(dlgId) {
	$(dlgId).hide();
	$(dlgId).remove();
	removeOverlayScreen('dialog');
}

// In order to make the operation possible, remove the screen.
function removeOverlayScreen(additionalID) {
	$('#screenoverlay-' + additionalID).hide();
	$('#screenoverlay-' + additionalID).remove();
}
// In order to make operation impossible, the screen is put up.
function putOverlayScreen(additionalID, targetScr) {
	var overlayCls = 'screenoverlay ' + targetScr;
	var overlayId = 'screenoverlay-' + additionalID;
	if ($('#' + overlayId).length == 0) {    // If it does not already exist, display overlay.
		$('body').append('<div id="' + overlayId + '" class="' + overlayCls + '"></div>');
		$('#' + overlayId).show();
	}
}

/**
 * Dialog TAB key control
 */
function dlgTabControl(dlgTabIdxTbl, event) {
	var crntElemId = event.target.id;
	var crntElemIndex = 0;
	for (i = 0; i < dlgTabIdxTbl.length; i++) {
		if (crntElemId == dlgTabIdxTbl[i]) {
			crntElemIndex = i;
			break;
		}
	}
	var nextElemIndex = (event.shiftKey) ? crntElemIndex - 1 : crntElemIndex + 1;
	if (nextElemIndex >= dlgTabIdxTbl.length) {
		nextElemIndex = 0;
	}
	else if (nextElemIndex < 0) {
		nextElemIndex = dlgTabIdxTbl.length - 1;
	}
	$('#' + dlgTabIdxTbl[nextElemIndex]).focus();
	return false;
}

/* === Dialog setting === */
var defineDialog = {};
$(function () {
	// Dialog process definition.
	defineDialog = {
		// Open dialog.
		// dlgArgs : .btnId = focus off target ID,
		//           .dlgId = Open Dialog ID,
		//           .textResId = Non-default resource ID
		open: function (dlgArgs) {
			dlgDefTbl = configDialog(dlgArgs.dlgId, dlgArgs.textResId);
			if (dlgArgs.btnId != null) {	// specified button id,
				$(dlgArgs.btnId).blur();	// focus off
			}
			putOverlayScreen('dialog', 'forpopup');
			centeringDialog(dlgArgs.dlgId);
			$(dlgArgs.dlgId).fadeIn("middle");
			// If specified default element id, set focus to default element id.
			var defElemId = dlgDefTbl.defaultId;
			if ((defElemId != null) && (defElemId.length > 0)) {
				if ($('#' + defElemId).is(':visible')) {
					$('#' + defElemId).focus();
				}
			} else {
				$('.btnok').focus();
			}
			// Processing of key input in dialog.
			$('#dlgbtnform').on('keydown', function (event) {
				if (event.which == KEYCODE_ENTER) {
					var focused = document.activeElement;
					$('#' + focused.id).click();
					return false;
				} else if (event.which == KEYCODE_ESC) {
					if ($('.btncancel').length > 0) {
						$('.btncancel').click();
					} else if ($('.btnng').length > 0) {
						$('.btnng').click();
					} else {
						$('.btnok').click();
					}
					return false;
				} else if (event.which == KEYCODE_TAB) {
					var dlgTabIdxTbl = dlgDefTbl.tabTable;
					dlgTabControl(dlgTabIdxTbl, event);
					return false;
				}
			});
			// Dialog button process
			// OK button / Yes button
			$('.btnok').on('click', function (event) {
				if (event.target.offsetParent == null) {
					return false;	// Discard the event of the button itself.
				}
				$('.btnok').blur();
				var dlgId = '#' + event.target.offsetParent.id;
				closeDialog(dlgId);
				// Call the OK method.
				var funcNum = $(this).attr('call');
				var callfunc = dlgDefTbl.btnDef[funcNum].call;
				var callParam = {};
				if (callfunc != undefined) {
					callfunc(callParam);
				}
				return false;
			});
			// Cancel button
			$('.btncancel').on('click', function (event) {
				if (event.target.offsetParent == null) {
					return false;
				}
				$('.btncancel').blur();
				var dlgId = '#' + event.target.offsetParent.id;
				closeDialog(dlgId);
				// Call the Cancel method.
				var funcNum = $(this).attr('call');
				var callfunc = dlgDefTbl.btnDef[funcNum].call;
				var callParam = {};
				if (callfunc != undefined) {
					callfunc(callParam);
				}
				return false;
			});
			// Title cancel button
			$('#dlgTtlCancel').on('click', function (event) {
				if ($('.btncancel').length > 0) {
					$('.btncancel').click();
					return;
				}
				if ($('.btnng').length > 0) {
					$('.btnng').click();
					return;
				}
				$('.btnok').click();
			});
			// No button
			$('.btnng').on('click', function (event) {
				var dlgId = '#' + event.target.offsetParent.id;
				closeDialog(dlgId);
				// Call the NG method.
				var funcNum = $(this).attr('call');
				var callfunc = dlgDefTbl.btnDef[funcNum].call;
				var callParam = {};
				if (callfunc != undefined) {
					callfunc(callParam);
				}
				return false;
			});
		}
	};

	/**
	* Define of require dialog process.
	**/
	// Centering Dialog window
	function centeringDialog(dlgId) {
		// Get content width/height
		var winX = $(window).innerWidth();
		var winY = $(window).innerHeight();
		var dlgW = $(dlgId).outerWidth();
		var dlgH = $(dlgId).outerHeight();
		// execute centering
		$(dlgId).css({ "left": ((winX - dlgW) / 2) + "px", "top": ((winY - (dlgH * 1.25)) / 2) + "px" });
	};

});

// Mouse Down on window
$(document).on('mousedown touchstart', function (evt) {
	if (evt.target.className == 'dlgtitlebar') {
		downDialogBox(evt);
		evt.preventDefault();
	}
});
// Mouse Up on window
$(document).on('mouseup touchend', function (evt) {
	if (evt.target.className == 'dlgtitlebar') {
		upDialogBox(evt);
		evt.preventDefault();
	}
});
// Mouse Move on window
$(document).on('mousemove touchmove', function (evt) {
	if (evt.target.className == 'dlgtitlebar') {
		if (isDragDialog()) {
			moveDialogBox(evt);
			evt.preventDefault();
		}
	}
});
// Mouse Out from window
$(document).on('mouseout', function (evt) {
	if (evt.target.className == 'dlgtitlebar') {
		upDialogBox(evt);
		evt.preventDefault();
	}
});

/** ===
 * Definition of methods that iPad clientX/Y convert.
 * (Porting from pceditor_util.js)
 **/
function getEventClient(event) {
	var evClient = { x: 0, y: 0 };
	if ((isTablet != undefined) && (isTablet)) {
		if (event.changedTouches == undefined) {	// Maybe event is click.
			evClient.x = event.clientX;
			evClient.y = event.clientY;
		} else {	// Maybe event is mouse event.
			var touchObject = event.changedTouches[0];
			evClient.x = touchObject.pageX;
			evClient.y = touchObject.pageY;
		}
	} else {
		evClient.x = event.clientX;
		evClient.y = event.clientY;
	}
	return evClient;
}
