/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2020
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: jquery.tabtpin.js
 *  
 *  Description:
 *    Hardkey plugin which sets TPIN
 * 
 *  Modification history:
 *  29-JAN-2020 Created by FSI for tablet TP based on jquery.spin.js.
 *  -
*/

//  Top web page must contain the following
//  <!--#include file=frh:\jcgtp\tpin.hdr -->
//
//  Usage
//  $('#j1p').tpin({ key_name: '+J1', tptext: '+X<br>(J1)', key_class: 'tpjog' });
//


; (function ($) {

  $.fn.tpin = function (settings) {
    $(this).bind('tpin', settings, function () { });
  };

  var key_touches = [];
  var touch_x;
  var touch_y;
  var delta_x;
  var delta_y;
  var visCngStr = "";
  var hiddenStr = "";

  var tpin = $.event.special.tpin = {
    // tpin plugin defaults
    key_name: '',
    key_class: 'tpbutton',
    key_value: 0,
    tptext: '',
    tpimage: '',
    $this: null,

    // Called each time an element does bind
    add: function (handleObj) {
      handleObj.data = $.extend({
        key_name: tpin.key_name,
        key_class: tpin.key_class,
        key_value: tpin.key_value,
        tptext: tpin.tptext,
        tpimage: tpin.tpimage,
        $this: $(this)
      }, handleObj.data || {});

      // Create button inside the element.
      var data = handleObj.data;
      data.$this.html('<button type="button" class="' + data.key_class + '">' + data.tptext + '</button>');
      if (data.tpimage != '') {
        $('button', data.$this).css('background-image', 'url("' + data.tpimage + '")');
        $('button', data.$this).css('background-position', 'center');
      }

      // Attach handler to button for click event.
      $('button', data.$this).click(function (event) {
        //rpcmc_rprintf('click');
        event.preventDefault();
        doClick(data);
      }); // click event

      // Attach handler to button for touchstart event.
      $('button', data.$this).bind('touchstart mousedown', function (event) {
        event.preventDefault();
        if (event.type == 'touchstart') {
          key_touches = event.originalEvent.touches;
          touch_x = event.originalEvent.touches[0].pageX;
          touch_y = event.originalEvent.touches[0].pageY;
          //rpcmc_rprintf('touchstart ' + touch_x + ',' + touch_y);
          doClick(data);
        }
        doPress(data);
      }); // touchstart event

      // Attach handler to button for touchend event.
      $('button', data.$this).bind('touchend mouseup', function (event) {
        event.preventDefault();
        if (event.type == 'touchend') {
          key_touches = event.originalEvent.touches;
          //rpcmc_rprintf('touchend');
        }
        doRelease(data);
      }); // touchend event

      // Attach handler to button for touchmove event.
      $('button', data.$this).bind('touchmove', function (event) {
        for (var i = 0; i < event.originalEvent.touches.length; i++){
          var trgtKey = event.originalEvent.touches[i];
          if(trgtKey.target.className == undefined) continue;
          delta_x = trgtKey.pageX;
          delta_y = trgtKey.pageY;
          // When the finger position is outside the target button, release the button.
          var btnElem = document.getElementById(trgtKey.target.parentElement.id);
          if(btnElem == null) continue;
          var btnRect = btnElem.getBoundingClientRect();
          var btnL = btnRect.left + window.pageXOffset;
          var btnR = btnRect.right + window.pageXOffset;
          var btnT = btnRect.top + window.pageYOffset;
          var btnB = btnRect.bottom + window.pageYOffset;
          if ((delta_x < btnL) ||
            (delta_x > btnR) ||
            (delta_y < btnT) ||
            (delta_y > btnB)) {
            event.preventDefault();
            if(data.$this[0].id == trgtKey.target.parentElement.id){
              doRelease(data);
            }
          }
        }
      }); // touchmove event

      // Display of a tooltip indicating buttons that can be double-clicked.
      if (top.isTabTP()) {
        // For TabletTP
        var targetElem = $('button', data.$this)[0];
        targetElem.addEventListener("pointerenter", function(event) {
          if (event.pointerType == "pen") doHover(data);
        });
        targetElem.addEventListener("pointerleave", function() {
          // Unconditionally remove tooltips if they exist.
          if (document.getElementById("ks_tooltip") != null) $('#ks_tooltip').remove();
        });
        // To process with "mouseleave" event if "pointerleve" event does not occur.
        targetElem.addEventListener("mouseleave", function() {
          // Unconditionally remove tooltips if they exist.
          if (document.getElementById("ks_tooltip") != null) $('#ks_tooltip').remove();
        });
      } else {
        $('button', data.$this).hover(function () {
          doHover(data);
        }, function () {
          // Mouse Out
          delTooltip(data);
        });
      }

      // Attach handler to button for mouseout event.
      $('button', data.$this).mouseout(function (event) {
        //rpcmc_rprintf('mouseout');
        event.preventDefault();
        doRelease(data);
      }); // mouseup event

      // Attach handler to button for touchcancel event.
      $('button', data.$this).bind('touchcancel', function (event) {
        key_touches = event.originalEvent.touches;
        //rpcmc_rprintf('touchcancel');
        event.preventDefault();
        doRelease(data);
      }); // touchcancel event

      // Set visibilitychange String
      if (typeof document.hidden !== "undefined") {
        hiddenStr = "hidden";
        visCngStr = "visibilitychange";
      } else if (typeof document.msHidden !== "undefined") {
        hiddenStr = "msHidden";
        visCngStr = "msvisibilitychange";
      } else if (typeof document.webkitHidden !== "undefined") {
        hiddenStr = "webkitHidden";
        visCngStr = "webkitvisibilitychange";
      }
      document.addEventListener(visCngStr, visChgFunc, false); // visibilitychange event
            
    }, // add
  

    // Called each time an element removes bind
    remove: function (handleObj) {
      handleObj.data = $.extend({
        key_name: tpin.key_name,
        key_class: tpin.key_class,
        key_value: tpin.key_value,
        tptext: tpin.tptext,
        tpimage: tpin.tpimage,
        $this: $(this)
      }, handleObj.data || {});

      // Release the key.
      var data = handleObj.data;
      if (data.key_value) {
        data.key_value = 0;
        if ($('button', data.$this).hasClass('tpenabled')) {
          // Before turning off Enable, we should release Deadman.
          // sendKey will send Enable off after a timer
          if (!(top.g_irprog && top.wait_key_open)) {
            // If the keysheet of iRProgrammer is switching, passed through.
            requestSendKey('Deadman-L', data.key_value);
          }
        }
        else {
          requestSendKey(data.key_name, data.key_value);
        }
      }

      // Detach all handlers for the button
      $('button', data.$this).unbind();

    } // remove
  };

  // Object used to determine double-click.
  var keyState = {
    SGL_INTERVAL: 200,
    DBL_INTERVAL: 150,
    STATE: {
      NONE: 0,
      RELEASED: 1, // The key is released.
      WAIT_SGLITVL: 2, // Waiting for click interval.
      SHIFTING: 3, // Only the shift button is pressed.
      SGL_CLICKED: 4, // The key is single-clicked.
      SGL_CLICKING: 5, // The key is being single-clicked.
      SFT_CLICKING: 6, // The key is being single-clicked with SHIFT.
      DBL_CLICKING: 7, // The key is being double-clicked.
      WAIT_DBLITVL: 8, // Waiting for shift operation.
      DBL_CLICKED: 9 // The key is double-clicked.
    },
    _state: 1,
    _SGL_timer: null,
    _DBL_timer: null,
    _key_data: {}, // The name of the key that was clicked.

    clear: function (timer) { // Initialize the object used to determine double-click.
      if (timer != undefined) {
        clearTimeout(timer);
      } else {
        this._state = this.STATE.RELEASED;
        clearTimeout(this._SGL_timer);
        clearTimeout(this._DBL_timer);
        this._key_data = {};
      }
    },

    // Controls the key state when a key is depressed.
    mousedown: function (key_data) {
      switch (this._state) {
        case this.STATE.RELEASED:
          if (key_data.key_name == "R-shift" || key_data.key_name == "L-shift") {
            this._state = this.STATE.SHIFTING;
            sendPress(key_data, this._state);
          } else {
            this._key_data = $.extend(true, {}, key_data);
            this._state = this.STATE.WAIT_SGLITVL;
            this._SGL_timer = setTimeout(function () {
              if (keyState._state == keyState.STATE.WAIT_SGLITVL) {
                keyState._state = keyState.STATE.SGL_CLICKING;
                sendPress(key_data, keyState._state);
              } else if (keyState._state == keyState.STATE.SGL_CLICKED) {
                sendPress(key_data, keyState._state);
                sendRelease(key_data, keyState._state);
                keyState.clear();
              } else {
                keyState.clear();
              }
            }, keyState.SGL_INTERVAL);
          }
          break;
        case this.STATE.SHIFTING:
          if (key_data.key_name == "R-shift" || key_data.key_name == "L-shift") {
            sendPress(key_data, this._state);
          } else {
            this._key_data = $.extend(true, {}, key_data);
            this._state = this.STATE.SFT_CLICKING;
            sendPress(key_data, this._state);
          }
          break;
        case this.STATE.SGL_CLICKED:
          if (this._key_data.key_name == key_data.key_name) {
            this.clear(this._SGL_timer);
            this._key_data = $.extend(true, {}, key_data);
            this._state = this.STATE.WAIT_DBLITVL;
            this._DBL_timer = setTimeout(function () {
              if (keyState._state == keyState.STATE.WAIT_DBLITVL) {
                keyState._state = keyState.STATE.DBL_CLICKING;
                sendPress(key_data, keyState._state);
              } else if (keyState._state == keyState.STATE.DBL_CLICKED) {
                sendPress(key_data, keyState._state);
                sendRelease(key_data, keyState._state);
                keyState.clear();
              } else {
                keyState.clear();
              }
            }, keyState.DBL_INTERVAL);
          } else {
            this.clear();
            this.mousedown(key_data);
          }
          break;
        case this.STATE.DBL_CLICKED:
          if (this._key_data.key_name == key_data.key_name) {
            this.clear(this._DBL_timer);
            sendPress(key_data, keyState._state);
            sendRelease(key_data, keyState._state);
            this._key_data = $.extend(true, {}, key_data);
            this._state = this.STATE.WAIT_SGLITVL;
            this._SGL_timer = setTimeout(function () {
              if (keyState._state == keyState.STATE.WAIT_SGLITVL) {
                keyState._state = keyState.STATE.SGL_CLICKING;
                sendPress(key_data, keyState._state);
              } else if (keyState._state == keyState.STATE.SGL_CLICKED) {
                sendPress(key_data, keyState._state);
                sendRelease(key_data, keyState._state);
                keyState.clear();
              } else {
                keyState.clear();
              }
            }, keyState.SGL_INTERVAL);
          } else {
            this.clear();
            this.mousedown(key_data);
          }
          break;
        default:
          // for example: Pressing a override button while double-clicking
          sendPress(key_data);
          break;
      }
    },

    // Controls the key state when the key is released.
    mouseup: function (key_data) {
      switch (this._state) {
        case this.STATE.SHIFTING:
          if (key_data.key_name == "R-shift" || key_data.key_name == "L-shift") {
            if (!($('.shiftkey').hasClass('tpshifted'))) {
              this.clear();
            }
            sendRelease(key_data, this._state);
          }
          break;
        case this.STATE.SFT_CLICKING:
          if (key_data.key_name == "R-shift" || key_data.key_name == "L-shift") {
            if (!($('.shiftkey').hasClass('tpshifted'))) {
              this._state = this.STATE.SGL_CLICKING;
            }
            sendRelease(key_data, this._state);
          } else {
            if (!($('button').hasClass('tpmotiondown')) && !($('button').hasClass('tpexecdown')) &&
                !($('button').hasClass('tpholddown')) && !($('button').hasClass('tpbutton_shiftabledown')) &&
                !($('button').hasClass('tpbutton_bl_shiftabledown')) && !($('button').hasClass('tpbutton_lbl_shiftabledown')) &&
                !($('button').hasClass('tpjogdown')) && !($('button').hasClass('tpbutton_fkeydown'))) {
              this._state = this.STATE.SHIFTING;
            }
            sendRelease(key_data, this._state);
          }
          break;
        case this.STATE.SGL_CLICKING:
          sendRelease(key_data, this._state);
          break;
        case this.STATE.WAIT_SGLITVL:
          if (this._key_data.key_name == key_data.key_name) {
            this._state = this.STATE.SGL_CLICKED;
          } else {
            sendRelease(key_data);
          }
          break;
        case this.STATE.WAIT_DBLITVL:
          this._state = this.STATE.DBL_CLICKED;
          sendPress(key_data, this._state);
          sendRelease(key_data, this._state);
          break;
        case this.STATE.DBL_CLICKING:
          if (this._key_data.key_name == key_data.key_name) {
            sendRelease(key_data, this._state);
          } else if (key_data.key_name == "R-shift" || key_data.key_name == "L-shift") {
            if (!($('.shiftkey').hasClass('tpshifted'))) {
              this._state = this.STATE.SGL_CLICKING;
            }
            sendRelease(key_data);
          } else {
            sendRelease(key_data); // for example: Pressing a override button while double-clicking
          }
          break;
        default:
          // Does not occur.
          sendRelease(key_data);
          break;
      }
    }
  };

  // Private functions
  // Button Click
  function doClick(data) {
    if ($('button', data.$this).hasClass('tpicon')) {
      $('button', data.$this).removeClass('tpicon');
      $('button', data.$this).addClass('tpiconed');
      data.key_value = 1;
      requestSendKey(data.key_name, data.key_value);
    }
    else if ($('button', data.$this).hasClass('tpiconed')) {
      $('button', data.$this).removeClass('tpiconed');
      $('button', data.$this).addClass('tpicon');
      data.key_value = 0;
      requestSendKey(data.key_name, data.key_value);
    }
    else if ($('button', data.$this).hasClass('tpenable')) {
      $('button', data.$this).removeClass('tpenable');
      $('button', data.$this).addClass('tpenabled');
      $('button', data.$this).css('background-image', 'url("tpenon.gif")');
      if (data.tpimage == 'off_switch.png') {
        $('button', data.$this).css('background-image', 'url("on_switch.png")');
      } else {
        $('button', data.$this).css('background-image', 'url("tpenon.gif")');
      }
      data.key_value = 1;
      requestSendKey(data.key_name, data.key_value);
      // sendKey will press Deadman, after turning on Enable
      if (top.g_irprog) {
        // Prevent tp-button bashing.
        $("#tpen").css({ "pointer-events": "none" });
        setTimeout(function () {
          $("#tpen").css({ "pointer-events": "auto" });
        }, 500);
      }
    }
    else if ($('button', data.$this).hasClass('tpenabled')) {
      $('button', data.$this).removeClass('tpenabled');
      $('button', data.$this).addClass('tpenable');
      $('button', data.$this).css('background-image', 'url("tpenoff.gif")');
      if (data.tpimage == 'on_switch.png') {
        $('button', data.$this).css('background-image', 'url("off_switch.png")');
      } else {
        $('button', data.$this).css('background-image', 'url("tpenoff.gif")');
      }
      // Before turning off Enable, we should release Deadman.
      // sendKey will send Enable off after a timer
      data.key_value = 0;
      requestSendKey('Deadman-L', data.key_value);
      if (top.g_irprog) {
        // Prevent tp-button bashing.
        $("#tpen").css({ "pointer-events": "none" });
        setTimeout(function () {
          $("#tpen").css({ "pointer-events": "auto" });
        }, 500);
      }
    }
  } // doClick

  // Button Press
  function doPress(data) {
    if ($('button', data.$this).hasClass('tpgroup')) {
      $('button', data.$this).removeClass('tpgroup');
      $('button', data.$this).addClass('tpgroupdown');
      sendPress(data);
    }
    else if ($('button', data.$this).hasClass('tpmotion')) {
      $('button', data.$this).removeClass('tpmotion');
      $('button', data.$this).addClass('tpmotiondown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpexec')) {
      $('button', data.$this).removeClass('tpexec');
      $('button', data.$this).addClass('tpexecdown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tphold')) {
      $('button', data.$this).removeClass('tphold');
      $('button', data.$this).addClass('tpholddown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_shiftable')) {
      $('button', data.$this).removeClass('tpbutton_shiftable');
      $('button', data.$this).addClass('tpbutton_shiftabledown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_bl_shiftable')) {
      $('button', data.$this).removeClass('tpbutton_bl_shiftable');
      $('button', data.$this).addClass('tpbutton_bl_shiftabledown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_lbl_shiftable')) {
      $('button', data.$this).removeClass('tpbutton_lbl_shiftable');
      $('button', data.$this).addClass('tpbutton_lbl_shiftabledown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpjog')) {
      $('button', data.$this).removeClass('tpjog');
      $('button', data.$this).addClass('tpjogdown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton')) {
      $('button', data.$this).removeClass('tpbutton');
      $('button', data.$this).addClass('tpbuttondown');
      sendPress(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_fkey')) {	
      $('button', data.$this).removeClass('tpbutton_fkey');
      $('button', data.$this).addClass('tpbutton_fkeydown');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('tpshift')) {
      $('button', data.$this).removeClass('tpshift');
      $('button', data.$this).addClass('tpshifted');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('fky_button')) {
      $('button', data.$this).removeClass('fky_button');
      $('button', data.$this).addClass('fky_button_down');
      keyState.mousedown(data);
    }
    else if ($('button', data.$this).hasClass('fky_button_right')) {
      $('button', data.$this).removeClass('fky_button_right');
      $('button', data.$this).addClass('fky_button_right_down');
      keyState.mousedown(data);
    }
  } // doPress

  // Send key for press.
  function sendPress(data, state) {
    if (typeof state === 'undefined') state = keyState.STATE.NONE;
    data.key_value = 1;
    if (state == keyState.STATE.DBL_CLICKING) {
      $('.shiftkey').removeClass('tpshift');
      $('.shiftkey').addClass('tpshifted');
      requestSendKey("R-shift", data.key_value);
      requestSendKey("L-shift", data.key_value);
      if ($('button', data.$this).hasClass('tpexecdown')){
        requestSendKey(data.key_name, data.key_value);
        data.key_value = 0;
      }
    }
    requestSendKey(data.key_name, data.key_value);
  }

  // Button Release
  function doRelease(data) {
    if ($('button', data.$this).hasClass('tpgroupdown')) {
      $('button', data.$this).removeClass('tpgroupdown');
      $('button', data.$this).addClass('tpgroup');
      sendRelease(data);
    }
    else if ($('button', data.$this).hasClass('tpmotiondown')) {
      $('button', data.$this).removeClass('tpmotiondown');
      $('button', data.$this).addClass('tpmotion');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpexecdown')) {
      $('button', data.$this).removeClass('tpexecdown');
      $('button', data.$this).addClass('tpexec');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpholddown')) {
      $('button', data.$this).removeClass('tpholddown');
      $('button', data.$this).addClass('tphold');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_shiftabledown')) {
      $('button', data.$this).removeClass('tpbutton_shiftabledown');
      $('button', data.$this).addClass('tpbutton_shiftable');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_bl_shiftabledown')) {
      $('button', data.$this).removeClass('tpbutton_bl_shiftabledown');
      $('button', data.$this).addClass('tpbutton_bl_shiftable');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_lbl_shiftabledown')) {
      $('button', data.$this).removeClass('tpbutton_lbl_shiftabledown');
      $('button', data.$this).addClass('tpbutton_lbl_shiftable');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpjogdown')) {
      $('button', data.$this).removeClass('tpjogdown');
      $('button', data.$this).addClass('tpjog');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('tpbuttondown')) {
      $('button', data.$this).removeClass('tpbuttondown');
      $('button', data.$this).addClass('tpbutton');
      sendRelease(data);
    }
    else if ($('button', data.$this).hasClass('tpbutton_fkeydown')) {
      $('button', data.$this).removeClass('tpbutton_fkeydown');
      $('button', data.$this).addClass('tpbutton_fkey');
      keyState.mouseup(data);
    }   
    else if ($('button', data.$this).hasClass('tpshifted')) {
      $('button', data.$this).removeClass('tpshifted');
      $('button', data.$this).addClass('tpshift');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('fky_button_down')) {
      $('button', data.$this).removeClass('fky_button_down');
      $('button', data.$this).addClass('fky_button');
      keyState.mouseup(data);
    }
    else if ($('button', data.$this).hasClass('fky_button_right_down')) {
      $('button', data.$this).removeClass('fky_button_right_down');
      $('button', data.$this).addClass('fky_button_right');
      keyState.mouseup(data);
    }
  } // doRelease

  // Send key for release.
  function sendRelease(data, state) {
    if (typeof state === 'undefined') state = keyState.STATE.NONE;
    data.key_value = 0;
    if (state == keyState.STATE.DBL_CLICKING) {
      $('.shiftkey').removeClass('tpshifted');
      $('.shiftkey').addClass('tpshift');
      requestSendKey("R-shift", data.key_value);
      requestSendKey("L-shift", data.key_value);
      keyState.clear();
    }
    requestSendKey(data.key_name, data.key_value);
    if (key_touches.length == 0 && state != keyState.STATE.DBL_CLICKED) releaseAllKeys();
  } // sendRelease

  // If there are any pressed keys, release them all.
  function releaseAllKeys() {
    var tpgroupdown = $.extend(true, [], document.getElementsByClassName('tpgroupdown'));
    for (var i = 0; i < tpgroupdown.length; i++) {
      var keyselector = '#' + tpgroupdown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpgroupdown');
      $(keyselector + " > button").addClass('tpgroup');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpmotiondown = $.extend(true, [], document.getElementsByClassName('tpmotiondown'));
    for (var i = 0; i < tpmotiondown.length; i++) {
      var keyselector = '#' + tpmotiondown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpmotiondown');
      $(keyselector + " > button").addClass('tpmotion');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpexecdown = $.extend(true, [], document.getElementsByClassName('tpexecdown'));
    for (var i = 0; i < tpexecdown.length; i++) {
      var keyselector = '#' + tpexecdown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpexecdown');
      $(keyselector + " > button").addClass('tpexec');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpholddown = $.extend(true, [], document.getElementsByClassName('tpholddown'));
    for (var i = 0; i < tpholddown.length; i++) {
      var keyselector = '#' + tpholddown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpholddown');
      $(keyselector + " > button").addClass('tphold');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpbutton_shiftabledown = $.extend(true, [], document.getElementsByClassName('tpbutton_shiftabledown'));
    for (var i = 0; i < tpbutton_shiftabledown.length; i++) {
      var keyselector = '#' + tpbutton_shiftabledown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpbutton_shiftabledown');
      $(keyselector + " > button").addClass('tpbutton_shiftable');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpbutton_bl_shiftabledown = $.extend(true, [], document.getElementsByClassName('tpbutton_bl_shiftabledown'));
    for (var i = 0; i < tpbutton_bl_shiftabledown.length; i++) {
      var keyselector = '#' + tpbutton_bl_shiftabledown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpbutton_bl_shiftabledown');
      $(keyselector + " > button").addClass('tpbutton_bl_shiftable');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpbutton_lbl_shiftabledown = $.extend(true, [], document.getElementsByClassName('tpbutton_lbl_shiftabledown'));
    for (var i = 0; i < tpbutton_lbl_shiftabledown.length; i++) {
      var keyselector = '#' + tpbutton_lbl_shiftabledown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpbutton_lbl_shiftabledown');
      $(keyselector + " > button").addClass('tpbutton_lbl_shiftable');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpjogdown = $.extend(true, [], document.getElementsByClassName('tpjogdown'));
    for (var i = 0; i < tpjogdown.length; i++) {
      var keyselector = '#' + tpjogdown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpjogdown');
      $(keyselector + " > button").addClass('tpjog');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpbuttondown = $.extend(true, [], document.getElementsByClassName('tpbuttondown'));
    for (var i = 0; i < tpbuttondown.length; i++) {
      var keyselector = '#' + tpbuttondown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpbuttondown');
      $(keyselector + " > button").addClass('tpbutton');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpbutton_fkeydown = $.extend(true, [], document.getElementsByClassName('tpbutton_fkeydown'));
    for (var i = 0; i < tpbutton_fkeydown.length; i++) {
      var keyselector = '#' + tpbutton_fkeydown[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpbutton_fkeydown');
      $(keyselector + " > button").addClass('tpbutton_fkey');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    var tpshifted = $.extend(true, [], document.getElementsByClassName('tpshifted'));
    for (var i = 0; i < tpshifted.length; i++) {
      var keyselector = '#' + tpshifted[i].parentElement.id;
      $(keyselector + " > button").removeClass('tpshifted');
      $(keyselector + " > button").addClass('tpshift');
      requestSendKey($(keyselector).attr('name'), 0);
    }
    keyState.clear();
  } // releaseAllKeys

  function requestSendKey(keyName, keyValue) {
    var UNSEND_SCREENS = /pmcdisp\.htm/;
    var UNSEND_KEYS = ["F1","F2","F3","F4","F5","Next"];
    var EXC_CLASS = ["fky_button","fky_button_down","fky_button_right","fky_button_right_down"];
    try {
      var actvScreen = top.mainfrm[top.getMyFrameId()].location.href;
      if(actvScreen.match(UNSEND_SCREENS) != null &&
        UNSEND_KEYS.indexOf(keyName) >= 0 &&
        EXC_CLASS.indexOf(keyState._key_data.key_class) < 0) {
        // no action
      } else {
        top.sendKey(keyName, keyValue);
      }
    } catch(e) {
      top.sendKey(keyName, keyValue);
    }
  }

  function doHover(data) {
    if ($('button', data.$this).hasClass('tpexec')
      || $('button', data.$this).hasClass('tpmotion')
      || $('button', data.$this).hasClass('tpbutton_shiftable')
      || $('button', data.$this).hasClass('tpbutton_bl_shiftable')
      || $('button', data.$this).hasClass('tpbutton_lbl_shiftable')
      || $('button', data.$this).hasClass('tpjog')
      || $('button', data.$this).hasClass('tphold')
      || $('button', data.$this).hasClass('tpbutton_fkey')) {
      dispTooltip(data);
    }
  }
  function dispTooltip(data) {
    if(document.getElementById("ks_tooltip") != null) $('#ks_tooltip').remove();
    var TOOLTIP_ARROW_HEIGHT = 11;
    var tgtPt = $(data.$this).offset();
    var refPt = $('#step').offset();
    var sheetRect = $('#tpkeysheet_area')[0].getBoundingClientRect();
    var maxWidth = 0;
    var tooltipPos = {};
    $(data.$this).append('<div id="ks_tooltip">' + resources['dlgMsgDCKey'] + '</div>');
    // Fix the position of the tooltip.
    if (tgtPt.left >= refPt.left) {
      maxWidth = tgtPt.left - sheetRect.left;
      tooltipPos.left = tgtPt.left - $('#ks_tooltip').outerWidth() + $(data.$this).width();
      if (tgtPt.top < refPt.top) {
        tooltipPos.top = tgtPt.top + ($(data.$this).height() + TOOLTIP_ARROW_HEIGHT);
        $("#ks_tooltip").addClass("ks_arrow_rt");
      } else {
        tooltipPos.top = tgtPt.top - ($('#ks_tooltip').outerHeight() + TOOLTIP_ARROW_HEIGHT);
        $("#ks_tooltip").addClass("ks_arrow_rb");
      }
    } else {
      maxWidth = sheetRect.right - tgtPt.left;
      tooltipPos.left = tgtPt.left;
      if (parseInt($(data.$this[0].children[0]).css('margin-left'), 10) > 0) {
        tooltipPos.left += parseInt($(data.$this[0].children[0]).css('margin-left'), 10);
      }
      if (tgtPt.top < refPt.top) {
        tooltipPos.top = tgtPt.top + ($(data.$this).height() + TOOLTIP_ARROW_HEIGHT);
        $("#ks_tooltip").addClass("ks_arrow_lt");
      } else {
        tooltipPos.top = tgtPt.top - ($('#ks_tooltip').outerHeight() + TOOLTIP_ARROW_HEIGHT);
        $("#ks_tooltip").addClass("ks_arrow_lb");
      }
    }
    if (maxWidth < $('#ks_tooltip').outerWidth()) {
      var paddingLeft = (getComputedStyle($('#ks_tooltip')[0]).paddingLeft).slice(0, -2);
      maxWidth -= (paddingLeft * 2);
      $('#ks_tooltip').css("width", maxWidth + "px");
      $('#ks_tooltip').css("white-space", "normal");
      if (tgtPt.left >= refPt.left) {
        tooltipPos.left = tgtPt.left - $('#ks_tooltip').outerWidth() + $(data.$this).width();
      }
      if (tgtPt.top >= refPt.top) {
        tooltipPos.top = tgtPt.top - ($('#ks_tooltip').outerHeight() + TOOLTIP_ARROW_HEIGHT);
      }
    }
    $('#ks_tooltip').offset(tooltipPos);
  } // doHover

  function delTooltip(data) {
    if ($('button', data.$this).hasClass('tpexec')
      || $('button', data.$this).hasClass('tpmotion')
      || $('button', data.$this).hasClass('tpbutton_shiftable')
      || $('button', data.$this).hasClass('tpbutton_bl_shiftable')
      || $('button', data.$this).hasClass('tpbutton_lbl_shiftable')
      || $('button', data.$this).hasClass('tpjog')
      || $('button', data.$this).hasClass('tphold')
      || $('button', data.$this).hasClass('tpbutton_fkey')

      || $('button', data.$this).hasClass('tpexecdown')
      || $('button', data.$this).hasClass('tpmotiondown')
      || $('button', data.$this).hasClass('tpbutton_shiftabledown')
      || $('button', data.$this).hasClass('tpbutton_bl_shiftabledown')
      || $('button', data.$this).hasClass('tpbutton_lbl_shiftabledown')
      || $('button', data.$this).hasClass('tpjogdown')
      || $('button', data.$this).hasClass('tpholddown')
      || $('button', data.$this).hasClass('tpbutton_fkeydown')) {
      $('#ks_tooltip').remove();
    }
  } // delTooltip

  // Release key of keysheet for tablet.
  function visChgFunc() {
    if (document[hiddenStr]) {
      $('#ks_tooltip').remove();  // remove tooltip
      releaseAllKeys();
      key_touches = [];
    }
  }

  // Changes the button sizes based on the size of the window.
  function resizeKSfunc() {
    var bw;
    var w = $('#funckeys', parent.document).width();

    $("#tpkey_func_vtcl").css("width", w + 'px');
    if ($("#tpkey_func_vtcl").css("display") != 'none') {
      $("#keysheet_box").css("width", w + 'px');
    } else {
      $("#keysheet_box").css("width", '');
    }

    if ((top.g_jtpinkey) && (!top.g_echo) && (!top.g_irprog) && (w > 640)) {
      bw = Math.floor(w * 0.65 / 7); // use 65% of the window size
    }
    else {
      bw = Math.floor(w / 7);
    }
    $(".func_fkey").css("width", bw-1 + 'px');
    $(".func_fkey button").css("width", bw - 1 + 'px');
  } // resize_funckeys

  this.g_releaseAllKeys = releaseAllKeys;
  this.g_resizeKSfunc = resizeKSfunc;
})(jQuery);
