/*
 *  This material is the joint property of FANUC America Corporation and
 *  FANUC LTD Japan, and must be returned to either FANUC America Corporation
 *  or FANUC LTD Japan immediately upon request.  This material and
 *  the information illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without the
 *  prior written consent of both FANUC America Corporation and FANUC LTD Japan.
 *
 *           All Rights Reserved
 *           Copyright (C)   2014
 *           FANUC America Corporation
 *           FANUC LTD Japan
 *  +
 *  Module: jquery.dlg.js
 *
 *  Description:
 *    Dialog plugin.
 *
 *  Author: Judy Evans
 *          FANUC America Corporation
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *
 *  Modification history:
 *  17-NOV-2014 EVANSJA  Initial version.
 *  05-DEC-2017 EVANSJA  pr50577 - Support for checking min and max in dialog input.
 *  30-JAN-2018 EVANSJA  pr50687 - Support for dialog boxes.
 *  05-MAR-2018 DOWMW    pr50762 - Support for Sub and Alpha Menus
 *  26-APR-2018 GRASSCM  pr50876 - Support for Sub Menu optional parameters.
 *  25-JAN-2019 MERCHANT pr51349 - Favorites dialog box needs additional height for scroll bars
 *  02-APR-2019 MERCHANT pr51466 - Other dialog boxes need additional height and width.
 *  02-MAY-2019 GRASSCM  pr51496 - Support MessageBox dialog
 *  08-MAY-2019 EVANSJA  pr51529 - Implement pulldown menus for jog menu.
 *                                 Add scrolling and resizing of popup menu.
 *  25-NOV-2019 EVANSJA  pr51981 - Support for password protected dialog items.
 *  03-DEC-2019 EVANSJA  pr51994 - Allow close button on dialog for Tablet TP.
 *  31-JAN-2020 EVANSJA  pr52071 - Reconnect to controller automatically if controller is turned off/on
 *  12-MAY-2020 EVANSJA  pr52260 - Fix PMEV_ALPHA_TYPE_MENU so it acts like PMEV_PLDn_MENU_C
 *  31-AUG-2020 EVANSJA  pr52488 - Add extra width for PMEV_SUB_MENU_C
 *  22-JUL-2022 LEESY    pr53647 - Fix blank item on popup menu not selectable
 *  02-FEB-2023 LEESY    pr53750 - Fix size of dialog box for DCS 2D VIEWS
 *  -
*/

var dlg_items;
var dlg_spage; // start page
var dlg_epage; // end page
var dlg_flyout;
var dlg_type;
var dlg_listener = null;
var dlg_selected = false;
var dlg_total;   // total number of pages in this set
var dlg_start;   // start page number of this set
var dlg_item;    // active item in page
var dlg_current; // current page
var dlg_items_per_page;  // number of items in each page
var dlg_items_this_page; // number of items if less than above
var dlg_notitle; // true if no title was supplied
var dlg_width = 0;
var dlg_extrawidth = 0;
var dlg_wtrc_width = 0;


var dlg_input_listener = null;
var dlg_input_selected = false;
var dlg_input_min_val = null;
var dlg_input_max_val = null;
var dlg_input_real = false;

var dlg_popup_closed = true;
var dlg_warn_closed = true;
var dlg_dialog_closed = true;
var dlg_msgbox_closed = true;
var dlg_input_closed = true;
var dlg_numeric_closed = true;
var dlg_shift_closed = true;
var dlg_numpad_closed = true;
var dlg_keypad_closed = true;
var dlg_jogkey_closed = true;
var touchjog_closed = true;

var dlg_dialog_inuse = 0;

$(function() {
  $("#selectable").selectable({
    stop: function() {
      var result = $("#select-result").empty();
      $(".ui-selected", this).each(function() {
        var index = $("#selectable li").index(this);
        index++; // index is 0 based
        if (index == dlg_items_per_page) {
          // Same as pressing 0
          dlg_handlekey('0'.charCodeAt(0));
        }
        else if ($(this).text() != '') {
          if ((dlg_type == PMEV_PLDn_MENU_C) || (dlg_type == PMEV_ALPHA_TYPE_MENU_C)) {
            // Keep the popup dialog open
            // Make controller select the same item by sending cursor keys
            dlg_selected = true;
            index += dlg_spage * (dlg_items_per_page-1);
            if (index > dlg_item) {
              for (; dlg_item < index; dlg_item++) {
                top.rpcmc_sendKeyCode(dn_arw_c);
              }
            }
            else {
              for (; dlg_item > index; dlg_item--) {
                top.rpcmc_sendKeyCode(up_arw_c);
              }
            }
          }
          else {
            // Same as pressing the key
            dlg_handlekey(index + '0'.charCodeAt(0));
          }
        }
        return false; // only select first one
      });
    }
  });

  $("#dlgITEXT").keypress(function(event) {
    if (event.keyCode == $.ui.keyCode.ENTER) {
      var str = $("#dlgITEXT").val();
      var charArray = [], chr, joinStr;
      for (var i = 0, len = str.length; i < len; i++) {
        chr = str.charAt(i);
        if (chr.charCodeAt(0) <= 127) {
          chr = encodeURIComponent(chr);
        }
        charArray.push(chr);
      }
      joinStr = charArray.join("");
      // Trigger a InputEvent on the input event listeners.
      dlg_input_listener.trigger("InputEvent", [joinStr]);
      dlg_input_selected = true;
      $("#dlgINPUT").dialog("close");
    }
  });

  $("#dlgNUM").keypress(function(event) {
    if (event.keyCode == $.ui.keyCode.ENTER) {
      var str = $("#dlgNUM").val();
      // Check min and max if given
      if ((dlg_input_min_val !== null) && (dlg_input_max_val !== null)) {
        if (dlg_input_real) {
          var value_float = parseFloat(str);
          if ((value_float < dlg_input_min_val) || (value_float > dlg_input_max_val)) {
            $("#dlgNUM").val("");
            return;
          }
        }
        else {
          var value_long = parseInt(str);
          if ((value_long < dlg_input_min_val) || (value_long > dlg_input_max_val)) {
            $("#dlgNUM").val("");
            return;
          }
        }
      }
      // Trigger a InputEvent on the input event listeners.
      dlg_input_listener.trigger("InputEvent", [str]);
      dlg_input_selected = true;
      $("#dlgNUMERIC").dialog("close");
    }
  });
});

function dlg_open(title) {
  // Always select the specified item
  $('#selectable .ui-selected').removeClass('ui-selected');
  $('#selectable :nth-child(' + dlg_item + ')').addClass('ui-selected');

  if ((title.length == 0) || (title == ' ')) {
    // If title was not provided, then display page number
    dlg_notitle = true;
    title = (top.dlg_spage+1) + '/' + (top.dlg_epage+1);
  }
  if (!dlg_width) {
    dlg_width = $("#dlgPOPUP").dialog("option", "width");
    dlg_extrawidth = dlg_width + 150;
    if (dlg_extrawidth > $(top.window).width() - 20) {
      dlg_extrawidth = $(top.window).width() - 20;
    }
  }
  switch (dlg_type) {
  case PMEV_MAIN_MENU_C:
  case PMEV_QUIC_MENU_C:
    // Top Left
    $("#dlgPOPUP").dialog({ title: title,  width: dlg_width,
      position: { my: "left top", at: "left+10, top+90", of: window }
    });
    break;
  case PMEV_FCTN_MENU_C:
    // Top Right
    $("#dlgPOPUP").dialog({ title: title,  width: dlg_width,
      position: { my: "right top", at: "right-10, top+90", of: window }
    });
    break;
  case PMEV_TYPE_MENU_C:
    if (top.g_irprog) {
      var type_pos = $(top.window).height() - top.$("#statbar").height() - top.$("#mainfrm").height();
      if (($(top.window).height() - type_pos) < type_pos) {
        // Top Left
        $("#dlgPOPUP").dialog({ title: title, width: dlg_width,
          position: { my: "left top", at: "left+20, top+90", of: window }
         });
      }
      else {
        type_pos = "left+20, bottom-" + type_pos;
        // Upper Funckey
        $("#dlgPOPUP").dialog({ title: title, width: dlg_width,
          position: { my: "left bottom", at: type_pos, of: window }
         });
      }
    }
    else {
      // Bottom Left
      $("#dlgPOPUP").dialog({ title: title, width: dlg_width,
        position: { my: "left bottom", at: "left+20, bottom-85", of: window }
      });
    }
    break;
  case PMEV_SUB_MENU_C:
    // Center with extra width
    $("#dlgPOPUP").dialog({ title: title, width: dlg_extrawidth,
      position: { my: "center top", at: "center, top+90", of: window }
    });
    break;
  default:
    // Center
    $("#dlgPOPUP").dialog({ title: title, width: dlg_width,
      position: { my: "center top", at: "center, top+90", of: window }
    });
    break;
  } // switch (dlg_type)

  // Open the popup dialog.
  dlg_popup_closed = false;
  $("#dlgPOPUP").dialog("open");
  if ((dlg_type == PMEV_PLDn_MENU_C) || (dlg_type == PMEV_ALPHA_TYPE_MENU_C)) {
    // Controller handles the keys
    document.onkeydown = wscr_dlg_keydown;
  }
  else {
    // We will handle both browser keys and controller keys.
    document.onkeydown = dlg_keydown;
    top.jQuery.keylis.bind('KeyPressedEvent', HandleKeyEvent);
  }
  dlg_selected = false;
}

function dlg_close() {
  // Close the popup dialog.
  $("#dlgPOPUP").dialog("close");
  document.onkeydown = null;
  if ((dlg_type != PMEV_PLDn_MENU_C) && (dlg_type != PMEV_ALPHA_TYPE_MENU_C)) {
    top.jQuery.keylis.unbind('KeyPressedEvent', HandleKeyEvent);
  }
  setTimeout(function(){dlg_close_setfocus();}, 100);
}

function dlg_close_setfocus() {
  // set focus to main frame
  var dev_list = ["prim", "dual", "third"];
  if (!document.activeElement.name && dlg_input_closed &&
      dlg_popup_closed && dlg_dialog_closed && dlg_warn_closed) {
    var ret_frame = top.find_frame(top, dev_list[top.g_device_id]);
    if (ret_frame) {
      ret_frame.focus();
    }
  }
}

function dlg_populate() {
  var index;
  var sindex = dlg_spage * (dlg_items_per_page-1);
  var eindex = sindex + (dlg_items_per_page-1);
  for (index = 1; index < dlg_items_per_page; sindex++, index++) {
    $('#selectable :nth-child(' + index + ')').removeClass('ui-state-disabled');
    if (sindex >= dlg_items.length) {
      $('#f' + index).text('');
    }
    // We use "space" for a blank item on popup menu, and the .buf.length = 0
    // so commemt out the lines below to support the blank item.
    //else if (!dlg_items[sindex].buf.length) {
    //  $('#f' + index).text('');
    //}
    else {
      $('#f' + index).text(index + ' ' + dlg_items[sindex].buf);
      if (dlg_items[sindex].dis == "1") {
        // Password protected item
        $('#selectable :nth-child(' + index + ')').addClass('ui-state-disabled');
      }
    }
    $('#selectable :nth-child(' + index + ')').show();
  }
  if (dlg_items_this_page < dlg_items_per_page) {
    // Hide any lines that are not being used
    for (index = dlg_items_per_page; index > dlg_items_this_page; index--) {
      $('#selectable :nth-child(' + index + ')').hide();
    }
  }
  else {
    $('#selectable :nth-child(' + dlg_items_per_page + ')').show();
  }
  if (dlg_notitle) {
    // If title was not provided, then display page number
    $("#dlgPOPUP").dialog("option", "title", (top.dlg_spage+1) + '/' + (top.dlg_epage+1));
  }
  // Lastly get rid of any extra items
  for (index = 10; index > dlg_items_per_page; index--) {
    $('#selectable :nth-child(' + index + ')').hide();
  }
}

function dlg_handlekey(key) {
  if ((dlg_type == PMEV_PLDn_MENU_C) || (dlg_type == PMEV_ALPHA_TYPE_MENU_C)) {
    // Controller handles the keys
    return;
  }
  var index = key - '0'.charCodeAt(0);
  if ((index >= 0) && (index <= (dlg_items_per_page-1))) {
    if (!index) {
      dlg_spage++;
      if (dlg_spage > dlg_epage) {
        dlg_spage = 0;
      }
      dlg_populate();
      // Always select the first item of a new page.
      $('#selectable .ui-selected').removeClass('ui-selected');
      $('#selectable :first-child').addClass('ui-selected');
      dlg_item = 1;
    }
    else if ($('#f' + index).text() != '') {
      if ($('#selectable :nth-child(' + index + ')').hasClass('ui-state-disabled')) {
        // Item is disabled
        return;
      }
      index += dlg_spage*(dlg_items_per_page-1);
      index--; // 0 based
      dlg_selected = true;
      if (!top.g_echo) {
        // Close the popup dialog.
        dlg_close();
        dlg_select(index);
      }
    }
  }
  else if ((key == 33) || (key == 38)) {
    // Scroll up one item.
    dlg_item--;
    if (dlg_item == 0) {
      dlg_item = dlg_items_this_page;
    }
    $('#selectable .ui-selected').removeClass('ui-selected');
    $('#selectable :nth-child(' + dlg_item + ')').addClass('ui-selected');
  }
  else if ((key == 34) || (key == 40)) {
    // Scroll down one item.
    dlg_item++;
    if (dlg_item > dlg_items_this_page) {
      dlg_item = 1;
    }
    $('#selectable .ui-selected').removeClass('ui-selected');
    $('#selectable :nth-child(' + dlg_item + ')').addClass('ui-selected');
  }
  else if (key == enter_c) {
    // Select the item.
    index = dlg_item;
    if (index == dlg_items_per_page) {
      // Same as pressing 0
      dlg_handlekey('0'.charCodeAt(0));
    }
    else {
      // Same as pressing the key
      dlg_handlekey(index + '0'.charCodeAt(0));
    }
  }
}

function dlg_keydown() {
  // These keys came from the browser.
  // We will handle internally.
  dlg_handlekey(window.event.keyCode);
}

function wscr_dlg_keydown() {
  // These keys came from the browser.
  // We will map to controller keys and send them.
  if (window.event.keyCode == 27) top.rpcmc_sendKeyCode(prev_c);
  else if (window.event.keyCode == 33) top.rpcmc_sendKeyCode(up_arw_s_c);
  else if (window.event.keyCode == 34) top.rpcmc_sendKeyCode(dn_arw_s_c);
  else if (window.event.keyCode == 37) top.rpcmc_sendKeyCode(lf_arw_c);
  else if (window.event.keyCode == 38) top.rpcmc_sendKeyCode(up_arw_c);
  else if (window.event.keyCode == 39) top.rpcmc_sendKeyCode(rt_arw_c);
  else if (window.event.keyCode == 40) top.rpcmc_sendKeyCode(dn_arw_c);
  else top.rpcmc_sendKeyCode(window.event.keyCode);
}

function HandleKeyEvent(event, key) {
  // These keys came from the controller.
  // We will map to browser keys and handle internally.
  if (key == up_arw_s_c) key = 33;
  else if (key == dn_arw_s_c) key = 34;
  else if (key == lf_arw_c) key = 37;
  else if (key == up_arw_c) key = 38;
  else if (key == rt_arw_c) key = 39;
  else if (key == dn_arw_c) key = 40;
  dlg_handlekey(key);
}

function dlg_select(index) {
  if (dlg_listener) {
    // Trigger a MenuEvent on the menu event listeners.
    dlg_listener.trigger("MenuEvent", [index]);
  }
  else if (dlg_type == PMEV_FCTN_MENU_C) {
    $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
      '&current=fctnmenu,' + dlg_items[index].spid + ',' + dlg_items[index].scid, function() {
    });
  }
  else if (dlg_type == PMEV_DISP_MENU_C) {
    $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
      '&current=dispmenu,' + dlg_items[index].spid + ',' + dlg_items[index].scid, function() {
    });
  }
  else if (dlg_type == PMEV_SUB_MENU_C) {
    $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
      '&current=submenu,' + dlg_items[index].lin + ',' + dlg_items[index].buf, function () {
      });
  }
  else if (dlg_type == PMEV_ALPHA_TYPE_MENU_C) {
    $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
      '&current=alphamenu,' + dlg_items[index].lin + ',' + dlg_items[index].buf, function () {
      });
  }
  else {
    // PMEV_MAIN_MENU_C and PMEV_TYPE_MENU_C
    if (top.g_irprog && (top.treemod != undefined) && (top.treemod.primScreenId != null)) {
      var url_str = top.treemod.setTreeUrlStr(parseInt(dlg_items[index].spid), parseInt(dlg_items[index].scid));
      $.get(url_str, function() { });
      var currentFrameId = top.getMyFrameId();
      var sel_url = '/softpart/genlink?current=menupage,' + dlg_items[index].spid + ',' + dlg_items[index].scid;
      if (currentFrameId == 'prim' || top.treemod.frcprim == true) {
        top.treemod.setUrlList(sel_url);
        top.treemod.primScreenId = sel_url;
      }
      else if (currentFrameId == 'dual'){
        top.treemod.dualScreenId = sel_url;
      }else {
        top.treemod.thirdScreenId = sel_url;
      }
    }
    else {
      $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
        '&current=menupage,' + dlg_items[index].spid + ',' + dlg_items[index].scid, function() {
      });
    }
  }
}

function dlg_input_open(title, guide, start_str) {
  // Open the input dialog.
  var dWidth = top.w_statbar;
  if (!dlg_dialog_closed) {
    // Make keyboard smaller
    dWidth = dWidth / 2;
  }
  $("#dlgINPUT").dialog("option", "width", dWidth);
  if (dWidth > 40) {
    $("#dlgITEXT").css("width", dWidth-40 + "px")
  }
  $("#dlgINPUT").dialog("option", "title", title);
  $('#dlgIGUIDE').html(guide);
  $("#dlgITEXT").val(start_str);

  dlg_input_closed = false;
  $("#dlgINPUT").dialog("open");
  dlg_input_selected = false;
  setTimeout(function() { document.getElementById("dlgITEXT").focus(); }, 100);
}

function dlg_input_close() {
  // Close the input dialog.
  dlg_input_closed = true;
  $("#dlgINPUT").dialog("close");

  // Refocus pane
  $.get('/SOFTPART/TPTX?connect_id=' + top.g_connect_id + '&fc=0x90055&device_id=0&request_code=79&key_code=255', function() { });
}

function dlg_numeric_open(title, guide, start_str) {
  // Open the numeric dialog.
  $("#dlgNUMERIC").dialog("option", "title", title);
  $('#dlgNGUIDE').html(guide);
  $("#dlgNUM").val(start_str);

  dlg_numeric_closed = false;
  $("#dlgNUMERIC").dialog("open");
  dlg_input_selected = false;
  setTimeout(function() { document.getElementById("dlgNUM").focus(); }, 100);
}

function dlg_numeric_close() {
  // Close the numeric dialog.
  dlg_numeric_closed = true;
  $("#dlgNUMERIC").dialog("close");
}

function dlg_numpad_open(title) {
  // Open the numpad dialog.
  $("#dlgNUMPAD").dialog("option", "title", title);
  dlg_numpad_closed = false;
  $("#dlgNUMPAD").dialog("open");
}

function dlg_numpad_close() {
  // Close the numpad dialog.
  dlg_numpad_closed = true;
  $("#dlgNUMPAD").dialog("close");
}

function dlg_keypad_open(title) {
  // Open the keypad dialog.
  $("#dlgKEYPAD").dialog("option", "title", title);
  dlg_keypad_closed = false;
  $("#dlgKEYPAD").dialog("open");
}

function dlg_keypad_close() {
  // Close the keypad dialog.
  dlg_keypad_closed = true;
  $("#dlgKEYPAD").dialog("close");
}

function dlg_jogkey_open(title) {
  // Open the jogkey dialog.
  $("#dlgJOGKEY").dialog("option", "title", title);
  dlg_jogkey_closed = false;
  $("#dlgJOGKEY").dialog("open");
}

function dlg_jogkey_close() {
  // Close the jogkey dialog.
  dlg_jogkey_closed = true;
  $("#dlgJOGKEY").dialog("close");
}

function dlg_shift_open() {
  if ((shiftfwd.g_running) && (!shiftfwd.g_step) && (!top.g_tabtp)) {
    // Open the shift dialog.
    dlg_shift_closed = false;
    $("#dlgSHIFTFWD").dialog("option", "title", top.srvap_getString("FWD_title"));
    $("#dlgSHIFTFWD").dialog("open");
  }
}

function dlg_shift_close() {
  // Close the shift dialog.
  dlg_shift_closed = true;
  $("#dlgSHIFTFWD").dialog("close");
}

function dlg_dialog_open(device_id, title, dlgwidth, dlgheight, posx, posy, url) {
  if (!dlg_dialog_closed) {
    // Already displayed
    top.rpcmc_tpextreq2(device_id, tp_thispanel_c, dialog_cancel_c);
    return;
  }
  if (dlg_dialog_inuse) {
    // We must wait until previous dialog box is removed.
    setTimeout(function() { dlg_dialog_open(device_id, title, dlgwidth, dlgheight, posx, posy, url); }, 100);
    return;
  }
  // Populate the dialog.
  if (dlgwidth == 640) {
    // This is the favorite dialog box
    dlgwidth = $(top.window).width();
    dlgheight += 40; //have to make room for scroll bars
    if (top.g_irprog) {
      var f_menu_btm =  $(top.window).height() - top.$("#statbar").height() - top.$("#mainfrm").height() - top.$("#funckeys").height();
      f_menu_btm = "left, bottom-" + f_menu_btm;
      if (dlgwidth > top.$("#mainfrm").outerWidth()) {
        dlgwidth = top.$("#mainfrm").outerWidth();
      }
      $("#dlgDIALOG").dialog({ title: title, width: dlgwidth, height: dlgheight,
        dialogClass: 'notitle',
        position: { my: "left bottom", at: f_menu_btm, of: window }
      });
    }
    else {
      $("#dlgDIALOG").dialog({ title: title, width: dlgwidth, height: dlgheight,
        dialogClass: 'notitle',
        position: { my: "left bottom", at: "left bottom", of: window }
      });
    }
    $('.notitle').find('.ui-dialog-titlebar').css('display','none');
  }
  else {
    if (dlgwidth == 638) {
      // This is the DCS 2D VIEWS dialog box
      posy = top.$("#statbar").height() + (top.$("#statbar").height() / 2);
      dlgwidth = top.$("#mainfrm").width();
      dlgheight = top.$("#mainfrm").height() - (top.$("#statbar").height() / 2);
    }
    else {
      // Dialog box itself uses 65w x 45h
      dlgwidth += 65;
      dlgheight += 45;
    }
    if (title != "") {
      dlgheight += 40;
    }
    var allow_close = false;
    if (title.charAt(0) == '*') {
      // Allow the close button
      allow_close = true;
      title = title.slice(1);
    }
    $("#dlgDIALOG").dialog({ title: title, width: dlgwidth, height: dlgheight,
      dialogClass: 'notitle',
      position: { my: "left top", at: "left+" + posx + " top+" + posy, of: window }
    });
    if (allow_close) {
      $('.notitle').find('.ui-dialog-titlebar').css('display','block');
    }
    else if (title != "") {
      // Keep the title but hide the close button
      $('.notitle').find('.ui-dialog-titlebar').css('display','block');
      $('.notitle').find('.ui-dialog-titlebar-close').hide();
    }
    else {
      $('.notitle').find('.ui-dialog-titlebar').css('display','none');
    }
  }
  $("#dlgDIALOG").html('<iframe id="dlgsrc" src=' + url + ' width="100%" height="100%"></iframe>');
  dlg_dialog_inuse = 1;

  // Open the dialog.
  dlg_dialog_closed = false;
  $("#dlgDIALOG").dialog("open");
  document.onkeydown = wscr_dlg_keydown;
}

function dlg_dialog_close() {
  dlg_dialog_closed = true;

  // We must exit the web page before closing or else top is not defined properly
  // dlgclr.htm will set dlg_dialog_inuse = 0
  $("#dlgsrc").attr("src", "/frh/jcgtp/dlgclr.htm");
  $("#dlgDIALOG").dialog("close");
  document.onkeydown = null;
}

// MessageBox type dialog is not tied to pendant keys
function dlg_msgbox_open(title_str, message_str) {
  if (!dlg_msgbox_closed) {
    // Already displayed
    return;
  }

  // Populate the dialog.
  $("#dlgMSGBOX").dialog("option", "title", title_str);
  $("#dlgMSGTXT").text(message_str);

  // Open the dialog
  dlg_msgbox_closed = false;
  $("#dlgMSGBOX").dialog("open");
}

function dlg_msgbox_close() {
  dlg_msgbox_closed = true;
}

function dlg_ping_open() {
  // Open the dialog
  var dlgwidth = $(top.window).width() - 20;
  var dlgheight = top.getDevW(0);
  $("#dlgPING").dialog({ title: getString('DISCONNECTERR_C'), width: dlgwidth, height: dlgheight});
  $("#dlgPINGTXT1").html(getString('HEARTBEAT_C'));
  $("#dlgPINGTXT2").html(getString('PING') + " " + location.host);
  $("#dlgPING").dialog("open");
}

function dlg_ping_progress(str) {
  if (str == "*") {
    $("#dlgPINGTXT3").append(str);
  }
  else {
    $("#dlgPINGTXT2").html(str);
    $("#dlgPINGTXT3").html("");
  }
}

function dlg_recon_open() {
  // Open the dialog
  if (!dlg_wtrc_width) {
    dlg_wtrc_width = $("#dlgWTRC").dialog("option", "width") + 150;
  }
  $("#dlgWTRC").dialog({ title: getString('COMM_IN_PROGRESS'), width: dlg_wtrc_width});
  $("#dlgWTRCTXT1").html(getString('WAIT_COMM_COMP'));
  if (!(navigator.userAgent.toLowerCase().indexOf("android") >= 0) || !(navigator.userAgent.toLowerCase().indexOf("wv") >= 0)) {
    $("#dlgWTRCTXT2").html(getString('TRY_RECONNECT'));
  }
  $("#dlgWTRC").dialog("open");
}

function dlg_recon_close() {
  // Close the dialog
  $("#dlgWTRC").dialog("close");
}
