/** 
 * @fileoverview JavaScript Library: This file is used to operate cookie.
 */

/** @namespace Holds functionality related to */
var Cookie = {};

/**
 * Sets a Cookie with the given name and value.
 * @param {string} name  Name of the cookie
 * @param {string} value Value of the cookie
 * @param {string} [expires = end of current session] Expiration date of the cookie
 * @param {string} [path = path of calling document] Path where the cookie is valid
 * @param {string} [domain = domain of calling document] Domain where the cookie is valid
 * @param {string} [secure] Boolean value indicating if the cookie transmission requires a secure transmission
 */
Cookie.store = function (name, value, expires, path, domain, secure) {
  document.cookie= name + "=" + escape(value) +
    ((expires) ? "; expires=" + expires.toGMTString() : "") +
    ((path) ? "; path=" + path : "") +
    ((domain) ? "; domain=" + domain : "") +
    ((secure) ? "; secure" : "");
};

/**
 * Gets the value of the specified cookie.
 * @param {string} name Name of the desired cookie.
 * @return  a string containing value of specified cookie, or null if cookie does not exist.
 */
Cookie.get = function (name) {
  var dc = document.cookie;
  var prefix = name + "=";
  var begin = dc.indexOf("; " + prefix);
  if (begin == -1) {
    begin = dc.indexOf(prefix);
    if (begin != 0) return null;
  }
  else {
    begin += 2;
  }
  var end = document.cookie.indexOf(";", begin);
  if (end == -1) {
    end = dc.length;
  }
  return unescape(dc.substring(begin + prefix.length, end));
};

/**
 * Removes the specified cookie.
 * @param {string} name name of the cookie
 * @param {string} [path] path of the cookie (must be same as path used to create cookie)
 * @param {string} [domain] domain of the cookie (must be same as domain used to create cookie)
 */
Cookie.remove = function (name, path, domain) {
  if (Cookie.get(name)) {
    var exp = new Date();
    exp.setTime (exp.getTime() - 1);  // This cookie is history
    document.cookie = name + "=" + 
      ((path) ? "; path=" + path : "") +
      ((domain) ? "; domain=" + domain : "") +
      "; expires=" + exp.toGMTString();
  }
};
