/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2016
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: grid.js
 *  
 *  Description:
 *    Grid plugin
 *
 *  Author: Judy Evans
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  15-APR-2015 EVANSJA Initial version
 *  03-AUG-2017 MERCHANT Functioning version
 *  18-AUG-2017 MERCHANT Grid file is no longer guaranteed to XML legal, 
 *                       so replaced DOM with private XML parser.
 *  02-APR-2018 MERCHANT Convert to HTML5 Canvas.
 *  26-Jun-2018 MERCHANT Save drawing elements as records instead of strings to support mod tags better.
 *  02-Apr-2019 MERCHANT Want dialog box for call program to be clickable.
 *  05-May-2019 MERCHANT Rewrite parser in order to keep up with massive data from PMC page.
 *  02-Jan-2020 MERCHANT PR51398 - add support for scroll bars.
 *  08-Jun-2021 MERCHANT PR52799B - Fix process logger bugs
 *  14-Oct-2022 LEESY    pr53689 - Fix the size of FAVORITES dialog box
 *  02-FEB-2023 LEESY    pr53750: Fix Lines and Circles for DCS 2D VIEWS
*  -
*/

// Private functions 
// Create control only when PMON Server is ready
function grid_ServerReady(data) {
  // Initialize control data.
  grid_InitCtlData(data);

  // Create control.
  grid_CreateCtl(data);
  try { //data.Data not guaranteed to be there
    if ((undefined !== data.Data) && (null !== data.Data)) { //treat object data like pipe data
      grid_ParsePipeData(data, data.Data);  
    }
  }
  catch (ex){ //no data.data then
    console.log("no data.Data");
  }

  finally { }

  // Initialize events.
  grid_InitRobotEvent(true, data);

} // grid_ServerReady

// New instance of control.
function grid_NPP_New(data) {

  if (undefined != data.PipeMonRate) {
    data.PipeMonRate = Number(data.PipeMonRate);
  }

  SetCommonParams(data);
  if (data.IsDialog) {
    grid_ServerReady(data);
  }
  else {
    top.rpcmc_getTaskIdx(data.fDeviceId, function (taskIdx) {
      data.fTaskIdx = taskIdx;

      // Complete after top.rpcmc_getTaskIdx completes.
      grid_ServerReady(data);
    });
  }

} // grid_NPP_New

// Destroy instance of control.
function grid_NPP_Destroy(data) {

  // Uninitialize events.
  grid_InitRobotEvent(false, data);

  // Delete control.
  grid_DeleteCtl(data);

} // grid_NPP_Destroy

// Private functions

// Set the border.
function grid_SetBorder(data) {

  var border = parseInt(data.Border);
  if (border > 0) {
    data.$this.css({ "border-style": "inset", "border-width": "' + border + 'px", "border-color": "white" });
  }
  else if (border < 0) {
    border *= -1;
    data.$this.css({ "border-style": "inset", "border-width": "' + border + 'px", "border-color": "white" });
  }
  else {
    data.$this.css("border-style", "none");
  }

} // grid_SetBorder

// Update Control.
function grid_UpdateCtl(data) {


} // grid_UpdateCtl

// Initialize or uninitialize pipe event.
function grid_InitPipeEvent(init, data) {

  //if (data.Interval < 0) {
  //  data.Interval = 1000;
  //}
  if (data.PipeMonRate < 0) {
    data.PipeMonRate = 1000;
  }
  // Notify PMON to start/stop monitoring our pipe
  if (init) {
    top.jQuery.filelis.bind('PipeEvent', data, grid_HandlePipeEvent); // Attach handler for GridEvent
    top.rpcmc_startPipeMonitor(data.Pipe, data.PipeMonRate); // Start PMON monitor for our pipe
  }
  else {
    top.rpcmc_stopPipeMonitor(data.Pipe); // Stop PMON monitor for our pipe
    top.jQuery.filelis.unbind('PipeEvent', grid_HandlePipeEvent); // Detach handler for GridEvent.
  }
} // grid_InitPipeEvent

// Replace any indirection with actual value.
function grid_IndirectReplace(data) {

  var l_ind;
  var pos;

  $.each(data, function (argn, argv) {
    if (typeof argv !== 'string') {
      return;
    }
    // Value contain !PaneId?
    if ((pos = argv.toLowerCase().indexOf("!paneid")) >= 0) {
      argv = argv.substring(0, pos) + data.fTaskIdx + argv.substring(pos + 7);
      data[argn] = argv;
    }
  });
} // grid_IndirectReplace

// Initialize Control Data.
function grid_InitCtlData(data) {

  // Process parameters.
  // Name not supported
  // Border
  // Pipe
  // Scene
  // SubPane
  // Verbose not supported
  // ExecConnectId not required (only for ActiveX Controls)

  data.Pipe = data.Pipe.toUpperCase();
  grid_IndirectReplace(data);

  data.SubPane = parseInt(data.SubPane);
  if (data.BackColor == "") {
    data.BackColor = data.InvisibleColor;
  }
  data.BackColor = GridFixColor(data.BackColor);

  if (data.ForeColor == "") {
    data.ForeColor = data.TextColor;
  }
  data.ForeColor = GridFixColor(data.ForeColor);
  if (data.ForeColor.toLowerCase() == data.BackColor.toLowerCase()) {
    data.ForeColor = InvertColor(data.ForeColor);
  }

} // grid_InitCtlData


function GridFixColor(Color) {
  if (Color.indexOf("#") == 0) {
    if (Color.length > 7) { //is rgba
      //Color = Color.substring(0, 7);
      Color = "";
    }
  } else if (Color.indexOf("rgba") == 0) {
    Color = rgb2hex(Color);
  } else if (Color.indexOf("rgb") == 0) {
    Color = rgb2hex(Color);
  } else {
    Color = translateColor(Color);
  }
  return Color;
}//GridFixColor


// Initialize or uninitialize events for each type.
function grid_InitRobotEvent(init, data) {
  // Start/stop the Pipe Event.
  grid_InitPipeEvent(init, data);

} // grid_InitRobotEvent


function grid_console(message) {
  //console.log(message);
} //grid_console


// Create Control.
function grid_CreateCtl(data) {

  data.$this.css("display", "inline-block");
  if (data.width.indexOf("%") >= 0) {
    var w = top.getLegacyW(data.fDeviceId);
    var dec = parseFloat(data.width);
    data.width = Math.round(w * dec / 100);
  }
  else if (data.width == 640) {
    // This is how the favorite resize the dialog box in jquert.gld.js
    // This is also required for the favorite resize the grid  	
    data.width = $(top.window).width();
    if (top.g_irprog) {
      if (data.width > top.$("#mainfrm").outerWidth()) {
        data.width = top.$("#mainfrm").outerWidth();
      }	
    }
  }  
  data.$this.css("width", data.width + "px");

  if (data.height.indexOf("%") >= 0) {
    var h = top.getLegacyH(data.fDeviceId);
    var dec = parseFloat(data.height);
    data.height = Math.round(h * dec / 100);
  }
  else if (data.height == 90) {
    // This is how the favorite resize the dialog box in jquert.gld.js
    // This is also required for the favorite resize the grid  	
    var h_temp = parseFloat(data.height);
    data.height = Math.round(h_temp + 25);
  }  

  data.$this.css("height", data.height + "px");
  data.$this.css("vertical-align", "top");

  grid_SetBorder(data);
  SetColors(data);
  SetFont(data);

  grid_console(data.Id + ".Create grid width: " + data.width + "px" + " height: " + data.height + "px on " + data.myPageName);
  // Attach handler for mousedown event.
  data.$this.bind("mousedown", data, grid_HandleMouseDownEvent);

  // Attach handler for mouseup event.
  data.$this.bind("mouseup", data, grid_HandleMouseUpEvent);

  // Attach handler for mouseout event.
  data.$this.bind("mouseout", data, grid_HandleMouseUpEvent);

} // grid_CreateCtl

// Handle Control events.
function grid_CtlEvent(data) {

  var sb_state;

  if (data.IsEcho) {
    return;
  }

} // grid_CtlEvent

// Delete Control Data.
var buttons = [];
var graphics = [];  //to save speed graphics are cached.
var grids = [];
var tiles = [];

function grid_DeleteCtl(data) {

  grid_console(data.Id + ".Delete");

  //flush buttons
  for (var ii = 0; ii < buttons.length;) {
    if (buttons[ii].GID == data.Id) {
      buttons.splice(ii, 1);
    } else {
      ii++;
    }
  }

  //flush tiles
  for (ii = 0; ii < tiles.length;) {
    if (tiles[ii].GID == data.Id) {
      grid_DeleteTile2(ii);
      tiles.splice(ii, 1);
    } else {
      ii++;
    }
  }

  for (ii = 0; ii < grids.length;) {
    if (grids[ii].ID == data.Id) {
      grids.splice(ii, 1);
    } else {
      ii++;
    }
  }
} // grid_DeleteCtl

// Private functions
function IsTag(tag, value) {
  if (tag.indexOf(value) == 0) {
    return true;
  } else {
    return false;
  }
}//IsTag

function normalize(istring) {

  var cc;
  var state = 0;
  var ostring = "";

  for (var ii = 0; ii < istring.length; ii++) {
    cc = istring.substr(ii, 1);
    switch (state) {
      case 0: //leave 1 leading space
        if (cc == ' ') {
          ostring = ostring + cc.toLocaleLowerCase();
        } else {
          ostring = ostring + ' ' + cc.toLocaleLowerCase();
        }
        state = 1;
        break;
      case 1: //remove leading spaces
        if (cc != ' ') {
          ostring = ostring + cc.toLocaleLowerCase();
          state = 2;
        }
        break;
      case 2:  //attribute name
        if (cc == ' ') {
          state = 3;
        } else if (cc == '=') {
          ostring = ostring + cc.toLocaleLowerCase();
          state = 4;
        } else {
          ostring = ostring + cc.toLocaleLowerCase();
        }
        break;
      case 3: //remove trailing spaces
        if (cc == '=') {
          ostring = ostring + cc.toLocaleLowerCase();
          state = 4;
        }
        break;
      case 4: //remove trailing spaces
        if (cc == '"') {
          ostring = ostring + cc.toLocaleLowerCase();
          state = 5;
        } else if (cc == "'") {
          ostring = ostring + cc.toLocaleLowerCase();
          state = 6;
        } else {
        }
        break;
      case 5: //attribute value
        ostring = ostring + cc;
        if (cc == '"') {
          state = 0;
        }
        break;
      case 6: //attribute value (alt)
        ostring = ostring + cc;
        if (cc == "'") {
          state = 0;
        }
        break;
      default:
        break;
    }
  }//for
  return ostring;

}//normalize

var XmlData = '';
var GridFormatted = false;
var gindex = -1;
var tindex = -1;


function grid_ParsePipeData(data, buffer) {

  var buffer2 = XmlData + buffer;
  var buffer3;
  var closetag;
  var exclamation;
  var iindex;
  var imageend;
  var lbracket;
  var lbracket2;
  var rbracket;
  var space;
  var substring;
  var temp;
  var temp2;
  var textend;
  var tileend;
  var XmlIndex = 0;
  var XmlTag;

  try {
    while (XmlIndex < buffer2.length) {
      lbracket = buffer2.indexOf('<', XmlIndex);
      if (lbracket == -1) {
        XmlData = "";
        buffer2 = "";
        return true;
      }
      rbracket = buffer2.indexOf('>', lbracket);
      if (rbracket == -1) {
        XmlData = buffer2.substr(lbracket);
        grid_console(data.Id + '.Wait (partial) : "' + XmlData + '"');
        buffer2 = "";
        return true;
      }
      temp = buffer2.substring(lbracket, rbracket + 1);
      //look for partial tags 
      lbracket2 = temp.indexOf('<', 1);
      if (lbracket2 > -1) { //got partial tag
        temp2 = temp.substring (0, lbracket2);
        console.log(data.Id + '.Discard (partial) : "' + temp2 + '"');
        temp = temp.substring (lbracket2); // move to real tag
        lbracket += lbracket2;
      }
      exclamation = temp.indexOf('!');
      space = temp.indexOf(' ');
      if (exclamation == 1) {
        XmlTag = temp;
      } else if (space > -1) {
        substring = temp.substring(space);
        XmlTag = temp.substring(0, space).toLocaleUpperCase() + normalize(substring);
      } else {
        XmlTag = temp.toLocaleUpperCase();
      }

      grid_console(data.Id + '.Read : ' + lbracket + ':' + rbracket + ' "' + XmlTag + '"');
      XmlIndex = rbracket + 1;
      buffer3 = buffer2.substr(XmlIndex);

      if (IsTag(XmlTag, '<!')) {
        grid_console(data.Id + '.SPECIALDISCARD : ' + XmlTag);
      } else if (IsTag(XmlTag, '<?')) {
        grid_console(data.Id + '.SPECIALDISCARD : ' + XmlTag);
      } else if (IsTag(XmlTag, '<BUTTON')) {
        tiles[tindex].BUTTON.push(grid_ParseButton(XmlTag, tindex));
      } else if (IsTag(XmlTag, '<CIRCLE')) {
        tiles[tindex].CIRCLE.push(grid_ParseCircle(XmlTag));
      } else if (IsTag(XmlTag, '<DELTILE')) {
        grid_DeleteTiles(data, XmlTag);
      } else if (IsTag(XmlTag, '<DISPLAY')) {
        grid_DisplayTiles(data, XmlTag, gindex);
        GridFormatted = true;
      } else if (IsTag(XmlTag, '<GRID')) {
        gindex = grid_CreateGrid(data, XmlTag);
        GridFormatted = false;
      } else if (IsTag(XmlTag, '<IMAGE')) {
        closetag = XmlTag.indexOf('/>');
        if (closetag > -1) {
          iindex = tiles[tindex].IMAGE.length;
          tiles[tindex].IMAGE.push(grid_ParseImage(XmlTag, tindex, iindex));
        } else {
          imageend = buffer2.indexOf('</IMAGE>', XmlIndex);
          if (imageend > -1) {
            XmlTag = XmlTag + buffer2.substring(XmlIndex, imageend + 8);
            grid_console(data.Id + '.ReRead : "' + XmlTag + '"');
            iindex = tiles[tindex].IMAGE.length;
            tiles[tindex].IMAGE.push(grid_ParseImage(XmlTag, tindex, iindex));
            XmlIndex = imageend + 8;
            buffer3 = buffer2.substr(XmlIndex);
          } else {
            XmlData = XmlTag + buffer3;
            grid_console(data.Id + '.Wait (partial IMAGE) : "' + XmlData + '"');
            buffer2 = "";
            return true;
          }
        }
      } else if (IsTag(XmlTag, '<LINE')) {
        tiles[tindex].LINE.push(grid_ParseLine(XmlTag));
      } else if (IsTag(XmlTag, '<MODBUTTON')) {
        grid_ModifyButton(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODCIRCLE')) {
        grid_ModifyCircle(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODIMAGE')) {
        grid_ModifyImage(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODLINE')) {
        grid_ModifyLine(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODPOLYGON')) {
        grid_ModifyPolygon(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODRECT')) {
        grid_ModifyRectangle(data, XmlTag);
      } else if (IsTag(XmlTag, '<MODTEXT')) {
        closetag = XmlTag.indexOf('/>');
        if (closetag > -1) {
          grid_ModifyText(data, XmlTag);
        } else {
          modtextend = buffer2.indexOf('</MODTEXT>', XmlIndex);
          if (modtextend > -1) {
            XmlTag = XmlTag + buffer2.substring(XmlIndex, modtextend + 10);
            grid_console(data.Id + '.ReRead : "' + XmlTag + '"');
            grid_ModifyText(data, XmlTag);
            XmlIndex = modtextend + 10;
            buffer3 = buffer2.substr(XmlIndex);
          } else {
            XmlData = XmlTag + buffer3;
            grid_console(data.Id + '.Wait (partial MODTEXT) : "' + XmlData + '"');
            buffer2 = "";
            return true;
          }
        }//else
      } else if (IsTag(XmlTag, '<MODTILE')) {
        grid_ModifyTile(data, XmlTag);
      } else if (IsTag(XmlTag, '<POLYGON')) {
        tiles[tindex].POLYGON.push(grid_ParsePolygon(XmlTag));
      } else if (IsTag(XmlTag, '<RECT')) {
        tiles[tindex].RECTANGLE.push(grid_ParseRectangle(XmlTag));
      } else if (IsTag(XmlTag, '<TEXT')) {
        closetag = XmlTag.indexOf('/>');
        if (closetag > -1) {
          tiles[tindex].TEXT.push(grid_ParseText(XmlTag, data.FontSize, data.FontName));
        } else {
          textend = buffer2.indexOf('</TEXT>', XmlIndex);
          if (textend > -1) {
            XmlTag = XmlTag + buffer2.substring(XmlIndex, textend + 7);
            grid_console(data.Id + '.ReRead : "' + XmlTag + '"');
            tiles[tindex].TEXT.push(grid_ParseText(XmlTag, data.FontSize, data.FontName));
            XmlIndex = textend + 7;
            buffer3 = buffer2.substr(XmlIndex);
          } else {
            XmlData = XmlTag + buffer3;
            grid_console(data.Id + '.Wait (partial TEXT) : "' + XmlTag + '"');
            buffer2 = "";
            return true;
          }
        }
      } else if (IsTag(XmlTag, '</TILE>')) {
        if (GridFormatted) {
          //tiles[tindex].TW = grids[gindex].TW - grids[gindex].PADH;
          //tiles[tindex].TH = grids[gindex].TH - grids[gindex].PADV;
          tiles[tindex].TW = grids[gindex].TW;
          tiles[tindex].TH = grids[gindex].TH;
          grid_FormatTile(tindex);
        } else {
          grid_console(data.Id + '.DISCARD : ' + XmlTag);
        }
      } else if (IsTag(XmlTag, '<TILE')) {
        tindex = grid_CreateTile(data, XmlTag, gindex);
      } else { //unknown tag
        grid_console(data.Id + '.DISCARD : ' + XmlTag);
      }//else

    }//while
  }

  catch (ex) {
    console.log("exception thrown in grid_HandlePipeEvent");
  }

  finally {
    return true;
  }

}


function grid_HandlePipeEvent(event, file, buffer) {

  event.preventDefault();
  var data = event.data || event;
  if (file != data.Pipe) return true;
  if (buffer.length == 0) return true;

  // Pipe Data may come over in pieces, so need to build a commplete message.
  // You may also need to parcel out messages.
  // And sometimes Data is so long that once you get a <GRID, you should 
  // start parsing each tile as you go.
  for (var ii = 0; ii < grids.length; ii++) {
    if (grids[ii].ID == data.Id) {
      gindex = ii;
    }//if
  }//for

  grid_ParsePipeData(data, buffer);

} // grid_HandlePipeEvent

function grid_HandleMouseDownEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  grid_CtlEvent(data);
  return true;
} // grid_HandleMouseDownEvent

function grid_HandleMouseUpEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  grid_CtlEvent(data);
  return true;
} // grid_HandleMouseUpEvent


function XmlGetAttribute(tag, attribute) {

  try {
    var attribute2 = attribute.toLocaleLowerCase();
    var StartIndex = 0;
    var quote;
    var value;
    var temp;
    var temp2;
    while ((tag.substr(StartIndex, 1) != '<') && (StartIndex < tag.length)) {
      StartIndex++;
    }
    if (StartIndex == tag.length) return "";
    var EndIndex = tag.indexOf('>');
    var tag2 = tag.substring(StartIndex, EndIndex + 1);

    var attr = tag2.indexOf(' ' + attribute2 + '=');
    if (attr == -1) return "";

    temp = tag2.substr(attr + attribute2.length + 1, tag2.length - attr);

    quote = temp.indexOf("'");
    if (quote > -1) {
      temp2 = temp.substr(quote + 1);
      quote = temp2.indexOf("'");
      if (quote > -1) {
        value = temp2.substr(0, quote);
        return value.toUpperCase();
      }
    }

    quote = temp.indexOf('"');
    if (quote > -1) {
      temp2 = temp.substr(quote + 1);
      quote = temp2.indexOf('"');
      if (quote > -1) {
        value = temp2.substr(0, quote);
        return value.toUpperCase();
      }
    }
    return "";
  }

  catch (ex) {
    console.log("exception thrown in XmlGetAttribute");
  }

  finally { }

}// XmlGetAttribute

function XmlGetText(tag) {

  try {
    var EndTag = tag.indexOf('>');
    var temp = tag.substr(EndTag + 1);
    var StartTag = temp.indexOf('<');
    return temp.substr(0, StartTag);
  }

  catch (ex) {
    console.log("exception thrown in XmlGetText");
  }

  finally { }
}//XmlGetText


function rgb2hex(rgb) {

  var lp = rgb.indexOf('(');
  var comma = rgb.indexOf(',');
  var r = rgb.substring(lp + 1, comma);
  var temp = rgb.substr(comma + 1);
  comma = temp.indexOf(',');
  var g = temp.substr(0, comma);
  var temp2 = temp.substr(comma + 1);
  var rp = temp2.indexOf(')');
  var b = temp2.substr(0, rp);
  var r2 = ~Number(r) & 0Xff;
  var rx = Number(r2).toString(16);
  if (rx.length < 2) {
    rx = '0' + rx;
  }
  var g2 = ~Number(g) & 0Xff;
  var gx = Number(g2).toString(16);
  if (gx.length < 2) {
    gx = '0' + gx;
  }
  var b2 = ~Number(b) & 0Xff;
  var bx = Number(b2).toString(16);
  if (bx.length < 2) {
    bx = '0' + bx;
  }
  var hex = '#' + rx + gx + bx;
  return hex.toLowerCase();
}//rgb2hex


function InvertColor(bclr) {

  var red = '0x' + bclr.substr(1, 2);
  var red2 = ~Number(red) & 0xff;
  var red3 = Number(red2).toString(16);
  if (red3.length < 2) {
    red3 = '0' + red3;
  }

  var green = '0x' + bclr.substr(3, 2);
  var green2 = ~Number(green) & 0xff;
  var green3 = Number(green2).toString(16);
  if (green3.length < 2) {
    green3 = '0' + green3;
  }

  var blue = '0x' + bclr.substr(5, 2);
  var blue2 = ~Number(blue) & 0xff;
  var blue3 = Number(blue2).toString(16);
  if (blue3.length < 2) {
    blue3 = '0' + blue3;
  }

  var bclr2 = '#' + red3 + green3 + blue3;
  return bclr2;

}//InvertColor


function ParseDimsAttr(tag) {

  try {
    var comma;
    var lim = {};
    var limits = XmlGetAttribute(tag, 'dims');
    if (limits.length > 0) {
      comma = limits.indexOf(',');
      lim.col = Number(limits.substr(0, comma));
      lim.row = Number(limits.substr(comma + 1));
      return (lim);
    }
    lim.col = 0;
    lim.row = 0;
    return (lim);
  }

  catch (ex) {
    console.log("exception thrown in ParseDimsAttr");
  }

  finally { }
} //ParseDimsAttr


function ParsePadAttr(tag) {

  try {
    var pad = XmlGetAttribute(tag, 'pad');
    if (pad.length == 0) {
      return 0;
    }
    return Number(pad);
  }

  catch (ex) {
    console.log("exception thrown in ParsePadAttr");
  }

  finally { }
} //ParsePadAttr

function ParsePad2Attr(tag, attr, pad2) {

  try {
    var pad = XmlGetAttribute(tag, attr);
    if (pad.length == 0) {
      return pad2;
    }
    return Number(pad);
  }

  catch (ex) {
    console.log("exception thrown in ParsePadAttr(2)");
  }

  finally { }
} //ParsePad2Attr

function ParseCursAttr(tag) {

  try {
    var cursor = XmlGetAttribute(tag, 'curs');
    if (cursor.length == 0) {
      return -1;
    }
    return Number(cursor);
  }

  catch (ex) {
    console.log("exception thrown in ParseCursAttr");
  }

  finally { }
} //ParseCursAttr


function ParsePosAttr(tag) {

  try {
    var position = XmlGetAttribute(tag, 'pos');
    var comma = position.indexOf(',');
    var column = Number(position.substr(0, comma));
    var row = Number(position.substr(comma + 1));
    var pos = {};
    pos.row = row;
    pos.col = column;
    return (pos);
  }

  catch (ex) {
    console.log("exception thrown in ParsePosAttr");
  }

  finally { }

} //ParsePosAttr

function ParseScrollBarAttr(tag) {

    try {
        var scrollbar = XmlGetAttribute(tag, 'scrollbar');
        if (scrollbar.length > 0) {
            return (scrollbar);
        } else {
            return (scrollbar);
        }
    }

    catch (ex) {
        console.log("exception thrown in ParseScrollBarAttr");
    }

    finally { }

} //ParsePosAttr


function DePercent(tag) {
  var percent = tag.indexOf('%');
  if (percent > -1) {
    return Number(tag.substr(0, percent));
  } else {
    return Number(tag);
  }
} //DePercent


function ParseSizeAttr(tag) {

  try {
    var size = XmlGetAttribute(tag, 'size');
    if (size.length > 0) {
      var comma = size.indexOf(',');
      var X = size.substr(0, comma);
      var X2 = DePercent(X);
      var Y = size.substr(comma + 1);
      var Y2 = DePercent(Y);
      return ({ lengthX: X2, lengthY: Y2 });
    } else {
      return ({ lengthX: -1, lengthY: -1 });
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseSizeAttr");
  }

  finally { }
} //ParseSizeAttr


function ParseLocAttr(tag) {

  try {
    var loc = XmlGetAttribute(tag, 'loc');

    if (loc.length > 0) {
      var comma = loc.indexOf(',');
      var X = loc.substr(0, comma);
      var X2 = DePercent(X);
      var Y = loc.substr(comma + 1);
      var Y2 = DePercent(Y);
      return ({ offsetX: X2, offsetY: Y2 });
    } else {
      return ({ offsetX: 50, offsetY: 50 });
    }

    return (location);
  }

  catch (ex) {
    console.log("exception thrown in ParseLocAttr");
  }

  finally { }


} //ParseLocAttr


function ParseElocAttr(tag) {

  try {
    var eloc = XmlGetAttribute(tag, 'eloc');

    if (eloc.length > 0) {
      var comma = eloc.indexOf(',');
      var X = eloc.substr(0, comma);
      var X2 = DePercent(X);
      var Y = eloc.substr(comma + 1);
      var Y2 = DePercent(Y);
      return ({ endX: X2, endY: Y2 });
    } else {
      return ({ endX: 0, endY: 0 });
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseElocAttr");
  }

  finally { }
} //ParseElocAttr


function ParseVrtxAttr(tag, vno) {

  try {
    var vrtx = XmlGetAttribute(tag, 'vrtx' + vno);

    if (vrtx.length > 0) {
      var comma = vrtx.indexOf(',');
      var X = vrtx.substr(0, comma);
      var X2 = DePercent(X);
      var Y = vrtx.substr(comma + 1);
      var Y2 = DePercent(Y);
      return ({ valid: 1, offsetX: X2, offsetY: Y2 });
    } else {
      return ({ valid: 0, offsetX: 0, offsetY: 0 });
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseVrtxAttr");
  }

  finally { }
} //ParseVrtxAttr

function ParseTouchAttr(tag) {

  try {
    var touch = XmlGetAttribute(tag, 'touch');
    if (touch.length > 0) {
      return (Number(touch));
    } else {
      return (4);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseTouchAttr");
  }

  finally { }
}//ParseTouchAttr

function ParseFontAttr(tag, fontsize) {

  try {
    var font = XmlGetAttribute(tag, 'font');
    var font2;

    if (font.length > 0) {
      var temp = Number(font);
      var plus = font.indexOf('+');
      var minus = font.indexOf('-');
      if (plus > -1) {
        font2 = Number(fontsize) + Number(font.slice(1));
      } else if (minus > -1) {
        font2 = Number(fontsize) - Number(font.slice(1));
      } else {
        font2 = Number(font);
      }
      return (font2.toString(10));
    } else {
      return (fontsize);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseFontAttr");
  }

  finally { }

} //ParseFontAttr


function ParseWtAttr(tag) {

  return (ParseWtAttr2(tag, "2"));

} //ParseWtAttr

function ParseWtAttr2(tag, defaultWt) {

  try {
    var weight = XmlGetAttribute(tag, 'wt');
    if (weight.length > 0) {
      return (weight);
    } else {
      return (defaultWt);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseWtAttr2");
  }

  finally { }

} //ParseWtAttr

function ParseLookAttr(tag) {

  try {
    var list = XmlGetAttribute(tag, 'look');
    if (list.length > 0) {
      return (ParseList(list));
    } else {
      var style = [];
      return (style);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseLookAttr");
  }

  finally { }

} //ParseLookAttr

function ParseClrAttr(tag, clr) {

  try {
    var clr2 = XmlGetAttribute(tag, 'clr');
    if (clr2.length == 0) {
      return (clr);
    } else {
      return (GridFixColor(clr2));
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseClrAttr");
  }

  finally { }
} //ParseClrAttr

function ParseBclrAttr(tag, bclr) {

  try {
    var bclr2 = XmlGetAttribute(tag, 'bclr');
    if (bclr2.length == 0) {
      return (bclr);
    } else {
      return (GridFixColor(bclr2));
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseBclrAttr");
  }

  finally { }
} //ParseBclrAttr

function ParseFclrAttr(tag, fclr) {

  try {
    var fclr2 = XmlGetAttribute(tag, 'fclr');
    if (fclr2.length == 0) {
      return (fclr);
    } else {
      return (GridFixColor(fclr2));
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseFclrAttr");
  }

  finally { }
} //ParseFclrAttr

function ParseSclrAttr(tag) {

  try {
    var sclr = XmlGetAttribute(tag, 'sclr');
    if (sclr.length == 0) {
      return (sclr);
    } else {
      return (GridFixColor(sclr));
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseSclrAttr");
  }

  finally { }
} //ParseSclrAttr


function ParseSlctAttr(tag) {

  try {
    var slct = XmlGetAttribute(tag, 'slct');
    if (slct.length == 0) {
      return (false);
    } else if (slct == '0') {
      return (false);
    } else if (slct == '1') {
      return (true);
    } else {
      return (false);;
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseSlctAttr");
  }

  finally { }

} //ParseSlctAttr


function ParseList(list) {

  try {
    var list2 = [];
    var temp = list;
    var comma;
    var value;

    while (temp.length > 0) {
      comma = temp.indexOf(',', 0);
      if (comma == -1) {
        list2.push(temp);
        return (list2);
      }//if
      value = temp.substring(0, comma);
      list2.push(value);
      temp = temp.substr(comma + 1);
    }//while
  }

  catch (ex) {
    console.log("exception thrown in ParseList");
  }

  finally { }
}//ParseList

function InList(value, list) {

  try {
    var value2;

    for (var ii = 0; ii < list.length; ii++) {
      value2 = list[ii];
      if (value == value2) {
        return (true);
      }//if
    }//for

    return (false);
  }

  catch (ex) {
    console.log("exception thrown in InList");
  }

  finally { }

}//InList


function InList2(value, list) {

  try {
    var value2;

    for (var ii = 0; ii < list.length; ii++) {
      value2 = list[ii];
      if (value == value2) {
        return (ii);
      }//if
    }//for

    return (-1);
  }

  catch (ex) {
    console.log("exception thrown in InList");
  }

  finally { }

}//InList2


function ParseLayerOnAttr(tag) {

  try {
    var list = XmlGetAttribute(tag, 'layeron');
    if (list.length > 0) {
      return (ParseList(list));
    } else {
      var layer = [];
      //layer.push('0');
      return (layer);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseLayerOnAttr");
  }

  finally { }

}//ParseLayerOnAttr

function ParseLayerOffAttr(tag) {

  try {
    var list = XmlGetAttribute(tag, 'layeroff');
    if (list.length > 0) {
      return (ParseList(list));
    } else {
      var layer = [];
      return (layer);
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseLayerOffAttr");
  }

  finally { }
}//ParseLayerOffAttr

function ParseAlignAttr(tag) {

  try {
    var alignment = {};
    var align = XmlGetAttribute(tag, 'align').toLowerCase();
    alignment.va = 'm';
    alignment.ha = 'c';
    if (align.length > 0) {
      var comma = align.indexOf(',');
      if (comma > -1) {
        alignment.va = align.substr(0, comma);
        alignment.ha = align.substr(comma + 1);
      } else {
        if (align == "left") {
          alignment.ha = align;
        } else if (align == "center") {
          alignment.ha = align;
        } else {
        }
      }
    }
    return (alignment);
  }

  catch (ex) {
    console.log("exception thrown in ParseAlignAttr");
  }

  finally { }
} //ParseAlignAttr

function ParseLayerAttr(tag) {

  try {
    var layer = XmlGetAttribute(tag, 'layer');
    if (layer.length > 0) {
      return (layer);
    } else {
      return ('0');
    }
  }

  catch (ex) {
    console.log("exception thrown in ParseAlignAttr");
  }

  finally { }
} //ParseAlignAttr

function grid_FindGrid(id) {

  try {

    if (id.length == 0) {
      return -1;
    }

    for (var ii = 0; ii < grids.length; ii++) {
      if (grids[ii].ID == id) {
        return ii;
      }//if
    }//for

    return -1;
  }

  catch (ex) {
    console.log("exception thrown in grid_FindGrid");
  }

  finally { }
}//grid_FindGrid

function GuessPath(path) {
  try {

    if (path.length > 0) {
      if (path == "/frh/jcgtp/jguidev0.stm") {
        return "/frh/gui/";
      //} else if (path == "/td/tpdbtest.stm") {
      //  return "/frh/cgtp/";
      } else {
        //return url2path(path);
        return "/frh/cgtp/";
      }
    }

    return "";
  }
  catch (ex) {
    console.log("exception thrown in GuessPath");
    return "";
  }

  finally { }
}//GuessPath

function url2path(url) {
  try {
    if (url.length == -1) return "";

    var path = "/";
    var subpath = "";

    var url2 = url.split("/");
    for (var intII = 1; intII < url2.length - 1; intII++) {
      subpath = url2[intII];
      path += subpath + "/";
    }
    return path;
  }
  catch (ex) {
    console.log("exception thrown in url2path");
    return "";
  }

  finally { }
}//url2path


function nopath(url) {
  try {
    if (url.length == -1) return "";

    var url2 = url.split("/");
    if (url2.length == -1) return url;
    if (url2.length == 1) return url;
    return url2[url2.length];
  }
  catch (ex) {
    console.log("exception thrown in nopath");
    return "";
  }

  finally { }
}//nopath

function grid_CreateGrid(data, tag) {

  try {
    //sometimes a grid comes thru twice, so you want to reuse whats there
    var url = window.location.pathname;
    var path = GuessPath(url);

    var pad;
    var gID = grid_FindGrid(data.Id);
    if (gID > -1) {
      grid_console(data.Id + ".ReUse");
      //flush buttons
      for (var ii = 0; ii < buttons.length;) {
        if (buttons[ii].GID == data.Id) {
          buttons.splice(ii, 1);
        } else {
          ii++;
        }
      }
      //flush tiles
      for (ii = 0; ii < tiles.length;) {
        if (tiles[ii].GID == data.Id) {
          grid_DeleteTile2(ii);
          tiles.splice(ii, 1);
        } else {
          ii++;
        }
      }
      //reset grid 
      grids[gID].URL = url;
      grids[gID].PATH = path;
      grids[gID].BCLR = ParseBclrAttr(tag, data.BackColor);
      grids[gID].CLR = ParseFclrAttr(tag, data.ForeColor);
      grids[gID].TM = ParseTouchAttr(tag);
      grids[gID].H = Number(data.height);
      grids[gID].W = Number(data.width);
      pad = ParsePadAttr(tag);
      grids[gID].PADL = ParsePad2Attr(tag, 'padl', pad);
      grids[gID].PADR = ParsePad2Attr(tag, 'padr', pad);
      grids[gID].PADT = ParsePad2Attr(tag, 'padt', pad);
      grids[gID].PADB = ParsePad2Attr(tag, 'padb', pad);
      grids[gID].PADH = 0;
      grids[gID].PADV = 0;
      grids[gID].FTH = 0.0;
      grids[gID].FTW = 0.0;
      grids[gID].TH = 0;
      grids[gID].TW = 0;
      grids[gID].DROW = 0;
      grids[gID].DCOL = 0;
      grids[gID].SROW = 0;
      grids[gID].SCOL = 0;
      while (grids[gID].IMAGES.length > 0) {
        grids[gID].IMAGES.pop();
      }

      grids[gID].CTX.clearRect(0, 0, grids[gID].W, grids[gID].H);
      return (gID);
    }

    //create a brand new grid

    //need to paint bclr and zero out previous tile activity
    var id = data.Id + 'C';
    var out = '<canvas id="' + id + '" width="' + data.width + '" height="' + data.height + '" style="background-color:transparent;"/>';
    //var out = '<canvas id="' + id + '" width="' + data.width + '" height="' + data.height + '" style="background-color:transparent; border:1px solid #000000;"/>';
    data.$this.html(out);

    var grid2 = {};
    grid2.ID = data.Id;
    grid2.ID2 = id;
    grid2.URL = url;
    grid2.PATH = path;
    grid2.BCLR = ParseBclrAttr(tag, data.BackColor);
    grid2.CLR = ParseFclrAttr(tag, data.ForeColor);
    grid2.TM = ParseTouchAttr(tag);
    grid2.H = Number(data.height);
    grid2.W = Number(data.width);
    pad = ParsePadAttr(tag);
    grid2.PADL = ParsePad2Attr(tag, 'padl', pad);
    grid2.PADR = ParsePad2Attr(tag, 'padr', pad);
    grid2.PADT = ParsePad2Attr(tag, 'padt', pad);
    grid2.PADB = ParsePad2Attr(tag, 'padb', pad);
    grid2.PADH = 0;
    grid2.PADV = 0;
    grid2.FTH = 0.0;
    grid2.FTW = 0.0;
    grid2.TH = 0;
    grid2.TW = 0;
    grid2.DROW = 0;
    grid2.DCOL = 0;
    grid2.SROW = 0;
    grid2.SCOL = 0;
    grid2.IMAGES = [];

    grid2.PAGE = document.getElementById(id);
    grid2.CTX = grid2.PAGE.getContext("2d");
    grid2.CTX.clearRect(0, 0, grid2.W, grid2.H);

    grid2.PAGE.addEventListener('mousedown', function (event) {
      grid_React_Down(event.offsetX, event.offsetY, data);
    }, false);

    grid2.PAGE.addEventListener('mouseup', function (event) {
      grid_React_Up(event.offsetX, event.offsetY, data);
    }, false);

    var gindex = grids.length;

    grids.push(grid2);
    return (gindex);
  }

  catch (ex) {
    console.log("exception thrown in grid_CreateGrid");
  }

  finally { }
} // grid_CreateGrid

function grid_SelectTile (tid) {

  var grid = grids[tiles[tid].GINDEX];
  var ctx = grid.CTX;

  if (tiles[tid].SLCT) {
    ctx.strokeStyle = tiles[tid].SCLR;
    ctx.lineWidth = "1";
  } else {
    ctx.strokeStyle = tiles[tid].BCLR;
    ctx.lineWidth = "2";
  } 
  ctx.strokeRect(tiles[tid].startX + 1, tiles[tid].startY + 1, tiles[tid].TW - 2, tiles[tid].TH - 2);
} //grid_SelectTile

// button fields
// GID - named ID needed for <grid>
// ID - named ID needed for <modbutton>
// X, Y - upper left corner
// lX, lY - width and height of button
// tw,th - width and height of cell 
// other - data.fDeviceId
// TPRQ - tprq
// KCOD = kcod
// bclr - background color

function grid_button_release(x, y, data) {

  var gID = data.Id;

  var button; //check buttons first
  for (var ii = 0; ii < buttons.length; ii++) {
    button = buttons[ii];
    if (button.GID == gID) {
      if ((x > button.X) && (x < button.X + button.lX)) {
        if ((y > button.Y) && (y < button.Y + button.lY)) {
          if (button.TPRQ.length > 0) {
            var tprq2 = button.TPRQ.split(",");
            var request_code = Number(tprq2[0]);
            var function_code = Number(tprq2[1]);
            if ((function_code < 0) && (function_code > -16)) {
              ++function_code;
              function_code = 0xFFFFFFFF + function_code;
            } else {
              function_code = function_code & 0x7FFFFFFF;
            }
            grid_console(gID + '.TPRQ : ' + data.fDeviceId + ', ' + request_code + ', ' + function_code);
            top.rpcmc_tpextreq2(data.fDeviceId, request_code, function_code);
          }
          if (button.KCOD.length > 0) {
            grid_console(gID + '.KCOD : ' + button.KCOD);
            top.rpcmc_sendKeyCode(Number(button.KCOD));
          }
          if (button.URL.length > 0) {
            grid_console(gID + '.URL (' + data.myFrameName + ') : ' + button.URL);
            top.rpcmc_tplink_new_url(data.fDeviceId, button.URL, data.myFrameName);
          }
          //sometimes when you press a tile it should be selected but you cannot count on the ap to do that
          //you also have to unselect the previously  selected tiles
          for (var jj = 0; jj < tiles.length; jj++) {
            if (tiles[jj].SLCT && (tiles[jj].SCLR.length > 0)) {
              tiles[jj].SLCT = false;
              grid_SelectTile (jj);
            }
          }
          if (tiles[button.TILE].SCLR.length > 0) {
            tiles[button.TILE].SLCT = true;
            grid_SelectTile (button.TILE);
          }
          return (1);
        } //not in y range
      } //not in x range
    }//not gID
  }//for

  return (0);

} //grid_button_release

function grid_button_press(x, y, data) {

  var gID = data.Id;

  var button; //check buttons first
  for (var ii = 0; ii < buttons.length; ii++) {
    button = buttons[ii];
    if (button.GID == gID) {
      if ((x > button.X) && (x < button.X + button.lX)) {
        if ((y > button.Y) && (y < button.Y + button.lY)) {
          if (button.TPRQ2.length > 0) {
            var tprq2 = button.TPRQ2.split(",");
            var request_code = Number(tprq2[0]);
            var function_code = Number(tprq2[1]);
            if ((function_code < 0) && (function_code > -16)) {
              ++function_code;
              function_code = 0xFFFFFFFF + function_code;
            } else {
              function_code = function_code & 0x7FFFFFFF;
            }
            grid_console(gID + '.TPRQ2 : ' + data.fDeviceId + ', ' + request_code + ', ' + function_code);
            top.rpcmc_tpextreq2(data.fDeviceId, request_code, function_code);
          }
          if (button.KCOD2.length > 0) {
            grid_console(gID + '.KCOD2 : ' + button.KCOD2);
            top.rpcmc_sendKeyCode(Number(button.KCOD2));
          }
          if (button.URL2.length > 0) {
            grid_console(gID + '.URL2 (' + data.myFrameName + ') : ' + button.URL2);
            top.rpcmc_tplink_new_url(data.fDeviceId, button.URL2, data.myFrameName);
          }
          if ((button.TPRQ2.length > 0) || (button.KCOD2.length > 0) || (button.URL2.length > 0)) {
            return (1);
          }
        } //not in y range
      } //not in x range
    }//not gID
  }//for

  return (0);

} //grid_button_press

function grid_get_scrollingtile(tile) {

  var rectangle;

  for (var rr = 0; rr < tile.RECTANGLE.length; rr++) {
    rectangle = tile.RECTANGLE[rr];
    if (rectangle.ID == "SCROLLRANGE") {
      return (1);
    }
  }//for
  return (0);

} //grid_get_scrollingtile

function grid_dialog_box_release(x, y, data, GID) {

  var gID = data.Id;

  if ((gID == "DlgBox") || (gID == "CGGrid2")) {
  } else {
    return (0);
  }

  var tile;
  var rr;
  var rectangle;
  var startX;
  var startY;
  var offsetX;
  var offsetY;
  var lengthX;
  var lengthY;
  var posX;
  var posY;
  var y2;
  var row;

  for (ii = 0; ii < tiles.length; ii++) {//special test for scrolling
    tile = tiles[ii];
    if (tile.GID == gID) {
      if (grid_get_scrollingtile(tile) > 0) {
        startX = tile.startX;
        startY = tile.startY;
        for (rr = 0; rr < tile.RECTANGLE.length; rr++) {
          rectangle = tile.RECTANGLE[rr];
          offsetX = Percent2Pixel(rectangle.LOC.offsetX, tile.TW);
          offsetY = Percent2Pixel(rectangle.LOC.offsetY, tile.TH);
          lengthX = Percent2Pixel(rectangle.SIZE.lengthX, tile.TW);
          lengthY = Percent2Pixel(rectangle.SIZE.lengthY, tile.TH);
          posX = grid_PosX(startX, offsetX, lengthX, rectangle.ALIGN.ha);
          posY = grid_PosY(startY, offsetY, lengthY, rectangle.ALIGN.va);
          if ((x > posX) && (x < posX + lengthX)) {
            y2 = 1.0 + (y / grids[GID].FTH); // do y to row conversion
            row = Math.round(y2);
            //if ((y > posY) && (y < posY + lengthY)) {
            top.rpcmc_setVar(tile.TP, mGrid_ReleaseRowVar, row);
            top.rpcmc_setVar(tile.TP, mGrid_ReleaseColVar, tile.COLUMN);
            top.rpcmc_setVar(tile.TP, mGrid_ReleaseEventVar, 1);
            grid_console(gID + '.KEY (' + data.myFrameName + ') : ^W');
            top.rpcmc_sendKeyCode(ctl_w_c); //send ^W (end of transmission block)
            return (1);
            //}//in y range
          }//in x range
        }//for
      }//scrollingtile
    }//not gID
  }//for

  for (ii = 0; ii < tiles.length; ii++) {//special test for tile clicks
    tile = tiles[ii];
    if (tile.GID == gID) {
      if (grid_get_scrollingtile(tile) == 0) {
        startX = CalcOffset(tile.COLUMN - grids[GID].SCOL, grids[GID].TW) + grids[GID].PADL;
        startY = CalcOffset(tile.ROW - grids[GID].SROW, grids[GID].TH) + grids[GID].PADT;
        for (rr = 0; rr < tile.RECTANGLE.length; rr++) {
          rectangle = tile.RECTANGLE[rr];
          offsetX = Percent2Pixel(rectangle.LOC.offsetX, tile.TW);
          offsetY = Percent2Pixel(rectangle.LOC.offsetY, tile.TH);
          lengthX = Percent2Pixel(rectangle.SIZE.lengthX, tile.TW);
          lengthY = Percent2Pixel(rectangle.SIZE.lengthY, tile.TH);
          posX = grid_PosX(startX, offsetX, lengthX, rectangle.ALIGN.ha);
          posY = grid_PosY(startY, offsetY, lengthY, rectangle.ALIGN.va);
          if ((x > posX) && (x < posX + lengthX)) {
            if ((y > posY) && (y < posY + lengthY)) {
              top.rpcmc_setVar(tile.TP, mGrid_ReleaseRowVar, tile.ROW);
              top.rpcmc_setVar(tile.TP, mGrid_ReleaseColVar, tile.COLUMN);
              top.rpcmc_setVar(tile.TP, mGrid_ReleaseEventVar, 1);
              grid_console(gID + '.KEY (' + data.myFrameName + ') : ^W');
              top.rpcmc_sendKeyCode(ctl_w_c); //send ^W (end of transmission block)
              return (1);
            }//not in y range
          }//not in x range
        }//for
      }//scrollingtile
    }//not gID
  }//for

  return (0);

} //grid_dialog_box_release

function grid_React_Up(x, y, data) {

  var gID = data.Id;
  if (grid_button_release(x, y, data) > 0) return;
  var GID = grid_FindGrid(gID);
  var pagellength = grids[GID].DROW; //Calculate how long a page is in rows


  //Special handling for PageDown
  if (gID == "PageDown") {
    grid_console(gID + '.KEY (' + data.myFrameName + ') : SHIFT DN Arrow');
    top.rpcmc_sendKeyCode(dn_arw_s_c);
    //for (ii = 2; ii < pagelength; ii++) {//don't want to scroll too much
    //  grid_console(gID + '.KEY (' + data.myFrameName + ') : DN Arrow');
    //  top.rpcmc_sendKeyCode(dn_arw_c);
    //}
    return;
  }

  if (gID == "PageUp") {
    grid_console(gID + '.KEY (' + data.myFrameName + ') : SHIFT UP Arrow');
    top.rpcmc_sendKeyCode(up_arw_s_c);
    //for (ii = 2; ii < pagelength; ii++) {//don't want to scroll too much
    //  grid_console(gID + '.KEY (' + data.myFrameName + ') : DN Arrow');
    //  top.rpcmc_sendKeyCode(up_arw_c);
    //}
    return;
  }
  var touchmask = parseInt(data.touch);
  if ((touchmask & mGrid_ReleaseBit) > 0) {
    if (grid_dialog_box_release(x, y, data, GID) > 0) return;
    var x2 = 1.0 + (x / grids[GID].FTW); // do x to column conversion
    var column = Math.round(x2);
    var y2 = 1.0 + (y / grids[GID].FTH); // do y to row conversion
    var row = Math.round(y2);

    top.rpcmc_setVar(data.Name, mGrid_ReleaseRowVar, row);
    top.rpcmc_setVar(data.Name, mGrid_ReleaseColVar, column);
    top.rpcmc_setVar(data.Name, mGrid_ReleaseEventVar, 1);
    grid_console(gID + '.KEY (' + data.myFrameName + ') : ^W');
    top.rpcmc_sendKeyCode(ctl_w_c); //send ^W (end of transmission block)
  }

}//grid_React_Up

function grid_React_Down(x, y, data) {

  if (grid_button_press(x, y, data) > 0) return;

  var touchmask = parseInt (data.touch);
  if ((touchmask & mGrid_PressBit) > 0) {
    var gID = data.Id;
    var GID = grid_FindGrid(gID);
    var x2 = 1.0 + (x / grids[GID].FTW); // do x to column conversion
    var column = Math.round(x2);
    var y2 = 1.0 + (y / grids[GID].FTH); // do y to row conversion
    var row = Math.round(y2);
    top.rpcmc_setVar(data.Name, mGrid_PressRowVar, row);
    top.rpcmc_setVar(data.Name, mGrid_PressColVar, column);
    top.rpcmc_setVar(data.Name, mGrid_PressEventVar, 1);
    grid_console(gID + '.KEY (' + data.myFrameName + ') : ^W');
    top.rpcmc_sendKeyCode(ctl_w_c); //send ^W (end of transmission block)
  }

}//grid_React_Down

function grid_FindTile(data, id, row, column) {

  try {
    var ii;

    //by ID
    if (id.length > 0) {
      for (ii = 0; ii < tiles.length; ii++) {
        if ((tiles[ii].GID == data.Id) && (tiles[ii].ID == id)) {
          return ii;
        }//if
      }//for
    }

    //by row and columnd
    for (ii = 0; ii < tiles.length; ii++) {
      if ((tiles[ii].GID == data.Id) && (tiles[ii].ROW == row) && (tiles[ii].COLUMN == column)) {
        return ii;
      }//if
    }//for

    return -1;
  }

  catch (ex) {
    console.log("exception thrown in grid_FindTile");
  }

  finally { }
}//grid_FindTile

function grid_CreateTile(data, tag, GID) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var position = ParsePosAttr(tag);
    var row = position.row;
    var column = position.col;

    var tid = grid_FindTile(data, id, row, column);
    if (tid > -1) {
      //grid_console(data.Id + '.ReUse tile(' + tid + ')');
      tiles[tid].bX = 0;
      tiles[tid].bY = 0;
      tiles[tid].blX = 0;
      tiles[tid].blY = 0;
      tiles[tid].startX = 0;
      tiles[tid].startY = 0;
      tiles[tid].tX = 0;
      tiles[tid].tY = 0;
      tiles[tid].tlX = 0;
      tiles[tid].tlY = 0;
      tiles[tid].SLCT = false;
      grid_ClearTile2(tid);
      grid_DeleteTile2(tid);
      return tid;
    }

    var clr = grids[GID].CLR;
    var touchmask = grids[GID].TM;
    var path = grids[GID].PATH;
    var layers = ParseLayerOnAttr(tag);
    var value;

    var tile2 = {};

    tile2.GID = data.Id;
    tile2.ID = id;
    tile2.otherID = data.fDeviceId;
    tile2.GINDEX = GID;
    tile2.PATH = path;
    tile2.ROW = row;
    tile2.COLUMN = column;
    tile2.GW = Number(data.width);
    tile2.GH = Number(data.height);
    tile2.TW = 0;
    tile2.TH = 0;
    tile2.BCLR = XmlGetAttribute(tag, 'bclr');
    if (tile2.BCLR.length == 0) {
      tile2.BCLR = data.BackColor;
    }
    tile2.SCLR = ParseSclrAttr(tag);
    tile2.SLCT = ParseSlctAttr(tag);
    tile2.CLR = ParseFclrAttr(tag, clr);
    tile2.TP = data.Name;
    tile2.TM = touchmask;
    tile2.LAYER = [];
    tile2.LAYER.push('0');
    for (var ii = 0; ii < layers.length; ii++) {
      value = layers[ii];
      if (!InList(value, tile2.LAYER)) {
        tile2.LAYER.push(value);
      }//if
    }
    tile2.FONTSIZE = Number(data.FontSize);
    tile2.FONTNAME = data.FontName;
    tile2.BUTTON = [];
    tile2.bX = 0;
    tile2.bY = 0;
    tile2.blX = 0;
    tile2.blY = 0;
    tile2.startX = 0;
    tile2.startY = 0;
    tile2.tX = 0;
    tile2.tY = 0;
    tile2.tlX = 0;
    tile2.tlY = 0;
    tile2.CIRCLE = [];
    tile2.LINE = [];
    tile2.IMAGE = [];
    tile2.POLYGON = [];
    tile2.RECTANGLE = [];
    tile2.TEXT = [];
    tid = tiles.length;

    tiles.push(tile2);
    //grid_console(data.Id + '.Create tile(' + tid + ')');
    return (tid);
  }

  catch (ex) {
    console.log("exception thrown in grid_CreateTile");
  }

  finally { }
} // grid_CreateTile

function grid_DisplayTiles(data, tag, GID) {

  try {
    pad = ParsePadAttr(tag);
    grids[GID].PADH = ParsePad2Attr(tag, 'padh', pad);
    if (grids[GID].PADH > 0) {
      grids[GID].PADL = grids[GID].PADH;
      grids[GID].PADR = grids[GID].PADH;
    }
    grids[GID].PADV = ParsePad2Attr(tag, 'padv', pad);
    if (grids[GID].PADV > 0) {
      grids[GID].PADT = grids[GID].PADV;
      grids[GID].PADB = grids[GID].PADV;
    }

    var position = ParsePosAttr(tag);
    var limits = ParseDimsAttr(tag);
    var tiles_wide = limits.col;
    var tiles_high = limits.row;
    var bclr = XmlGetAttribute(tag, 'bclr');
    var scrollbar = ParseScrollBarAttr(tag);

    grid_console(data.Id + '.DISPLAY Rows: ' + tiles_high + ' Columns: ' + tiles_wide + ' BCLR: ' + bclr);

    grids[GID].SROW = position.row;
    grids[GID].SCOL = position.col;
    grids[GID].DROW = limits.row
    grids[GID].DCOL = limits.col;
    grids[GID].FTW = grids[GID].W / tiles_wide;
    grids[GID].FTH = grids[GID].H / tiles_high;
    grids[GID].TW = Math.floor((grids[GID].W - (grids[GID].PADL + grids[GID].PADR)) / tiles_wide);
    grids[GID].TH = Math.floor((grids[GID].H - (grids[GID].PADT + grids[GID].PADB)) / tiles_high);
    if (bclr.length > 0) {
      grids[GID].BCLR = bclr;
    }//if

    for (var ii = 0; ii < tiles.length; ii++) {
      if (tiles[ii].GID == data.Id) {
        //tiles[ii].TW = grids[GID].TW - grids[GID].PADH;
        //tiles[ii].TH = grids[GID].TH - grids[GID].PADV;
        tiles[ii].TW = grids[GID].TW;
        tiles[ii].TH = grids[GID].TH;
        if (bclr.length > 0) {
          tiles[ii].BCLR = bclr;
        }
      }//if
    }//for

    if (!data.Invisible) {
      grids[GID].CTX.clearRect(0, 0, grids[GID].W, grids[GID].H);
      //PreGrid(GID);
      grid_console(data.Id + '.DISPLAY start');
      for (ii = 0; ii < tiles.length; ii++) {
        if (tiles[ii].GID == data.Id) {
          grid_FormatTile(ii);
        }//if
      }//for
      grid_console(data.Id + '.DISPLAY end');
      if (grids[GID].PADB > 0) {
        grids[GID].CTX.clearRect(0, (grids[GID].H - grids[GID].PADB), grids[GID].W, grids[GID].PADB);
      }
    }//if

  }//try

  catch (ex) {
    console.log("exception thrown in grid_DisplayTiles");
  }

  finally { }

} // grid_DisplayTiles

function CalcOffset(column, twidth) {

  var x = column * twidth;
  return x;
}

function Percent2Pixel(percent, reference) {
  return Math.round((reference * percent) / 100);
}

function PreGrid(GID) {

  try {
    var trows = grids[GID].DROW;
    var tcolumns = grids[GID].DCOL;
    var twidth = grids[GID].TW;
    var theight = grids[GID].TH;
    var row;
    var column;
    var startX;
    var startY;
    var ctx = grids[GID].CTX;
    ctx.strokeStyle = '#00FF00';
    ctx.lineWidth = "1";
    ctx.fillStyle = '#00FF00';
    ctx.font = '9px Verdana';

    for (row = 0; row < trows; row++) {
      startY = CalcOffset(row, theight) + grids[GID].PADT;
      for (column = 0; column < tcolumns; column++) {
        startX = CalcOffset(column, twidth) + grids[GID].PADL;
        ctx.strokeRect(startX, startY, twidth, theight);
        ctx.fillText((row + 1) + ', ' + (column + 1), startX + 5, startY + 10);
      }
    }
  }

  catch (ex) {
    console.log("exception thrown in PreGrid");
  }

  finally { }

}//PreGrid


function grid_FormatTile(TID) {

  try {

    var tile = tiles[TID];
    //sometimes the parse routines will call grid_FormatTile prematurely
    //tile.TW and tile.TH get set after parsing is done
    if ((tile.TW == 0) && (tile.TH == 0)) return;

    var row = tile.ROW;
    var column = tile.COLUMN;
    var grid = grids[tile.GINDEX];

    if ((row < grid.SROW) || (row > (grid.SROW + grid.DROW))) {
      grid_console(tile.GID + '.TILE SkipRow Row: ' + row + ' not in rows: ' + grid.SROW + ' to ' + (grid.SROW + grid.DROW));
      return;
    } else {
      if ((column < grid.SCOL) || (column > (grid.SCOL + grid.DCOL))) {
        grid_console(tile.GID + '.TILE SkipCol Column: ' + column + ' not in cols: ' + grid.SCOL + ' to ' + (grid.SCOL + grid.DCOL));
        return;
      }
    }

    var startX = CalcOffset(column - grid.SCOL, grid.TW) + grid.PADL;
    var startY = CalcOffset(row - grid.SROW, grid.TH) + grid.PADT;
    tiles[TID].startX = startX;
    tiles[TID].startY = startY;

    var ctx = grid.CTX;

    grid_console(tile.GID + '.TILE Row: ' + row + ' Column: ' + column + ' as Row: ' + (row - grid.SROW) + ' as Column: ' + (column - grid.SCOL) + ' (X: ' + startX + ' Y: ' + startY + ')' +
      ' GTW: ' + grid.TW + ' GTH: ' + grid.TH + ' TW: ' + tile.TW + ' TH: ' + tile.TH +
      ' BCLR: ' + tile.BCLR + ' CLR: ' + tile.CLR);

    if (tile.BCLR.length > 0) {
      ctx.fillStyle = tile.BCLR;
      ctx.fillRect(startX, startY, tile.TW, tile.TH);
    }

    //you have to draw a box to show selected
    if ((tile.SCLR.length > 0) && tile.SLCT) {
      grid_SelectTile (TID);
    } 

    //these vars are used in calculating where to place images
    tiles[TID].blX = 0;
    tiles[TID].lY = 0;
    tiles[TID].bX = 0;
    tiles[TID].bY = 0;
    tiles[TID].tX = 0;
    tiles[TID].tY = 0;
    tiles[TID].tlX = 0;
    tiles[TID].tlY = 0;
    if (tile.IMAGE.length > 0) {
      if (tile.BUTTON.length > 0) {
        var bid = tiles[TID].BUTTON[0];
        var offsetX = Percent2Pixel(buttons[bid].LOC.offsetX, tiles[TID].TW);
        var offsetY = Percent2Pixel(buttons[bid].LOC.offsetY, tiles[TID].TH);
        var lengthX = Percent2Pixel(buttons[bid].SIZE.lengthX, tiles[TID].TW);
        if (lengthX == 0) {
          lengthX = tiles[TID].TW;
        }
        var lengthY = Percent2Pixel(buttons[bid].SIZE.lengthY, tiles[TID].TH);
        if (lengthY == 0) {
          lengthY = tiles[TID].TH;
        }
        var posX = grid_PosX(startX, offsetX, lengthX, buttons[bid].ALIGN.ha);
        var posY = grid_PosY(startY, offsetY, lengthY, buttons[bid].ALIGN.va);
        tiles[TID].blX = lengthX;
        tiles[TID].lY = lengthY;
        tiles[TID].bX = posX;
        tiles[TID].bY = posY;
      }
      if (tile.TEXT.length > 0) {
        var text2 = tiles[TID].TEXT[0];
        ctx.font = text2.FONTSIZE.toString(10) + "px" + " " + text2.FONTNAME;
        var offsetX = Percent2Pixel(text2.LOC.offsetX, tile.TW);
        var offsetY = Percent2Pixel(text2.LOC.offsetY, tiles[TID].TH);
        var text3 = text2.TEXT;
        if (text3.length > 0) {
          var metric = ctx.measureText(text3);
          var lengthX2 = Math.round(metric.width);
          var lengthY2 = text2.FONTSIZE;
          var posX2 = grid_PosX(startX, offsetX, lengthX2, text2.ALIGN.ha);
          var posY2 = grid_PosY(startY, offsetY, lengthY2, text2.ALIGN.va);
          tiles[TID].tX = posX2;
          tiles[TID].tY = posY2;
          tiles[TID].tlX = lengthX2;
          tiles[TID].tlY = lengthY2;
        }
      }
    }

    grid_FormatButton(ctx, startX, startY, TID);
    grid_FormatImage(ctx, startX, startY, TID);
    grid_FormatCircle(ctx, startX, startY, TID);
    grid_FormatLine(ctx, startX, startY, TID);
    grid_FormatPolygon(ctx, startX, startY, TID);
    grid_FormatRectangle(ctx, startX, startY, TID);
    grid_FormatText(ctx, startX, startY, TID);
    //    PreGrid(tile.GINDEX);
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatTile");
  }

  finally { }

} // grid_FormatTile

function grid_ReportLayers(TID) {

  var tile = tiles[TID];
  var layers = "";
  for (var ii = 0; ii < tile.LAYER.length; ii++) {
    layers = layers + ' ' + tile.LAYER[ii];
  }
  return layers;

} //grid_ReportLayers


function grid_ParseButton(tag, tid) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var size = ParseSizeAttr(tag);
    var loc = ParseLocAttr(tag);
    var tprq = XmlGetAttribute(tag, 'tprq');
    var kcod = XmlGetAttribute(tag, 'kcod');
    var url = XmlGetAttribute(tag, 'url');
    var tprq2 = XmlGetAttribute(tag, 'tprq2');
    var kcod2 = XmlGetAttribute(tag, 'kcod2');
    var url2 = XmlGetAttribute(tag, 'url2');
    var layer = ParseLayerAttr(tag);
    var style = ParseLookAttr(tag);
    var button2 = {};
    button2.ID = id;
    button2.ALIGN = ParseAlignAttr(tag);
    button2.BCLR = XmlGetAttribute(tag, 'bclr');
    button2.GID = 0;
    button2.KCOD = kcod;
    button2.KCOD2 = kcod2;
    button2.LAYER = layer;
    button2.LOC = loc;
    button2.lX = 0;
    button2.lY = 0;
    button2.other = 0;
    button2.SIZE = size;
    button2.STYLE = style;
    button2.th = 0;
    button2.TILE = tid;
    button2.TPRQ = tprq;
    button2.TPRQ2 = tprq2;
    button2.tw = 0;
    button2.URL = url;
    button2.URL2 = url2;
    button2.X = 0;
    button2.Y = 0;

    var bid = buttons.length;
    buttons.push(button2);
    return (bid);
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseButton");
  }

  finally { }
} //grid_ParseButton

function grid_PosX(startX, offsetX, lengthX, alignment) {

  try {
    var ha = alignment.substr(0, 1);
    var fudgeX = 0;

    if ((ha == 'C') || (ha == 'c')) {
      fudgeX = -Math.round(lengthX / 2);
    } else if ((ha == 'M') || (ha == 'm')) {  //sometimes ha and va get interchanged
      fudgeX = -Math.round(lengthX / 2);
    } else if ((ha == 'R') || (ha == 'r')) {
      fudgeX = -lengthX;
    } else if ((ha == 'B') || (ha == 'b')) {  //sometimes ha and va get interchanged
      fudgeX = -lengthX;
    } else { //Left
      fudgeX = 0;
    }
    var posX = (startX + offsetX) + fudgeX;
    if (posX < 0) {
      posX = 0;
    }
    return posX;
  }

  catch (ex) {
    console.log("exception thrown in grid_PosX");
  }

  finally { }
}//grid_PosX

function grid_PosY(startY, offsetY, lengthY, alignment) {


  try {
    var va = alignment.substr(0, 1);
    var fudgeY = 0;

    if ((va == 'M') || (va == 'm')) {
      fudgeY = -Math.round(lengthY / 2);
    } else if ((va == 'C') || (va == 'c')) {  //sometimes ha and va get interchanged
      fudgeY = -Math.round(lengthY / 2);
    } else if ((va == 'T') || (va == 't')) {
      fudgeY = 0;
      //fudgeY = -lengthY;
    } else if ((va == 'L') || (va == 'l')) {  //sometimes ha and va get interchanged
      fudgeY = 0;
    } else { //Bottom
      fudgeY = -lengthY;
      //fudgeY = 0;
    }
    var posY = (startY + offsetY) + fudgeY;
    if (posY < 0) {
      posY = 0;
    }
    return posY;
  }
  catch (ex) {
    console.log("exception thrown in grid_PosY");
  }

  finally { }
}//grid_PosY

function grid_PosX2(startX, offsetX, lengthX, alignment) {

  try {
    var ha = alignment.substr(0, 1);
    var fudgeX = 0;

    if ((ha == 'L') || (ha == 'l')) {
      fudgeX = Math.round(lengthX / 2);
    } else if ((ha == 'T') || (ha == 't')) {  //sometimes ha and va get interchanged
      fudgeX = Math.round(lengthX / 2);
    } else if ((ha == 'R') || (ha == 'r')) {
      fudgeX = -Math.round(lengthX / 2);
    } else if ((ha == 'B') || (ha == 'b')) {  //sometimes ha and va get interchanged
      fudgeX = -Math.round(lengthX / 2);
    } else { //Center, Middle
      fudgeX = 0;
    }
    var posX = (startX + offsetX) + fudgeX;
    if (posX < 0) {
      posX = 0;
    }
    return posX;
  }

  catch (ex) {
    console.log("exception thrown in grid_PosX");
  }

  finally { }
}//grid_PosX2

function grid_PosY2(startY, offsetY, lengthY, alignment) {


  try {
    var va = alignment.substr(0, 1);
    var fudgeY = 0;

    if ((va == 'T') || (va == 't')) {
      fudgeY = Math.round(lengthY / 2);
    } else if ((va == 'L') || (va == 'l')) {  //sometimes ha and va get interchanged
      fudgeY = Math.round(lengthY / 2);
    } else if ((va == 'B') || (va == 'b')) {
      fudgeY = -Math.round(lengthY / 2);
    } else if ((va == 'R') || (va == 'r')) {  //sometimes ha and va get interchanged
      fudgeY = -Math.round(lengthY / 2);
    } else { //Middle, Center
      fudgeY = 0;
    }
    var posY = (startY + offsetY) + fudgeY;
    if (posY < 0) {
      posY = 0;
    }
    return posY;
  }
  catch (ex) {
    console.log("exception thrown in grid_PosY");
  }

  finally { }
}//grid_PosY2

function grid_ClearButton2(ctx, bid) {

  try {
    var button = buttons[bid];

    if (InList("2", buttons[bid].STYLE) || InList("3", buttons[bid].STYLE)) {
        ctx.clearRect(button.X, button.Y, button.lX, button.lY);
        grid_console(tile.GID + '.CLEAR BUTTON X:' + button.X + ' Y: ' + button.Y + ' lengthX: ' + button.lX + ' lengthY: ' + button.lY);
    }
  }
  catch (ex) {
    console.log("exception thrown in grid_ClearButton2");
  }

  finally { }
} //grid_ClearButton2

function grid_FormatButton(ctx, startX, startY, TID) {

  try {
    if (tiles[TID].BUTTON != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
      if (tiles[TID].BUTTON.length > 0) {
        var bid = tiles[TID].BUTTON[0];
        var layer = buttons[bid].LAYER;
        if (InList(layer, tiles[TID].LAYER)) {
          grid_FormatButton2(ctx, startX, startY, TID);
        } else {
          grid_console(tiles[TID].GID + '.BUTTON layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }
      }
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatButton");
  }

  finally { }
} //grid_FormatButton

function grid_FormatButton2(ctx, startX, startY, TID) {

  try {

    if (tiles[TID].BUTTON[0] == undefined) return;
    var bid = tiles[TID].BUTTON[0];
    var offsetX = Percent2Pixel(buttons[bid].LOC.offsetX, tiles[TID].TW);
    var offsetY = Percent2Pixel(buttons[bid].LOC.offsetY, tiles[TID].TH);

    var lengthX = Percent2Pixel(buttons[bid].SIZE.lengthX, tiles[TID].TW);
    if (lengthX == 0) {
      lengthX = tiles[TID].TW;
    }

    var lengthY = Percent2Pixel(buttons[bid].SIZE.lengthY, tiles[TID].TH);
    if (lengthY == 0) {
      lengthY = tiles[TID].TH;
    }

    var posX = grid_PosX(startX, offsetX, lengthX, buttons[bid].ALIGN.ha);
    var posY = grid_PosY(startY, offsetY, lengthY, buttons[bid].ALIGN.va);

    var style = '?';
    if (InList("0", buttons[bid].STYLE)) {
      //what does this really mean?
        style = '0';
    }
    if (InList("1", buttons[bid].STYLE)) {
        style = '1';
        ctx.fillStyle = buttons[bid].BCLR;
        ctx.fillRect(posX, posY, lengthX, lengthY);
    }
    if (InList("2", buttons[bid].STYLE)) {
      style = '2';
      ctx.strokeStyle = '#000000';
      ctx.lineWidth = "1";
      ctx.strokeRect(posX, posY, lengthX, lengthY);
    }
    if (InList("3", buttons[bid].STYLE)) {
      style = '3';
      ctx.strokeStyle = InvertColor(ctx.fillStyle);
      //ctx.strokeStyle = '#FFFFFF';
      grid_console(tiles[TID].GID + '.BUTTON BOUNDING BOX: ' + ctx.strokeStyle);
      ctx.lineWidth = "2";
      ctx.strokeRect(posX, posY, lengthX, lengthY);
      ctx.lineWidth = "4";
      ctx.strokeStyle = '#000000';
      ctx.beginPath();
      if ((lengthX > 0) && (lengthY > 0)) {
        ctx.moveTo(posX + lengthX, posY);
        ctx.lineTo(posX + lengthX, posY + lengthY);
        ctx.lineTo(posX, posY + lengthY);
      } else if ((lengthX > 0) && (lengthY < 0)) {
        ctx.moveTo(posX + lengthX, posY + lengthY);
        ctx.lineTo(posX + lengthX, posY);
        ctx.lineTo(posX, posY);
      } else if ((lengthX < 0) && (lengthY > 0)) {
        ctx.moveTo(posX, posY);
        ctx.lineTo(posX, posY + lengthY);
        ctx.lineTo(posX + lengthX, posY + lengthY);
      } else if ((lengthX < 0) && (lengthY < 0)) {
        ctx.moveTo(posX, posY + lengthY);
        ctx.lineTo(posX + lengthX, posY);
        ctx.lineTo(posX + lengthX, posY);
      } else {//should never get here
      }
      ctx.stroke();
    }
    grid_console(tiles[TID].GID + '.BUTTON H:' + buttons[bid].ALIGN.ha + '/V:' + buttons[bid].ALIGN.va +
                 ' X:' + posX + ' Y: ' + posY + ' lengthX: ' + lengthX + ' lengthY: ' + lengthY +
                 ' BCLR: ' + ctx.fillStyle + ' LOOK: ' + style);

    // additional button fields
    // GID - named ID needed for <grid>
    // X, Y - upper left corner
    // lX, lY - width and height of button
    // tw,th - width and height of cell, used to calc tile position based on row and column
    // other - data.fDeviceId

    buttons[bid].GID = tiles[TID].GID;
    buttons[bid].TID = TID; // makes cleanup for reused tile easier
    buttons[bid].X = posX;
    buttons[bid].Y = posY;
    buttons[bid].lX = lengthX;
    buttons[bid].lY = lengthY;
    buttons[bid].tw = tiles[TID].TW;
    buttons[bid].th = tiles[TID].TH;
    buttons[bid].other = tiles[TID].otherID;

    //Check for bleed onto existing buttons. 
    //You would think that buttons would not overlap but ...
      //So make button smaller so that there is no overlap.
    var diff = 0;
    for (var ii = 0; ii < buttons.length; ii++) {
        if (buttons[ii].GID == tiles[TID].GID) {
            if (TID != buttons[ii].TID) {
                if ((posX == buttons[ii].X) || (posY == buttons[ii].Y)) {
                    if ((posY > buttons[ii].Y) && (posY < buttons[ii].Y + buttons[ii].lY)) {
                        buttons[ii].lY = (posY - buttons[ii].Y) - 1;
                        diff = 1;
                    }
                    if ((posX > buttons[ii].X) && (posX < buttons[ii].X + buttons[ii].lX)) {
                        buttons[ii].lX = (posX - buttons[ii].X) - 1;
                        diff = 1;
                    }
                }
            } else if ((posX > buttons[ii].X) && (posX < buttons[ii].X + buttons[ii].lX)) {
                    if ((posY > buttons[ii].Y) && (posY < buttons[ii].Y + buttons[ii].lY)) {
                        buttons[ii].lX = (posX - buttons[ii].X) - 1;
                        buttons[ii].lY = (posY - buttons[ii].Y) - 1;
                    }
            } else {
            }
        }//if
    }//for
  }//try

  catch (ex) {
    console.log("exception thrown in grid_FormatButton2");
  }

  finally { }
} //grid_FormatButton2


function grid_ParseText(tag, fontsize, fontname) {

  try {
    var alignment = ParseAlignAttr(tag);
    var clr = XmlGetAttribute(tag, 'clr');
    var cursor = ParseCursAttr(tag);
    var fontsize2 = ParseFontAttr(tag, fontsize);
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);

    var text = XmlGetText(tag);
    var lt = text.indexOf('&lt;');
    while (lt > -1) {
      text = text.substr(0, lt) + '<' + text.substr(lt + 4);
      lt = text.indexOf('&lt;');
    }
    var gt = text.indexOf('&gt;');
    while (gt > -1) {
      text = text.substr(0, gt) + '>' + text.substr(gt + 4);
      gt = text.indexOf('&gt;');
    }

    return ({ ID: id, ALIGN: alignment, CLR: clr, CURSOR: cursor, FONTNAME: fontname, FONTSIZE: Number(fontsize2), LAYER: layer, LOC: loc, TEXT: text });
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseText");
  }

  finally { }

} // grid_ParseText


function grid_ClearText2(ctx, startX, startY, TID, tid) {

  try {
    var tile = tiles[TID];
    var text2 = tile.TEXT[tid];

    var text3 = text2.TEXT;
    if (text3.length == 0) return;

    //<text> may over ride font 
    ctx.font = text2.FONTSIZE.toString(10) + "px" + " " + text2.FONTNAME;
    var offsetX = Percent2Pixel(text2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(text2.LOC.offsetY, tile.TH);

    var text3A;
    var text3B;
    var newline = text3.indexOf('\\n', 0);
    var newlinecount = 0;

    while (newline > -1) {
      text3A = text3.substr(0, newline);
      text3B = text3.substr(newline + 2);
      grid_ClearText3(ctx, startX, startY, offsetX, offsetY, text2, text3A, newlinecount, tile.GID);
      text3 = text3B;
      newline = text3.indexOf('\\n', 0);
      newlinecount++;
    }
    grid_ClearText3(ctx, startX, startY, offsetX, offsetY, text2, text3, newlinecount, tile.GID);
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearText2");
  }

  finally { }
}//grid_ClearText2

function grid_ClearText3(ctx, startX, startY, offsetX, offsetY, text2, text3, cnt, gID) {

  try {
    if (text3.length == 0) return;

    var metric = ctx.measureText(text3);
    var lengthX = Math.ceil(metric.width); //roundup
    var lengthY = text2.FONTSIZE;

    var fudgeX = 2;
    var fudgeY = cnt * (text2.FONTSIZE + 2);

    var posX = grid_PosX(startX, offsetX, lengthX, text2.ALIGN.ha) - fudgeX;
    var posY = grid_PosY(startY, offsetY, lengthY, text2.ALIGN.va) + fudgeY;

    //ctx.clearRect(posX - 1, posY - 1, lengthX + 2, lengthY + 2);
    ctx.clearRect(posX, posY, lengthX, lengthY);

    grid_console(gID + '.CLEAR TEXT(' + cnt + ') ' + text2.ID + ' H:' + text2.ALIGN.ha + '/V:' + text2.ALIGN.va +
      ' X:' + posX + ' Y: ' + posY + ' lengthX: ' + lengthX + ' lengthY: ' + lengthY + ' LAYER: ' + text2.LAYER + ' ' + text3);

    if ((text2.CURSOR > 0) && (cnt == 0)) {
      var x2A = (lengthX / text3.length) + 1;
      var x1A = (x2A * (text2.CURSOR - 1)) - 1;
      ctx.clearRect(posX + x1A, posY + 1, x2A, 2);
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearText3");
  }

  finally { }
} // grid_ClearText3


function grid_FormatText(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var text2;
    var layer;

    for (var ii = 0; ii < tile.TEXT.length; ii++) {
      text2 = tile.TEXT[ii];
      if (text2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = text2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatText2(ctx, startX, startY, TID, ii);
        } else {
          grid_console(tiles[TID].GID + '.TEXT layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatText");
  }

  finally { }
} // grid_FormatText

function grid_FormatText2(ctx, startX, startY, TID, tid) {

  try {
    var text2 = tiles[TID].TEXT[tid];
    if (text2.TEXT.length == 0) return;

    if (text2.CLR.length > 0) {
      ctx.fillStyle = text2.CLR;
    } else {
      ctx.fillStyle = tiles[TID].CLR;
    }

    //<text> may over ride font 
    ctx.font = text2.FONTSIZE.toString(10) + "px " + text2.FONTNAME;
    var offsetX = Percent2Pixel(text2.LOC.offsetX, tiles[TID].TW);
    var offsetY = Percent2Pixel(text2.LOC.offsetY, tiles[TID].TH);

    var text3 = text2.TEXT;
    var text3A;
    var text3B;
    var newline = text3.indexOf('\\n', 0);
    var newlinecount = 0;
    var gID = tiles[TID].GID;

    //shadow tile fields
    //tiles[TID].tY = posY;

    while (newline > -1) {
      text3A = text3.substr(0, newline);
      text3B = text3.substr(newline + 2);
      grid_FormatText3(ctx, startX, startY, offsetX, offsetY, text2, text3A, newlinecount, gID);
      text3 = text3B;
      newline = text3.indexOf('\\n', 0);
      newlinecount++;
    }
    grid_FormatText3(ctx, startX, startY, offsetX, offsetY, text2, text3, newlinecount, gID);
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatText2");
  }

  finally { }
}// grid_FormatText2

function grid_FormatText3(ctx, startX, startY, offsetX, offsetY, text2, text3, cnt, gID) {

  try {

    if (text3.length == 0) return;

    var metric = ctx.measureText(text3);
    var lengthX = Math.ceil (metric.width); //roundup
    var lengthY = text2.FONTSIZE;
    var fudgeX = 2;
    var fudgeY = (text2.FONTSIZE - 2) + (cnt * (text2.FONTSIZE + 2));
    var posX = grid_PosX(startX, offsetX, lengthX, text2.ALIGN.ha) - fudgeX;
    var posY = grid_PosY(startY, offsetY, lengthY, text2.ALIGN.va) + fudgeY;

    ctx.fillText(text3, posX, posY);
    grid_console(gID + '.TEXT(' + cnt + ') ID: ' + text2.ID + ' H:' + text2.ALIGN.ha + '/V:' + text2.ALIGN.va +
      ' X:' + posX + ' Y: ' + posY + ' FONT: ' + ctx.font + ' CLR: ' + ctx.fillStyle + ' LAYER: ' + text2.LAYER + 
      ' "' + text3 + '"');

    if ((text2.CURSOR > 0) && (cnt == 0)) {
      ctx.strokeStyle = ctx.fillStyle;
      ctx.lineWidth = "2";
      ctx.beginPath();
      var x2A = lengthX / text3.length;
      var x1A = x2A * (text2.CURSOR - 1);
      ctx.moveTo(posX + x1A, posY + 2);
      ctx.lineTo(posX + (x1A + x2A), posY + 2);
      ctx.stroke();
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatText3");
  }

  finally {
    return true;
  }

} // grid_FormatText3


function grid_ParseImage(tag, tid, iid) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);
    var alignment = ParseAlignAttr(tag);
    var name = XmlGetAttribute(tag, 'path').toLowerCase();  //file name could be a path attribute
    if (name.length == 0) {
      name = XmlGetText(tag).toLowerCase();  //or filename could be text
    }
    var size = ParseSizeAttr(tag);

    var image = {};
    image.ID = id;
    image.LAYER = layer;
    image.LOC = loc;
    image.ALIGN = alignment;
    image.BCLR = XmlGetAttribute(tag, 'bclr');
    image.NAME = name;
    image.NAME2 = nopath(name);
    //make erasing image easier
    image.iX = 0;
    image.iY = 0;
    image.ilX = 0;
    image.ilY = 0;
    if (name.length == 0) {
      grid_console("No image");
      image.REF = -1;
    } else {
      image.REF = grid_parseimage2(name, tid, iid);
    }//if
    image.SIZE = size;
    return (image);
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseImage");
  }

  finally { }
} // grid_ParseImage

function grid_parseimage2(name, tid, iid) {

  try {
    //grid_console("Searching graphics[] for " + name);
    var image_ref = -1;

    var name_index;
    for (var jj = 0; jj < graphics.length; jj++) {
      name_index = graphics[jj].object.src.indexOf(name);
      if (name_index > -1) {
        //grid_console(name + " is already queued graphics[" + jj + "]");
        image_ref = jj; //forward
        graphics[jj].TID.push(tid); //back
        graphics[jj].IID.push(iid); //back
        //grid_console(name + " incremented to " + graphics[jj].TID.length);
      }
    }//for
    if (image_ref == -1) {
      var path = name.substr(0, 5);
      if (path == '/frh/') {
      } else {
        path = tiles[tid].PATH;
        name = path + name;
        //name = '/frh/cgtp/' + name;
      }
      var img = document.createElement("img");
      img.src = name;
      var graphic = {};
      graphic.object = img;
      graphic.status = 0;
      graphic.TID = [];
      graphic.TID.push(tid);
      graphic.IID = [];
      graphic.IID.push(iid);
      var offset = graphics.length;
      graphics.push(graphic);
      image_ref = offset;
      //grid_console(name + " queued graphics[" + offset + "]");
      img.onerror = function () { grid_FormatImage3(offset) };
      img.onload = function () { grid_FormatImage4(offset) };
    }//if
    return image_ref;
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseImage2");
  }

  finally { }
}//grid_parseimage2

function grid_ClearImage2(ctx, startX, startY, TID, tid) {

  try {
    var tile = tiles[TID];
    var image2 = tile.IMAGE[tid];

    if (image2.REF == -1) {
      return;
    }

    if (graphics[image2.REF].status == 0) {
      return;
    }


    var lengthX = image2.ilX;
    var lengthY = image2.ilY;
    var posX = image2.iX;
    var posY = image2.iY;

    var img = graphics[image2.REF].object;
    var pic = img.src;
    var offset = pic.lastIndexOf('/');
    if (offset > -1) {
      var pic2 = pic.substr(offset + 1);
    } else {
      pic2 = pic;
    }

    ctx.clearRect(posX, posY, lengthX, lengthY);
    grid_console(tile.GID + '.CLEAR IMAGE X: ' + posX + '  Y: ' + posY + '  sizeX: ' + lengthX + ' sizeY: ' + lengthY + ' "' + pic2 + '"');
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearImage2");
  }

  finally { }
} // grid_ClearImage2


function grid_FormatImage(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var image2;
    var layer;
    var jj;

    for (var ii = 0; ii < tile.IMAGE.length; ii++) {
      image2 = tile.IMAGE[ii];
      if (image2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = image2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatImage2(ctx, startX, startY, TID, ii);
        } else {
          grid_console(tiles[TID].GID + '.IMAGE[' + ii + '] layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage");
  }

  finally { }
} // grid_FormatImage

function grid_FormatImage2(ctx, startX, startY, TID, iid) {

  try {
    var tile = tiles[TID];
    var image2 = tile.IMAGE[iid];
    if (image2.REF == -1) {
      return;
    }

    if (graphics[image2.REF].status == 0) {
      //grid_console('IMAGE Pending ' + graphics[image2.REF].object.src);
      return;
    }//if

    var img = graphics[image2.REF].object;
    var pic = img.src;
    var offset = img.src.lastIndexOf('/');
    if (offset > -1) {
      pic = img.src.substr(offset + 1);
    }

    var offsetX = Percent2Pixel(image2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(image2.LOC.offsetY, tile.TH);

    var lengthX;
    var lengthY;
    var checksize = false;

    if ((image2.SIZE.lengthX == 0) && (image2.SIZE.lengthY == 0)) {
      lengthX = img.width;
      lengthY = img.height;
    } else if ((image2.SIZE.lengthX > -1) && (image2.SIZE.lengthY > -1)) {
      lengthX = Percent2Pixel(image2.SIZE.lengthX, tile.TW);
      lengthY = Percent2Pixel(image2.SIZE.lengthY, tile.TH);
    } else if (tile.blY > 0) {
      lengthY = Math.round(0.40 * tile.blY);
      lengthX = Math.round(img.width * (lengthY / img.height));
    } else {
      lengthX = img.width;
      lengthY = img.height;
      checksize = true;
    }

    var posX = grid_PosX(startX, offsetX, lengthX, image2.ALIGN.ha);
    var posY = grid_PosY(startY, offsetY, lengthY, image2.ALIGN.va);
    if (posY < tile.bY) {
      posY = tile.bY;
    }

    //if a graphic will cover up text, then rescale graphic smaller
    if (checksize) {
      if (tile.tY > 0) {
        var lowerimage = posY + lengthY;
        var imagestart = posX;
        var imageend = posX + lengthX;
        var topedge = tile.tY;
        var leftedge = tile.tX;
        var rightedge = tile.tX + tile.tlX;
        if (lowerimage > topedge) {
          if (((leftedge > imagestart) && (leftedge < imageend)) ||
              ((rightedge > imagestart) && (rightedge < imageend)) ||
              ((leftedge < imagestart) && (imagestart < rightedge)) ||
              ((leftedge < imageend) && (imageend < rightedge))) {
            var lengthY2 = tile.tY - posY;
            var lengthX2 = Math.round((lengthX * lengthY2) / lengthY);
            grid_console(tile.GID + '.IMAGERESIZE ' + pic + ' from ' + lengthX + 'x' + lengthY + ' to ' + lengthX2 + 'x' + lengthY2);
            lengthX = lengthX2;
            lengthY = lengthY2;
          } //posX
        } //posY + lengthY
      } //tile.tY
  } //checksize

    // make erase easier
    tiles[TID].IMAGE[iid].iX = posX;
    tiles[TID].IMAGE[iid].iY = posY;
    tiles[TID].IMAGE[iid].ilX = lengthX;
    tiles[TID].IMAGE[iid].ilY = lengthY;

    if (image2.BCLR.length > 0) {
      ctx.fillStyle = image2.BCLR;
      ctx.fillRect(posX, posY, lengthX, lengthY);
      grid_console(tile.GID + '.IMAGE (fill) X: ' + posX + '  Y: ' + posY + '  sizeX: ' + lengthX + ' sizeY: ' + lengthY + ' BCLR: ' + ctx.fillStyle);
    }

    ctx.drawImage(img, posX, posY, lengthX, lengthY);
    grid_console(tile.GID + '.IMAGE  H:' + image2.ALIGN.ha + '/V:' + image2.ALIGN.va +
      ' X: ' + image2.LOC.offsetX + '% (' + posX + 'px) Y: ' + image2.LOC.offsetY + '% (' + posY + 'px) SizeX: ' + lengthX + ' SizeY: ' + lengthY + ' "' +
      pic + '"');
    return;
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage2");
  }

  finally { }

} // grid_FormatImage2


function grid_FormatImage3(offset) {

  try {
    var tid = graphics[offset].TID[0];
    var iid = graphics[offset].IID[0];
    var tile = tiles[tid];
    var image = tile.IMAGE[iid];
    console.log('IMAGE load failed for ' + image.NAME);
    console.log('  ' + image.NAME2 + ' not found in ' + graphics[offset].object.src);

    var gid = tile.GINDEX;
    var grid = grids[gid];
    var url = grid.URL;

    graphics[offset].object.src = url2path(url) + image.NAME2;
    console.log('  ' + image.NAME2 + ' trying ' + graphics[offset].object.src);
    graphics[offset].object.onerror = function () { grid_FormatImage3A(offset) };
    graphics[offset].object.onload = function () { grid_FormatImage4(offset) };

  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage3");
  }

  finally { }

} // grid_FormatImage3

function grid_FormatImage3A(offset) {

  try {
    var tid = graphics[offset].TID[0];
    var iid = graphics[offset].IID[0];
    var tile = tiles[tid];
    var image = tile.IMAGE[iid];
    console.log('IMAGE load failed for ' + image.NAME);
    console.log('  ' + image.NAME2 + ' not found in ' + graphics[offset].object.src);
    graphics[offset].object.src = '/frh/cgtp/' + image.NAME2;
    console.log('  ' + image.NAME2 + ' trying ' + graphics[offset].object.src);
    graphics[offset].object.onerror = function () { grid_FormatImage3B(offset) };
    graphics[offset].object.onload = function () { grid_FormatImage4(offset) };
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage3A");
  }

  finally { }

} // grid_FormatImage3A


function grid_FormatImage3B(offset) {

  try {
    var tid = graphics[offset].TID[0];
    var iid = graphics[offset].IID[0];
    var tile = tiles[tid];
    var image = tile.IMAGE[iid];
    console.log('IMAGE load failed for ' + image.NAME);
    console.log('  ' + image.NAME2 + ' not found in ' + graphics[offset].object.src);

    graphics[offset].object.src = '/frh/gui/' + image.NAME2;
    console.log('  ' + image.NAME2 + ' trying ' + graphics[offset].object.src);
    graphics[offset].object.onerror = function () { grid_FormatImage3C(offset) };
    graphics[offset].object.onload = function () { grid_FormatImage4(offset) };
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage3B");
  }

  finally { }

} // grid_FormatImage3B


function grid_FormatImage3C(offset) {

  try {
    
    var tid = graphics[offset].TID[0];
    var iid = graphics[offset].IID[0];
    var tile = tiles[tid];
    var image = tile.IMAGE[iid];
    var gid = tile.GINDEX;
    var grid = grids[gid];
    var url = grid.URL;

    grid_console('IMAGE load failed for ' + image.NAME2);

    var path = image.NAME2.substr(0, 5);
    if (path == '/frh/') {
      console.log('  ' + image.NAME2 + ' not found in ' + image.NAME);
    } else {
      console.log('  ' + image.NAME2 + ' not found in ' + tile.PATH);
    }
    console.log('  ' + image.NAME2 + ' not found in ' + tile.PATH);
    console.log('  ' + image.NAME2 + ' not found in ' + url2path(url));
    console.log('  ' + image.NAME2 + ' not found in /frh/cgtp/');
    console.log('  ' + image.NAME2 + ' not found in /frh/gui/');
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage3C");
  }

  finally { }

} // grid_FormatImage3C


function grid_FormatImage4(offset) {

  try {
    graphics[offset].status = 1;
    grid_console('IMAGE Loaded ' + graphics[offset].object.src);

    var pic = graphics[offset].object.src;
    var tid;
    var iid;
    var tile;
    var image;
    var layer;

    while (graphics[offset].TID.length > 0) {
      tid = graphics[offset].TID[0];
      graphics[offset].TID.splice(0, 1);
      iid = graphics[offset].IID[0];
      graphics[offset].IID.splice(0, 1);
      tile = tiles[tid];
      image = tile.IMAGE[iid];
      layer = image.LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_FormatTile(tid);
      }
    }//while
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatImage4");
  }

  finally { }
} // grid_FormatImage4


function grid_ParseCircle(tag) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);
    var dia = XmlGetAttribute(tag, 'diameter');
    var dia2 = Number(dia);
    var radius = dia2 / 2;
    var weight = ParseWtAttr(tag);
    var clr = XmlGetAttribute(tag, 'clr');
    var fill = XmlGetAttribute(tag, 'fill');
    var alignment = ParseAlignAttr(tag);
    return ({ ID: id, LAYER: layer, LOC: loc, CLR: clr, RAD: radius, WT: weight, FILL: fill, ALIGN: alignment });
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseCircle");
  }

  finally { }
} // grid_ParseCircle


function grid_ClearCircle2(ctx, startX, startY, TID, cid) {

  try {
    var tile = tiles[TID];
    var circle2 = tile.CIRCLE[cid];

    var offsetX = Percent2Pixel(circle2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(circle2.LOC.offsetY, tile.TH);
    var lengthX = 2 * circle2.RAD;
    var lengthY = 2 * circle2.RAD;
    var posX = grid_PosX2(startX, offsetX, lengthX, circle2.ALIGN.ha) - circle2.RAD;
    var posY = grid_PosY2(startY, offsetY, lengthY, circle2.ALIGN.va) - circle2.RAD;
    var weight = Math.round(circle2.WT);
    ctx.clearRect(posX - (weight/2), posY - (weight/2), lengthX + weight, lengthY + weight);

    grid_console(tile.GID + '.CLEAR CIRCLE  X: ' + posX + 'Y: ' + posY + ' RADIUS: ' + circle2.RAD);
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearCircle2");
  }

  finally { }

} // grid_ClearCircle2


function grid_FormatCircle(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var circle2;
    var layer;

    for (var cc = 0; cc < tile.CIRCLE.length; cc++) {
      circle2 = tile.CIRCLE[cc];
      if (circle2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = circle2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatCircle2(ctx, startX, startY, TID, cc);
        } else {
          grid_console(tiles[TID].GID + '.CIRCLE layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatCircle");
  }

  finally { }

} // grid_FormatCircle


function grid_FormatCircle2(ctx, startX, startY, TID, cid) {

  try {
    var tile = tiles[TID];
    circle2 = tile.CIRCLE[cid];
    var offsetX = Percent2Pixel(circle2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(circle2.LOC.offsetY, tile.TH);
    
    var radiusX = Percent2Pixel(circle2.RAD, tile.TW);
    var radiusY = Percent2Pixel(circle2.RAD, tile.TH);
    circle2.RAD = Math.min(radiusX, radiusY); 
    
    var lengthX = 2 * circle2.RAD;
    var lengthY = 2 * circle2.RAD;	
    var posX = grid_PosX2(startX, offsetX, lengthX, circle2.ALIGN.ha);
    var posY = grid_PosY2(startY, offsetY, lengthY, circle2.ALIGN.va);

    ctx.lineWidth = circle2.WT;
    if (circle2.CLR.length > 0) {
      ctx.strokeStyle = circle2.CLR;
    } else {
      ctx.strokeStyle = tile.CLR;
    }
    ctx.beginPath();
    ctx.arc(posX, posY, circle2.RAD, 0, 2 * Math.PI);
    ctx.stroke();
    if (circle2.FILL.length > 0) {
      ctx.fillStyle = circle2.FILL;
      ctx.fill();
    }
    ctx.closePath();
    grid_console(tile.GID + '.CIRCLE  H:' + circle2.ALIGN.ha + '/V:' + circle2.ALIGN.va + ' X: ' + posX + 'Y: ' + posY + ' RADIUS: ' + circle2.RAD);
  }

  catch (ex) {
    console.log("exception thrown in grid_grid_FormatCircle2");
  }

  finally { }

} // grid_FormatCircle2


function grid_ParseLine(tag) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);
    var eloc = ParseElocAttr(tag);
    var clr = XmlGetAttribute(tag, 'clr');
    var weight = '2';
    var alignment = ParseAlignAttr(tag);
    return ({ ID: id, LAYER: layer, LOC: loc, ELOC: eloc, CLR: clr, WT: weight, ALIGN: alignment });
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseLine");
  }

  finally { }

} // grid_ParseLine

function grid_ClearLine2(ctx, startX, startY, TID, lid) {

  try {
    var tile = tiles[TID];
    var line2 = tile.LINE[lid];

    var offsetX = Percent2Pixel(line2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(line2.LOC.offsetY, tile.TH);
    var endX = Percent2Pixel(line2.ELOC.endX, tile.TW);
    var endY = Percent2Pixel(line2.ELOC.endY, tile.TH);
    var width = endX - offsetX;
    var height = endY - offsetY;
    var weight = Math.round(line2.WT);
    ctx.clearRect(startX + offsetX - (weight/2), startY + offsetY - (weight/2), width + weight, height + weight);

    grid_console(tile.GID + '.CLEAR LINE X1: ' + (startX + offsetX) + ' Y1: ' + (startY + offsetY) + ' X2: ' + (startX + endX) + ' Y2: ' + (startY + endY));
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearLine2");
  }

  finally { }
} // grid_ClearLine2

function grid_FormatLine(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var line2;
    var layer;

    for (var ll = 0; ll < tile.LINE.length; ll++) {
      line2 = tile.LINE[ll];
      if (line2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = line2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatLine2(ctx, startX, startY, TID, ll);
        } else {
          grid_console(tiles[TID].GID + '.LINE layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatLine");
  }

  finally { }

} // grid_FormatLine

function grid_FormatLine2(ctx, startX, startY, TID, lid) {

  try {
    var tile = tiles[TID];
    var line2 = tile.LINE[lid];

    var offsetX = Percent2Pixel(line2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(line2.LOC.offsetY, tile.TH);
    var endX = Percent2Pixel(line2.ELOC.endX, tile.TW);
    var endY = Percent2Pixel(line2.ELOC.endY, tile.TH);

    if (line2.CLR.length > 0) {
      ctx.strokeStyle = line2.CLR;
    } else {
      ctx.strokeStyle = tile.CLR;
    }//if

    ctx.lineWidth = line2.WT;
    ctx.beginPath();
    ctx.moveTo(startX + offsetX, startY + offsetY);
    ctx.lineTo(startX + endX, startY + endY);
    ctx.stroke();

    grid_console(tile.GID + '.LINE X1: ' + (startX + offsetX) + 'Y1: ' + (startY + offsetY) + ' X2: ' + (startX + endX) + ' Y2: ' + (startY + endY));
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatLine2");
  }

  finally { }

} // grid_FormatLine2


function grid_ParsePolygon(tag) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);
    var alignment = ParseAlignAttr(tag);
    var vrtx1 = ParseVrtxAttr(tag, '1');
    var vrtx2 = ParseVrtxAttr(tag, '2');
    var vrtx3 = ParseVrtxAttr(tag, '3');
    var vrtx4 = ParseVrtxAttr(tag, '4');
    var clr = XmlGetAttribute(tag, 'clr');
    var weight = '2';
    var fill = XmlGetAttribute(tag, 'fill');
    return ({ ID: id, LAYER: layer, LOC: loc, ALIGN: alignment, V1: vrtx1, V2: vrtx2, V3: vrtx3, V4: vrtx4, CLR: clr, WT: weight, FILL: fill });
  }

  catch (ex) {
    console.log("exception thrown in grid_ParsePolygon");
  }

  finally { }
} // grid_ParsePolygon


function grid_ClearPolygon2(ctx, startX, startY, TID, pid) {

  try {
    var tile = tiles[TID];
    var polygon2 = tile.POLYGON[pid];

    var v1x = Percent2Pixel(polygon2.V1.offsetX, tile.TW);
    var minX = v1x;
    var maxX = v1x;

    var v1y = Percent2Pixel(polygon2.V1.offsetY, tile.TH);
    var minY = v1y;
    var maxY = v1y;

    var v2x = Percent2Pixel(polygon2.V2.offsetX, tile.TW);
    if (v2x < minX) { minX = v2x; }
    if (v2x > maxX) { maxX = v2x; }

    var v2y = Percent2Pixel(polygon2.V2.offsetY, tile.TH);
    if (v2y < minY) { minY = v2y; }
    if (v2y > maxY) { maxY = v2y; }

    var v3x = Percent2Pixel(polygon2.V3.offsetX, tile.TW);
    if (v3x < minX) { minX = v3x; }
    if (v3x > maxX) { maxX = v3x; }

    var v3y = Percent2Pixel(polygon2.V3.offsetY, tile.TH);
    if (v3y < minY) { minY = v3y; }
    if (v3y > maxY) { maxY = v3y; }

    if (polygon2.V4.valid > 0) {
      var v4x = Percent2Pixel(polygon2.V4.offsetX, tile.TW);
      if (v4x < minX) { minX = v4x; }
      if (v4x > maxX) { maxX = v4x; }
      var v4y = Percent2Pixel(polygon2.V4.offsetY, tile.TH);
      if (v4y < minY) { minY = v4y; }
      if (v4y > maxY) { maxY = v4y; }
    }

    var posX2 = (startX - minX) - 1;
    var posY2 = (startY - minY) - 1;
    var lengthX2 = (maxX - minX) + 2;
    var lengthY2 = (maxY - minY) + 2;
    //ctx.strokeStyle = '#000000';
    //ctx.strokeRect(posX2, posY2, lengthX2, lengthY2);
    ctx.clearRect(posX2, posY2, lengthX2, lengthY2);

    grid_console(tile.GID + '.CLEAR POLYGON using box X: ' + posX2 + ' Y: ' + posY2 + ' LX: ' + lengthX2 + ' LY: ' + lengthY2);
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearPolygon2");
  }

  finally { }
} // grid_ClearPolygon2

function grid_FormatPolygon(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var polygon2;
    var layer;

    for (var pp = 0; pp < tile.POLYGON.length; pp++) {
      polygon2 = tile.POLYGON[pp];
      if (polygon2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = polygon2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatPolygon2(ctx, startX, startY, TID, pp);
        } else {
          grid_console(tiles[TID].GID + '.POLYGON layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatPolygon");
  }

  finally { }

} // grid_FormatPolygon


function grid_FormatPolygon2(ctx, startX, startY, TID, pid) {

  try {
    var tile = tiles[TID];
    var polygon2 = tile.POLYGON[pid];

    if (polygon2.CLR.length > 0) {
      ctx.strokeStyle = polygon2.CLR;
    } else {
      ctx.strokeStyle = tile.CLR;
    }

    if (polygon2.FILL.length > 0) {
      ctx.fillStyle = polygon2.FILL;
    }

    ctx.lineWidth = polygon2.WT;

    var v1x = Percent2Pixel(polygon2.V1.offsetX, tile.TW);
    var v1y = Percent2Pixel(polygon2.V1.offsetY, tile.TH);
    var v2x = Percent2Pixel(polygon2.V2.offsetX, tile.TW);
    var v2y = Percent2Pixel(polygon2.V2.offsetY, tile.TH);
    var v3x = Percent2Pixel(polygon2.V3.offsetX, tile.TW);
    var v3y = Percent2Pixel(polygon2.V3.offsetY, tile.TH);
    if (polygon2.V4.valid > 0) {
      var v4x = Percent2Pixel(polygon2.V4.offsetX, tile.TW);
      var v4y = Percent2Pixel(polygon2.V4.offsetY, tile.TH);
    }

    ctx.beginPath();
    ctx.moveTo(startX + v1x, startY + v1y);
    grid_console(tile.GID + '.POLYGON P1 X: ' + (startX + v1x) + ' Y: ' + (startY + v1y) + 'CLR : ' + ctx.strokeStyle);
    ctx.lineTo(startX + v2x, startY + v2y);
    grid_console(tile.GID + '.POLYGON P2 X: ' + (startX + v2x) + ' Y: ' + (startY + v2y) + 'CLR : ' + ctx.strokeStyle);
    ctx.lineTo(startX + v3x, startY + v3y);
    grid_console(tile.GID + '.POLYGON P3 X: ' + (startX + v3x) + ' Y: ' + (startY + v3y) + 'CLR : ' + ctx.strokeStyle);

    if (polygon2.V4.valid > 0) {
      ctx.lineTo(startX + v4x, startY + v4y);
      grid_console(tile.GID + '.POLYGON P4 X: ' + (startX + v4x) + ' Y: ' + (startY + v4y) + 'CLR : ' + ctx.strokeStyle);
    }

    ctx.closePath();
    ctx.stroke();
    if (polygon2.FILL.length > 0) {
      ctx.fill();
      grid_console(tile.GID + '.POLYGON fill: ' + polygon2.FILL);
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatPolygon2");
  }

  finally { }

} // grid_FormatPolygon2

function grid_ParseRectangle(tag) {

  try {
    var id = XmlGetAttribute(tag, 'id');
    var layer = ParseLayerAttr(tag);
    var loc = ParseLocAttr(tag);
    var size = ParseSizeAttr(tag);
    var weight = ParseWtAttr(tag);
    var clr = XmlGetAttribute(tag, 'clr');
    var fill = XmlGetAttribute(tag, 'fill');
    var focus = 0;
    if ((fill == "FFFFFF") || (fill == "#FFFFFF")) {
      focus = 0;
      //grid_console('Rect "' + id + '" has NO focus');
    } else if (fill.length > 0) {
      focus = 1;
      //grid_console('Rect "' + id + '" has focus');
    } else {
      focus = 0;
      //grid_console('Rect "' + id + '" has NO focus');
    }
    var alignment = ParseAlignAttr(tag);
    return ({ ID: id, LAYER: layer, SIZE: size, LOC: loc, CLR: clr, WT: weight, FILL: fill, ALIGN: alignment, FOCUS: focus });
  }

  catch (ex) {
    console.log("exception thrown in grid_ParseRectangle");
  }

  finally { }

} // grid_ParseRectangle


function grid_ClearRectangle2(ctx, startX, startY, TID, rid) {

  try {
    var tile = tiles[TID];
    var rectangle2 = tile.RECTANGLE[rid];

    var offsetX = Percent2Pixel(rectangle2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(rectangle2.LOC.offsetY, tile.TH);

    var lengthX = Percent2Pixel(rectangle2.SIZE.lengthX, tile.TW);
    var lengthY = Percent2Pixel(rectangle2.SIZE.lengthY, tile.TH);

    var posX = grid_PosX(startX, offsetX, lengthX, rectangle2.ALIGN.ha);
    var posY = grid_PosY(startY, offsetY, lengthY, rectangle2.ALIGN.va);

    //need to erase an area slightly bigger than the rectangle
    posX -= 2;
    lengthX += 4;
    posY -= 2;
    lengthY += 4;

    ctx.clearRect(posX, posY, lengthX, lengthY);
    grid_console(tile.GID + '.CLEAR RECT H:' + rectangle2.ALIGN.ha + '/V:' + rectangle2.ALIGN.va + ' X: ' + posX + ' Y: ' + posY + ' LX: ' + lengthX + ' LY: ' + lengthY);
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearRectangle2");
  }

  finally { }

} // grid_ClearRectangle2


function grid_FormatRectangle(ctx, startX, startY, TID) {

  try {
    var tile = tiles[TID];
    var rectangle2;
    var layer;

    for (var rr = 0; rr < tile.RECTANGLE.length; rr++) {
      rectangle2 = tile.RECTANGLE[rr];
      if (rectangle2 != undefined) {//this should not be needed, but I have seen tearing in dynamic allocated structures
        layer = rectangle2.LAYER;
        if (InList(layer, tile.LAYER)) {
          grid_FormatRectangle2(ctx, startX, startY, TID, rr);
        } else {
          grid_console(tiles[TID].GID + '.RECTANGLE layer ' + layer + ' not in layers' + grid_ReportLayers(TID));
        }//if
      }
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatRectangle");
  }

  finally { }

} // grid_FormatRectangle

function grid_FormatRectangle2(ctx, startX, startY, TID, rid) {

  try {
    var tile = tiles[TID];
    var rectangle2 = tile.RECTANGLE[rid];

    var offsetX = Percent2Pixel(rectangle2.LOC.offsetX, tile.TW);
    var offsetY = Percent2Pixel(rectangle2.LOC.offsetY, tile.TH);

    var lengthX = Percent2Pixel(rectangle2.SIZE.lengthX, tile.TW);
    var lengthY = Percent2Pixel(rectangle2.SIZE.lengthY, tile.TH);

    var posX = grid_PosX(startX, offsetX, lengthX, rectangle2.ALIGN.ha);
    var posY = grid_PosY(startY, offsetY, lengthY, rectangle2.ALIGN.va);

    //rectangles the same as the tile need to be slightly smaller
    //other wise tiles to the right and below remove part of the rectangle
    if ((startX == posX) && (startY == posY)) {
      if ((lengthX == tile.TW) && (lengthY == tile.TH)) {
        posX++;
        lengthX -= 2;
        posY++;
        lengthY -= 2;
      }
    }

    if (rectangle2.FILL.length > 0) {
      ctx.fillStyle = rectangle2.FILL;
      ctx.fillRect(posX, posY, lengthX, lengthY);
      grid_console(tile.GID + '.RECT H:' + rectangle2.ALIGN.ha + '/V:' + rectangle2.ALIGN.va + ' X: ' + posX + ' Y: ' + posY + ' LX: ' + lengthX + ' LY: ' + lengthY + ' CLR: ' + ctx.strokeStyle + ' FILL: ' + rectangle2.FILL + ' FOCUS: ' + rectangle2.FOCUS);
    } else {
      grid_console(tile.GID + '.RECT H:' + rectangle2.ALIGN.ha + '/V:' + rectangle2.ALIGN.va + ' X: ' + posX + ' Y: ' + posY + ' LX: ' + lengthX + ' LY: ' + lengthY + ' CLR: ' + ctx.strokeStyle + ' WT: ' + rectangle2.WT + ' FOCUS: ' + rectangle2.FOCUS);
    }

    ctx.lineWidth = rectangle2.WT;
    if (rectangle2.CLR.length > 0) {
      ctx.strokeStyle = rectangle2.CLR;
    } else {
      ctx.strokeStyle = tile.CLR;
    }
    ctx.strokeRect(posX, posY, lengthX, lengthY);
  }

  catch (ex) {
    console.log("exception thrown in grid_FormatRectangle2");
  }

  finally { }

} // grid_FormatRectangle2


function grid_DeleteTiles(data, tag) {

  try {
    var pos = ParsePosAttr(tag);
    var start_row = pos.row;
    var start_col = pos.col;
    var dims = ParseDimsAttr(tag);
    var delta_row = dims.row;
    var delta_col = dims.col;
    var tile;
    var row;
    var col;

    for (var ii = 0; ii < tiles.length;) {
      if (tiles[ii].GID == data.Id) {
        tile = tiles[ii];
        row = tile.ROW;
        col = tile.COLUMN;
        if ((row >= start_row) && (row < (start_row + delta_row))) {
          if ((col >= start_col) && (col < (start_col + delta_col))) {
            grid_ClearTile2(ii);
            grid_DeleteTile2(ii);
            tiles.splice(ii, 1);
          } else {
            ii++;
          }//if
        } else {
          ii++;
        }//if
      } else {
        ii++;
      }
    }//for
  }//try

  catch (ex) {
    console.log("exception thrown in grid_DeleteTiles");
  }

  finally { }

} // grid_DeleteTiles


function grid_ClearTile2(tid) {

  try {
    var tile = tiles[tid];
    var row = tile.ROW;
    var column = tile.COLUMN;

    if ((row < grids[tile.GINDEX].SROW) || (row > (grids[tile.GINDEX].SROW + grids[tile.GINDEX].DROW))) {
      if ((column < grids[tile.GINDEX].SCOL) || (column > (grids[tile.GINDEX].SCOL + grids[tile.GINDEX].DCOL))) {
        grid_console(tile.GID + '.CLEARTILE Skip Row: ' + row + ' Column: ' + column);
        return;
      }
    }
    grid_console(tile.GID + ".CLEARTILE row: " + row + " column: " + column);

    var startX = CalcOffset(column - grids[tile.GINDEX].SCOL, grids[tile.GINDEX].TW) + grids[tile.GINDEX].PADL;
    var startY = CalcOffset(row - grids[tile.GINDEX].SROW, grids[tile.GINDEX].TH) + grids[tile.GINDEX].PADT;

    var ctx = grids[tile.GINDEX].CTX;
    var layer;

    if (tile.BUTTON.length > 0) {
      var bid = tile.BUTTON[0];
      layer = buttons[bid].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearButton2(ctx, bid);
      }
    }

    var tt;
    for (tt = 0; tt < tile.IMAGE.length; tt++) {
      layer = tile.IMAGE[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearImage2(ctx, startX, startY, tid, tt);
      }
    }

    for (tt = 0; tt < tile.CIRCLE.length; tt++) {
      layer = tile.CIRCLE[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearCircle2(ctx, startX, startY, tid, tt);
      }
    }

    for (tt = 0; tt < tile.LINE.length; tt++) {
      layer = tile.LINE[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearLine2(ctx, startX, startY, tid, tt);
      }
    }

    for (tt = 0; tt < tile.POLYGON.length; tt++) {
      layer = tile.POLYGON[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearPolygon2(ctx, startX, startY, tid, tt);
      }
    }

    for (tt = 0; tt < tile.RECTANGLE.length; tt++) {
      layer = tile.RECTANGLE[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearRectangle2(ctx, startX, startY, tid, tt);
      }
    }

    for (tt = 0; tt < tile.TEXT.length; tt++) {
      layer = tile.TEXT[tt].LAYER;
      if (InList(layer, tile.LAYER)) {
        grid_ClearText2(ctx, startX, startY, tid, tt);
      }
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_ClearTile2");
  }

  finally { }

} //grid_ClearTile2


function grid_DeleteTile2(tid) {

  try {
    var row = tiles[tid].ROW;
    var column = tiles[tid].COLUMN;

    grid_console(tiles[tid].GID + ".grid_DeleteTile2 row: " + row + " column: " + column);

    //flush buttons left behind by old tile.
    for (var ii = 0; ii < buttons.length;) {
      if ((buttons[ii].GID == tiles[tid].GID) && (buttons[ii].TID == tid)) {
        buttons.splice(ii, 1);
      } else {
        ii++;
      }
    }

    if (tiles[tid].BUTTON.length > 0) {
      tiles[tid].BUTTON.pop();
    }

    while (tiles[tid].IMAGE.length > 0) {
      tiles[tid].IMAGE.pop();
    }

    while (tiles[tid].CIRCLE.length > 0) {
      tiles[tid].CIRCLE.pop();
    }

    while (tiles[tid].LINE.length > 0) {
      tiles[tid].LINE.pop();
    }

    while (tiles[tid].POLYGON.length > 0) {
      tiles[tid].POLYGON.pop();
    }

    while (tiles[tid].RECTANGLE.length > 0) {
      tiles[tid].RECTANGLE.pop();
    }

    while (tiles[tid].TEXT.length > 0) {
      tiles[tid].TEXT.pop();
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_DeleteTile2");
  }

  finally { }
} //grid_DeleteTile2


function grid_ModifyTile(data, tag) {

  try {
    var id = XmlGetAttribute(tag, 'id');

    if (id.length > 0) {
      for (var ii = 0; ii < tiles.length; ii++) {
        if ((tiles[ii].GID == data.Id) && (tiles[ii].ID.length > 0)) {
          if (id == tiles[ii].ID) {
            grid_ModifyTile2(ii, tag);
            return;
          }//if
        }//if 
      }//for
    } else {
      var position = ParsePosAttr(tag);
      var row = position.row;
      var column = position.col;
      var trow;
      var tcol;
      for (var ii = 0; ii < tiles.length; ii++) {
        if (tiles[ii].GID == data.Id) {
          trow = tiles[ii].ROW;
          tcol = tiles[ii].COLUMN;
          if ((row == trow) && (column == tcol)) {
            grid_ModifyTile2(ii, tag);
            return;
          }//if
        }//if 
      }//for

    }//if
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyTile");
  }

  finally { }
} //grid_ModifyTile


function grid_ModifyTile2(tid, tag) {

  try {
    var row = tiles[tid].ROW;
    var column = tiles[tid].COLUMN;

    grid_console(tiles[tid].GID + ".MODTILE row: " + row + " column: " + column);

    var differences = 0;

    var newcolor = "";
    var bclr = XmlGetAttribute(tag, 'bclr');
    if (bclr.length > 0) {
      newcolor = GridFixColor(bclr);
      if (newcolor != tiles[tid].BCLR) {
        tiles[tid].BCLR = newcolor;
        differences++;
      }
    }

    var clr = XmlGetAttribute(tag, 'fclr');
    if (clr.length > 0) {
      newcolor = GridFixColor(clr);
      if (newcolor != tiles[tid].CLR) {
          tiles[tid].CLR = newcolor;
          differences++;
      }
    }

    var sclr = XmlGetAttribute(tag, 'sclr');
    if (sclr.length > 0) {
      newcolor = GridFixColor(sclr);
      if (newcolor != tiles[tid].SCLR) {
          tiles[tid].SCLR = newcolor;
          differences++;
      }
    }

    var slct = XmlGetAttribute(tag, 'slct');
    if (slct.length > 0) {
      var newattr = ParseSlctAttr(tag);
      if (newattr != tiles[tid].SLCT) {
          tiles[tid].SLCT = newattr;
          grid_SelectTile (tid);
          //differences++;
      }
    }

    var layerOn = ParseLayerOnAttr(tag);
    var value;

    //add layers
    var ii;
    if (layerOn.length > 0) {
      for (ii = 1; ii < tiles[tid].LAYER.length;) {
        value = tiles[tid].LAYER[ii];
        tiles[tid].LAYER.splice(ii, 1);
        differences++;
      }
      for (var ii = 0; ii < layerOn.length; ii++) {
        value = layerOn[ii];
        if (!InList(value, tiles[tid].LAYER)) {
          tiles[tid].LAYER.push(value);
          differences++;
        }//if
      }//for
    }//if

    //remove layers
    var layerOff = ParseLayerOffAttr(tag);
    if (layerOff.length > 0) {
      if (layerOff[0] == '*') {
        for (ii = 1; ii < tiles[tid].LAYER.length;) {
          value = tiles[tid].LAYER[ii];
          tiles[tid].LAYER.splice(ii, 1);
          differences++;
        }
      } else {
        var where;
        var where2;
        for (ii = 0; ii < layerOff.length;) {
          value = layerOff[ii];
          where = InList2(value, tiles[tid].LAYER);
          if (where > -1) {
            where2 = tiles[tid].LAYER[where];
            tiles[tid].LAYER.splice(where, 1);
            differences++;
          } else {
            ii++;
          }
        }//for
      }
    }

    if (differences > 0) {
      grid_console(tiles[tid].GID + '.ModifyTile NO Changes');
    } else {
      grid_FormatTile(tid);
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyTile2");
  }

  finally { }

} //grid_ModifyTile2


function grid_ModifyButton(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");
    var bid;

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        if (tiles[cell].BUTTON.length > 0) {
          bid = tiles[cell].BUTTON[0];
          if (id == buttons[bid].ID) {
            grid_ModifyButton2(cell, tag);
            return;
          }//if
        }//for if if
      }//for if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyButton");
  }

  finally { }
} //grid_ModifyButton


function grid_ModifyButton2(cell, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;
    var ctx = grids[tiles[cell].GINDEX].CTX;

    var bid = tiles[cell].BUTTON[0];
    var differences = 0;

    var size = XmlGetAttribute(tag, 'size');
    if (size.length > 0) {
      var newsize = ParseSizeAttr(tag);
      if (newsize != buttons[bid].SIZE) {
        buttons[bid].SIZE = newsize;
        differences++;
      }
    }

    var tprq = XmlGetAttribute(tag, 'tprq');
    if (tprq.length > 0) {
      if (tprq != buttons[bid].TPRQ) {
        buttons[bid].TPRQ = tprq;
        differences++;
      }
    }

    var tprq2 = XmlGetAttribute(tag, 'tprq2');
    if (tprq2.length > 0) {
      if (tprq2 != buttons[bid].TPRQ2) {
        buttons[bid].TPRQ2 = tprq2;
        differences++;
      }
    }

    var kcod = XmlGetAttribute(tag, 'kcod');
    if (kcod.length > 0) {
      if (kcod != buttons[bid].KCOD) {
        buttons[bid].KCOD = kcod;
        differences++;
      }
    }
    var kcod2 = XmlGetAttribute(tag, 'kcod2');
    if (kcod2.length > 0) {
      if (kcod2 != buttons[bid].KCOD2) {
        buttons[bid].KCOD2 = kcod2;
        differences++;
      }
    }

    var url = XmlGetAttribute(tag, 'url');
    if (url.length > 0) {
      if (url != buttons[bid].URL) {
        buttons[bid].URL = url;
        differences++;
      }
    }

    var url2 = XmlGetAttribute(tag, 'url2');
    if (url2.length > 0) {
      if (url2 != buttons[bid].URL2) {
        buttons[bid].URL2 = url2;
        differences++;
      }
    }

    var bclr = XmlGetAttribute(tag, 'bclr');
    if (bclr.length > 0) {
      if (bclr != buttons[bid].BCLR) {
        buttons[bid].BCLR = bclr;
        differences++;
      }
    }
    
    if (differences == 0) {
      grid_console(tiles[cell].GID + '.ModifyButton NO Changes');
    } else{
      grid_FormatTile(cell);
    }

  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyButton2");
  }

  finally { }
} //grid_ModifyButton


function grid_ModifyText(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].TEXT.length; tt++) {
          if (id == tiles[cell].TEXT[tt].ID) {
              grid_ModifyText2(cell, tt, data.FontSize, tag);
              return;
          }//for if for if
        }//for if for
      }//for if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyText");
  }

  finally { }
} //grid_ModifyText

function grid_ModifyText2(cell, tt, fontsize, tag) {

  try {
    var differences = 0;
    
    var align = XmlGetAttribute(tag, 'align');
    if (align.length > 0) {
      var alignattr = ParseAlignAttr(tag);
      if (alignattr != tiles[cell].TEXT[tt].ALIGN) {
          tiles[cell].TEXT[tt].ALIGN = alignattr;
          differences++;
      }
    }

    var clr = XmlGetAttribute(tag, 'clr');
    if (clr.length > 0) {
      if (clr != tiles[cell].TEXT[tt].CLR) {
        tiles[cell].TEXT[tt].CLR = clr;
        differences++;
      }
    }

    var cursor = XmlGetAttribute(tag, 'curs');
    if (cursor.length > 0) {
      var cursorattr = ParseCursAttr(tag);
      if (cursorattr != tiles[cell].TEXT[tt].CURSOR) {
          tiles[cell].TEXT[tt].CURSOR = cursorattr;
          differences++;
      }
    }

    var fontname = XmlGetAttribute(tag, 'fontname');
    if (fontname.length > 0) {
        if (fontname != tiles[cell].TEXT[tt].FONTNAME) {
          tiles[cell].TEXT[tt].FONT = fontname;
          differences++;
      }
    }

    var font = XmlGetAttribute(tag, 'font');
    if (font.length > 0) {
      var fontsize2 = ParseFontAttr(tag, fontsize);
      if (Number(fontsize2) != tiles[cell].TEXT[tt].FONTSIZE) {
          tiles[cell].TEXT[tt].FONTSIZE = Number(fontsize2);
          differences++;
      }
    }

    var layer2 = XmlGetAttribute(tag, 'layer');
    if (layer2.length > 0) {
      var layerattr = ParseLayerAttr(tag);
      if (layerattr != tiles[cell].TEXT[tt].LAYER) {
          tiles[cell].TEXT[tt].LAYER = layerattr;
          differences++;
      }
    }

    var loc = XmlGetAttribute(tag, 'loc');
    if (loc.length > 0) {
      var locattr = ParseLocAttr(tag);
      if (locattr != tiles[cell].TEXT[tt].LOC) {
          tiles[cell].TEXT[tt].LOC = locattr;
          differences++;
      }
    }

    var layer = tiles[cell].TEXT[tt].LAYER;

    var shorttag = tag.indexOf('/>');
    if (shorttag == -1) {
      var text = XmlGetText(tag);
      if (text.length > 0) {
        var lt = text.indexOf('&lt;');
        while (lt > -1) {
          text = text.substr(0, lt) + '<' + text.substr(lt + 4);
          lt = text.indexOf('&lt;');
        }
        var gt = text.indexOf('&gt;');
        while (gt > -1) {
          text = text.substr(0, gt) + '>' + text.substr(gt + 4);
          gt = text.indexOf('&gt;');
        }
      }
      if (text != tiles[cell].TEXT[tt].TEXT) {
        grid_console(tiles[cell].GID + '.Update text from :"' + tiles[cell].TEXT[tt].TEXT + '" to : "' + text + '"');
        if (InList(layer, tiles[cell].LAYER)) {
          var ctx = grids[tiles[cell].GINDEX].CTX;
          var startX =  tiles[cell].startX;
          var startY = tiles[cell].startY;
          grid_ClearText2(ctx, startX, startY, cell, tt);
        }
        tiles[cell].TEXT[tt].TEXT = text;
        differences++;
      }
    }

    if (differences == 0) {
      grid_console(tiles[cell].GID + '.ModifyText NO Changes');
    } else {
      if (InList(layer, tiles[cell].LAYER)) {
        //you would think you could just rewrite the text but you cannot.
        //you would get background, or erasement, or cursor wrong.
        //But for DlgBox2, if you repaint the tile (which overlaps onto 
        //a button in the next tile), you over write the button next door.
        if (tiles[cell].GID != "DlgBox2") {
          grid_FormatTile(cell);
        } else {
          grid_FormatText2(ctx, startX, startY, cell, tt);
        }
      }
    }
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyText2");
  }

  finally { }

} //grid_ModifyText2


function grid_ModifyImage(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].IMAGE.length; tt++) {
          if (id == tiles[cell].IMAGE[tt].ID) {
            grid_ModifyImage2(cell, tt, tag);
          }//for if for if
        }//for if for
      }//for if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyImage");
  }

  finally { }
}//grid_ModifyImage

function grid_ModifyImage2(cell, tt, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;
    var ctx = grids[tiles[cell].GINDEX].CTX;

    var layer = tiles[cell].IMAGE[tt].LAYER;
    if (InList(layer, tiles[cell].LAYER)) {
      grid_ClearImage2(ctx, startX, startY, cell, tt);
    }

    var layer2 = XmlGetAttribute(tag, 'layer');
    if (layer2.length > 0) {
      tiles[cell].IMAGE[tt].LAYER = ParseLayerAttr(tag);
    }

    var loc = XmlGetAttribute(tag, 'loc');
    if (loc.length > 0) {
      tiles[cell].IMAGE[tt].LOC = ParseLocAttr(tag);
    }

    var bclr = XmlGetAttribute(tag, 'bclr');
    if (bclr.length > 0) {
      tiles[cell].IMAGE[tt].BCLR = bclr;
    }

    var size = XmlGetAttribute(tag, 'size');
    if (size.length > 0) {
      tiles[cell].IMAGE[tt].SIZE = ParseSizeAttr(tag);
    }

    //make erasing image easier
    tiles[cell].IMAGE[tt].iX = 0;
    tiles[cell].IMAGE[tt].iY = 0;
    tiles[cell].IMAGE[tt].ilX = 0;
    tiles[cell].IMAGE[tt].ilY = 0;

    var name = XmlGetAttribute(tag, 'path').toLowerCase();
    if (name.length > 0) {
      if (name.toLowerCase() == 'null') {
        //grid_console(tiles[cell].GID + " MODIMAGE '" + tiles[cell].IMAGE[tt].NAME + "' -> No image");
        tiles[cell].IMAGE[tt].NAME = '';
        tiles[cell].IMAGE[tt].NAME2 = '';
        tiles[cell].IMAGE[tt].REF = -1;
      } else {
        tiles[cell].IMAGE[tt].NAME = name;
        tiles[cell].IMAGE[tt].NAME2 = nopath(name);
        tiles[cell].IMAGE[tt].REF = grid_parseimage2(name, cell, tt);
      }//if   
    }//if if
    grid_FormatTile(cell);
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyImage2");
  }

  finally { }
}//grid_ModifyImage2


function grid_ModifyCircle(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].CIRCLE.length; tt++) {
          if (id == tiles[cell].CIRCLE[tt].ID) {
            grid_ModifyCircle2(cell, tt, tag);
          }//for if for
        }//for if for
      }//for if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyCircle");
  }

  finally { }
}//grid_ModifyCircle

function grid_ModifyCircle2(cell, tt, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;
    var ctx = grids[tiles[cell].GINDEX].CTX;

    var layer = tiles[cell].CIRCLE[tt].LAYER;
    if (InList(layer, tiles[cell].LAYER)) {
      grid_ClearCircle2(ctx, startX, startY, cell, tt);
    }

    var layer = XmlGetAttribute(tag, 'layer');
    if (layer.length > 0) {
      tiles[cell].CIRCLE[tt].LAYER = ParseLayerAttr(tag);
    }

    var loc = XmlGetAttribute(tag, 'loc');
    if (loc.length > 0) {
      tiles[cell].CIRCLE[tt].LOC = ParseLocAttr(tag);
    }

    var diameter = XmlGetAttribute(tag, 'diameter');
    if (diameter.length > 0) {
      var dia = Number(diameter);
      var radius = dia / 2;
      tiles[cell].CIRCLE[tt].RAD = radius;
    }
    var weight = XmlGetAttribute(tag, 'wt');
    if (weight.length > 0) {
      tiles[cell].CIRCLE[tt].WT = ParseWtAttr(tag);
    }
    var clr = XmlGetAttribute(tag, 'clr');
    if (clr.length > 0) {
      tiles[cell].CIRCLE[tt].CLR = clr;
    }

    var fill = XmlGetAttribute(tag, 'fill');
    if (fill.length > 0) {
      tiles[cell].CIRCLE[tt].FILL = fill;
    }

    grid_FormatTile(cell);

  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyCircle2");
  }

  finally { }
}//grid_ModifyCircle2


function grid_ModifyLine(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].LINE.length; tt++) {
          if (id == tiles[cell].LINE[tt].ID) {
            grid_ModifyLine2(cell, tt, tag);
          }//for if for if
        }//for if for
      }//for if
    }//for 
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyLine");
  }

  finally { }

}//grid_ModifyLine



function grid_ModifyLine2(cell, tt, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;
    var ctx = grids[tiles[cell].GINDEX].CTX;

    var layer = tiles[cell].LINE[tt].LAYER;
    if (InList(layer, tiles[cell].LAYER)) {
      grid_ClearLine2(ctx, startX, startY, cell, tt);
    }

    var layer = XmlGetAttribute(tag, 'layer');
    if (layer.length > 0) {
      tiles[cell].LINE[tt].LAYER = ParseLayerAttr(tag);
    }

    var loc = XmlGetAttribute(tag, 'loc');
    if (loc.length > 0) {
      tiles[cell].LINE[tt].LOC = ParseLocAttr(tag);
    }

    var eloc = XmlGetAttribute(tag, 'eloc');
    if (eloc.length > 0) {
      tiles[cell].LINE[tt].ELOC = ParseElocAttr(tag);
    }

    var clr = XmlGetAttribute(tag, 'clr');
    if (clr.length > 0) {
      tiles[cell].LINE[tt].CLR = clr;
    }

    var weight = XmlGetAttribute(tag, 'wt');
    if (weight.length > 0) {
      tiles[cell].LINE[tt].WT = ParseWtAttr(tag);
    }

    grid_FormatTile(cell);

  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyLine2");
  }

  finally { }
}//grid_ModifyLine2

function grid_ModifyPolygon(data, tag) {


  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].POLYGON.length; tt++) {
          if (id == tiles[cell].POLYGON[tt].ID) {
            grid_ModifyPolygon2(cell, tt, tag);
          }//for if for if
        }//for if for
      }//for if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyPolygon");
  }

  finally { }
}//grid_ModifyPolygon

function grid_ModifyPolygon2(cell, tt, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;
    var ctx = grids[tiles[cell].GINDEX].CTX;

    var layer = tiles[cell].POLYGON[tt].LAYER;
    if (InList(layer, tiles[cell].LAYER)) {
      grid_ClearPolygon2(ctx, startX, startY, cell, tt);
    }

    var layer = XmlGetAttribute(tag, 'layer');
    if (layer.length > 0) {
      tiles[cell].POLYGON[tt].LAYER = ParseLayerAttr(tag);
    }

    var vrtx1 = XmlGetAttribute(tag, 'vrtx1');
    if (vrtx1.length > 0) {
      tiles[cell].POLYGON[tt].V1 = ParseVrtxAttr(tag, '1');
    }

    var vrtx2 = XmlGetAttribute(tag, 'vrtx2');
    if (vrtx2.length > 0) {
      tiles[cell].POLYGON[tt].V2 = ParseVrtxAttr(tag, '2');
    }

    var vrtx3 = XmlGetAttribute(tag, 'vrtx3');
    if (vrtx3.length > 0) {
      tiles[cell].POLYGON[tt].V3 = ParseVrtxAttr(tag, '3');
    }

    var vrtx4 = XmlGetAttribute(tag, 'vrtx4');
    if (vrtx4.length > 0) {
      tiles[cell].POLYGON[tt].V4 = ParseVrtxAttr(tag, '4');
    } else {
      tiles[cell].POLYGON[tt].V4 = ""
    }

    var weight = XmlGetAttribute(tag, 'wt');
    if (weight.length > 0) {
      tiles[cell].RECTANGLE[tt].WT = ParseWtAttr(tag);
    }

    var clr = XmlGetAttribute(tag, 'clr');
    if (clr.length > 0) {
      tiles[cell].POLYGON[tt].CLR = clr;
    }

    var fill = XmlGetAttribute(tag, 'fill');
    if (fill.length > 0) {
      tiles[cell].POLYGON[tt].FILL = fill;
    }

    grid_FormatTile(cell);

  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyPolygon2");
  }

  finally { }
}//grid_ModifyPolygon2

function grid_ModifyRectangle(data, tag) {

  try {
    var id = XmlGetAttribute(tag, "id");

    for (var cell = 0; cell < tiles.length; cell++) {
      if (tiles[cell].GID == data.Id) {
        for (var tt = 0; tt < tiles[cell].RECTANGLE.length; tt++) {
          if (id == tiles[cell].RECTANGLE[tt].ID) {
            grid_ModifyRectangle2(cell, tt, tag);
          }
        }//for
      }//if
    }//for
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyRectangle");
  }

  finally { }
} //grid_ModifyRectangle


function grid_ModifyRectangle2(cell, tt, tag) {

  try {
    var row = tiles[cell].ROW;
    var column = tiles[cell].COLUMN;
    var startX = CalcOffset(column - grids[tiles[cell].GINDEX].SCOL, grids[tiles[cell].GINDEX].TW) + grids[tiles[cell].GINDEX].PADL;
    var startY = CalcOffset(row - grids[tiles[cell].GINDEX].SROW, grids[tiles[cell].GINDEX].TH) + grids[tiles[cell].GINDEX].PADT;

    var ctx = grids[tiles[cell].GINDEX].CTX;

    var layer = tiles[cell].RECTANGLE[tt].LAYER;
    if (InList(layer, tiles[cell].LAYER)) {
      grid_ClearRectangle2(ctx, startX, startY, cell, tt);
    }

    var layer = XmlGetAttribute(tag, 'layer');
    if (layer.length > 0) {
      tiles[cell].RECTANGLE[tt].LAYER = ParseLayerAttr(tag);
    }

    var loc = XmlGetAttribute(tag, 'loc');
    if (loc.length > 0) {
      tiles[cell].RECTANGLE[tt].LOC = ParseLocAttr(tag);
    }

    var size = XmlGetAttribute(tag, 'size');
    if (size.length > 0) {
      tiles[cell].RECTANGLE[tt].SIZE = ParseSizeAttr(tag);
    }

    var weight = XmlGetAttribute(tag, 'wt');
    if (weight.length > 0) {
      tiles[cell].RECTANGLE[tt].WT = ParseWtAttr(tag);
    }

    var clr = XmlGetAttribute(tag, 'clr');
    if (clr.length > 0) {
      tiles[cell].RECTANGLE[tt].CLR = clr;
    }

    var fill = XmlGetAttribute(tag, 'fill').toUpperCase();
    if (fill.length > 0) {
      tiles[cell].RECTANGLE[tt].FILL = fill;
      if ((fill == "FFFFFF") || (fill == "#FFFFFF")) {
        tiles[cell].RECTANGLE[tt].FOCUS = 0;
      } else {
        tiles[cell].RECTANGLE[tt].FOCUS = 1;
      }
    }

    grid_FormatTile(cell);
  }

  catch (ex) {
    console.log("exception thrown in grid_ModifyRectangle2");
  }

  finally { }
} //grid_ModifyRectangle2
