/*
 *  This material is the joint property of FANUC America Corporation and
 *  FANUC LTD Japan, and must be returned to either FANUC America Corporation
 *  or FANUC LTD Japan immediately upon request.  This material and
 *  the information illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without the
 *  prior written consent of both FANUC America Corporation and FANUC LTD Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2017
 *           FANUC America Corporation
 *           FANUC LTD Japan
 *  +
 *  Module: draw2d.js
 *  
 *  Description:
 *    Draw2D plugin
 *
 *  Author: Michael Dow
 *          FANUC America Corporation
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  19-OCT-2017 DOWMW    pr50481 - Initial version
 *  15-FEB-2018 EVANSJA  pr50687b - Support for draw2d on dialog boxes.
 *  28-MAR-2018 DOWMW    pr50481a - Fix some bugs and add new features
 *  24-MAY-2019 DOWMW    v930pr50376b - Change logic for End of Transmission
 *  20-JUN-2019 DOWMW    pr51640 - Fixed lots of issues found during testing.
 *  20-SEP-2019 DOWMW    pr51640b - Fixed logic where ID is not needed.
 *  18-SEP-2020 DOWMW    pr51640c - Fixed some issues with invalid characters.
 *  -
*/

var canvasIDs = [];
var current_canvas = "canvas"

var enabledLayers = [0];
var invertY = 0; /* default origin to top */
//var combinedBuffer = '';
var drawEntities;
var drawEntities_array = [];
var forecolor = "#000000";
var backcolor = "#000000";
var fontsize = 12;

/** The table of color definition for DRAW2D*/
var colorTable = {
  red:        "#FF0000",
  orange:     "#FFC800",
  yellow:     "#FFFF00",
  gray:       "#808080",
  lightgray:  "#C0C0C0",
  darkgray:   "#404040",
  green:      "#00FF00",
  black:      "#000000",
  magenta:    "#FF00FF",
  cyan:       "#00FFFF",
  blue:       "#0000FF",
  white:      "#FFFFFF"
};

var drawEntities_temp = {
  combinedBuffer: "",
  text: {
    id: "",
    layer: 0,
    font: "",
    locate: [0, 0],
    color: 'black',
    bold: false,
    italic: false
  },
  line: {
    id: "",
    layer: 0,
    points: [0, 0, 1, 1], // Only 4 points allowed
    color: 'black',
  },
  path: {
    id: "",
    layer: 0,
    points: [],
    color: 'black',
  },
  circ: {
    id: "",
    layer: 0,
    locate: [0, 0],
    color: 'black',
    radius: 1,
    weight: 1,
    fillColor: "",
    halign: "",
    valign: "",
  },
  rect: {
    id: "",
    layer: 0,
    locate: [0, 0],
    color: 'black',
    height: 1,
    width: 1,
    fillColor: "",
    halign: "",
    valign: "",
    radius: 0,
  },
  diam: {
    id: "",
    layer: 0,
    locate: [0, 0],
    color: 'black',
    height: 1,
    width: 1,
    halign: "",
    valign: "",
  },
  imag: {
    id: "",
    layer: 0,
    locate: [0, 0],
    height: 1,
    width: 1,
    halign: "",
    valign: "",
    imagename: "",
  }
};

// Private functions 
// Create control only when PMON Server is ready
function draw2d_ServerReady(data) {
  // Initialize control data.
  draw2d_InitCtlData(data);

  // Create control.
  draw2d_CreateCtl(data);

  // Initialize events.
  draw2d_InitRobotEvent(true, data);

} // draw2d_ServerReady

function WebGLDetector() {

    try {
        var canvas = document.createElement('canvas'); return !!(window.WebGLRenderingContext && (canvas.getContext('webgl') || canvas.getContext('experimental-webgl')));
    } catch (e) {
        return false;
    }
}
// New instance of control.
function draw2d_NPP_New(data) {
  if (!WebGLDetector()) {
    window.location.href = "/frh/diageg/TPIF.htm#TPIF-266";
  }
  if (undefined != data.PipeMonRate) {
    data.PipeMonRate = Number(data.PipeMonRate);
  }

  SetCommonParams(data);
  if (data.IsDialog) {
    draw2d_ServerReady(data);
  }
  else { 
    top.rpcmc_getTaskIdx(data.fDeviceId, function(taskIdx) { 
      data.fTaskIdx = taskIdx;

      // Complete after top.rpcmc_getTaskIdx completes.
      draw2d_ServerReady(data);
    });
  }

} // draw2d_NPP_New

// Destroy instance of control.
function draw2d_NPP_Destroy(data) {

  // Uninitialize events.
  draw2d_InitRobotEvent(false, data);

  // Delete control.
  draw2d_DeleteCtl(data);

} // draw2d_NPP_Destroy

// Private functions

// Set the border.
function draw2d_SetBorder(data) {

  var border = parseInt(data.Border);
  if (border > 0) {
    data.$this.css({"border-style":"inset", "border-width":"' + border + 'px", "border-color":"white"});
  }
  else if (border < 0) {
    border *= -1;
    data.$this.css({"border-style":"inset", "border-width":"' + border + 'px", "border-color":"white"});
  }
  else {
    data.$this.css("border-style", "none");
  }

} // draw2d_SetBorder

// Update Control.
function draw2d_UpdateCtl(data) {


} // draw2d_UpdateCtl

function draw2d_SendPkt(fc, data, optional) {
} // draw2d_UpdateCtl

// Initialize or uninitialize pipe event.
function draw2d_InitPipeEvent(init, data) { 
  /* Get pipe data rate - Default 1sec */
  if (data.PipeMonRate < 0) {
    data.PipeMonRate = 1000;
  }
  
  // Notify PMON to start/stop monitoring our pipe
  if (init) {
    top.jQuery.filelis.bind('PipeEvent', data, draw2d_HandlePipeEvent); // Attach handler for PipeEvent
    top.rpcmc_startPipeMonitor(data.Pipe, data.PipeMonRate); // Start PMON monitor for our pipe
    // Attach handler for KeyReleasedEvent - This is used to listen to end of pipe transmission
    top.jQuery.keylis.bind('KeyReleasedEvent', draw2d_HandleKeyEvent);
  }
  else {
    top.rpcmc_stopPipeMonitor(data.Pipe); // Stop PMON monitor for our pipe
    top.jQuery.filelis.unbind('PipeEvent', draw2d_HandlePipeEvent); // Detach handler for PipeEvent.
    // Detach handler for KeyReleasedEvent.
    top.jQuery.keylis.unbind('KeyReleasedEvent', draw2d_HandleKeyEvent);
  }
} // draw2d_InitPipeEvent

// Replace any indirection with actual value.
function draw2d_IndirectReplace(data) {

  var l_ind;
  var pos;

 $.each(data, function( argn, argv ) { 
    if (typeof argv !== 'string') {
      return;
    }
    // Value contain !PaneId?
    if ((pos = argv.toLowerCase().indexOf("!paneid")) >= 0) {
      argv = argv.substring(0, pos) + data.fTaskIdx + argv.substring(pos+7);
      data[argn] = argv;
    }
  });
} // draw2d_IndirectReplace

// Initialize Control Data.
function draw2d_InitCtlData(data) {

  // Process parameters.
  // Name not supported
  // Border
  // Pipe
  // Scene
  // SubPane
  // Verbose not supported
  // ExecConnectId not required (only for ActiveX Controls)

  data.Pipe = data.Pipe.toUpperCase();
  draw2d_IndirectReplace(data);

  data.SubPane = parseInt(data.SubPane);

  if (data.BackColor == "") {
    data.BackColor = data.InvisibleColor;
  }
  else {
    data.BackColor = translateColor(data.BackColor);
  }
  backcolor = data.BackColor;
  if (data.ForeColor == "") {
    data.ForeColor = data.TextColor;
  }
  else {
    data.ForeColor = translateColor(data.ForeColor);
  }
  forecolor = data.ForeColor;

  if (data.FontSize !== "undefined" && parseFloat(data.FontSize) > 0) {
    fontsize = parseFloat(data.FontSize);
  }
} // draw2d_InitCtlData

// Initialize or uninitialize events for each type.
function draw2d_InitRobotEvent(init, data) {
  // Start/stop the Pipe Event.
  draw2d_InitPipeEvent(init, data);

} // draw2d_InitRobotEvent

// Create Control.
function draw2d_CreateCtl(data) {

  data.$this.css("display", "inline-block");

  if (data.width.indexOf("%") >= 0) {
    var w = top.getLegacyW(data.fDeviceId);
    var dec = parseFloat(data.width);
    data.width = Math.round(w*dec/100);
  }
  data.$this.css("width", data.width + "px");

  if (data.height.indexOf("%") >= 0) {
    var h = top.getLegacyH(data.fDeviceId);
    var dec = parseFloat(data.height);
    data.height = Math.round(h*dec/100);
  }
  data.$this.css("height", data.height + "px");

  data.$this.css("vertical-align", "top");
  draw2d_SetBorder(data);
  SetColors(data);
  SetFont(data);

  draw2d_InitDraw(data);

  // Attach handler for click event.
  data.$this.bind("click", data, draw2d_HandleClickEvent);

} // draw2d_CreateCtl

// Handle Control events.
function draw2d_CtlEvent(data) {

  var sb_state;

  if (data.IsEcho) {
    return;
  }
  if (data.dlgDismiss == "1") {
    // Dismiss dialog box.
    top.rpcmc_sendKeyCode(tk_cancel_c);
  }

} // draw2d_CtlEvent

// Delete Control Data.
function draw2d_DeleteCtl(data) {
} // draw2d_DeleteCtl

// Private functions

function draw2d_HandleKeyEvent(event, key) {
  // These keys came from the controller.
  /* Indicates the pipe transmission is complete */
  if (key == 4) {
    /* Erase buffer here incase processing later fails */
    var drawBuffer = drawEntities.combinedBuffer;
    drawEntities.combinedBuffer = '';
    /* Strip off sequence number if it contains it */
    if (drawBuffer.includes('?_=')) {
      drawBuffer = drawBuffer.substring(0, drawBuffer.indexOf('?_='));
    }
    draw2d_ParseData(drawBuffer);
  }
}

function draw2d_HandlePipeEvent(event, file, buffer) {
  event.preventDefault();
  var data = event.data || event;
  if (file == data.Pipe) {
    if(data.Id) {
      current_canvas = data.Id;
      for (var idx = 0; idx < canvasIDs.length; idx++) {
        if(canvasIDs[idx] == data.Id) {
          drawEntities = drawEntities_array[idx];
          current_canvas = canvasIDs[idx];
        }
      }
    }
    if (buffer.length > 0) {
      drawEntities.combinedBuffer = drawEntities.combinedBuffer.concat(buffer);
    }
  }
  return true;
} // draw2d_HandlePipeEvent

function draw2d_getXmlAsString(xmlDom){
  return (typeof XMLSerializer!=="undefined") ? 
         (new window.XMLSerializer()).serializeToString(xmlDom) : 
          xmlDom.xml;
} // draw2d_getXmlAsString

function draw2d_HandleClickEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  draw2d_CtlEvent(data);
  return true;
} // draw2d_HandleClickEvent

function draw2d_invertY(yData) {
  if(invertY > 0) {
    return (invertY - yData);
  }
  else {
    return yData;
  }
}

function draw2d_DrawText(textData) {
  var ctx = document.getElementById(current_canvas).getContext("2d");

  /* Start Path */
  ctx.beginPath();

  /* Set Font Style */
  /* TODO - Lots of missing features, only supporting size */
  /* default fontsize */
  eval("ctx.font = \"" + fontsize + "px Arial\"");
  if (textData.font) {
    eval("ctx.font = \"" + textData.font + "px Arial\"");
  }

  /* Set font color */
  ctx.fillStyle = forecolor;
  if (typeof textData.color !== 'undefined') {
    if(!isNaN(textData.color) || textData.color.indexOf('#') == -1) {
      ctx.fillStyle = translateColor(parseInt(textData.color));
    }
    else {
      ctx.fillStyle = textData.color;
    }
  }

  /* TODO - Can we support bold and italic? */

  /* Set text halign - center by default (Spec doesnt even say it is supported) */
  ctx.textAlign = 'center';
  if (textData.halign) {
    if (textData.halign.toLowerCase() == "left") {
      ctx.textAlign = 'left';
    }
    else if (textData.halign.toLowerCase() == "right") {
      ctx.textAlign = 'right';
    }
    else if (textData.halign.toLowerCase() == "center") {
      // Do nothing
    }
  }
  
  /* Set circle valign - top by default (Spec doesnt even say it is supported) */
  ctx.textBaseline = 'top'
  if (textData.valign) {
    if (textData.valign.toLowerCase() == "top") {
      // Do nothing
    }
    else if (textData.valign.toLowerCase() == "bottom") {
      ctx.textBaseline = 'bottom'
    }
    else if (textData.valign.toLowerCase() == "middle") {
      ctx.textBaseline = 'middle'
    }
  }
  
  ctx.fillText(textData.string, textData.locate[0], draw2d_invertY(textData.locate[1]));

}

function draw2d_DrawLine(lineData) {
  var ctx = document.getElementById(current_canvas).getContext("2d");

  /* Start Path */
  ctx.beginPath();

  /* Plot Lines */
  /* First two points are the starting location */
  ctx.moveTo(lineData.points[0], draw2d_invertY(lineData.points[1]));
  for (var j = 2; j < lineData.points.length - 1; j += 2) {
    ctx.lineTo(lineData.points[j], draw2d_invertY(lineData.points[j + 1]));
  }

  /* Set line color */
  ctx.strokeStyle = forecolor
  if (typeof lineData.color !== 'undefined') {
    if(!isNaN(lineData.color) || lineData.color.indexOf('#') == -1) {
      ctx.strokeStyle = translateColor(parseInt(lineData.color));
    }
    else {
      ctx.strokeStyle = lineData.color;
    }
  }

  ctx.stroke();
}

function draw2d_DrawCircle(circleData) {
  var ctx = document.getElementById(current_canvas).getContext("2d");

  var locX, locY, radius;

  /* Start Path */
  ctx.beginPath();

  /* Set Circle Location */
  locX = circleData.locate[0];
  locY = draw2d_invertY(circleData.locate[1]);

  /* Set circle color */
  ctx.strokeStyle = forecolor;
  if (typeof circleData.color !== 'undefined') {
    if(!isNaN(circleData.color) || circleData.color.indexOf('#') == -1) {
      ctx.strokeStyle = translateColor(parseInt(circleData.color));
    }
    else {
      ctx.strokeStyle = circleData.color;
    }
  }

  /* Set circle radius */
  if (circleData.radius) {
    radius = circleData.radius;
  }

  /* Set circle weight */
  if (circleData.weight) {
    ctx.lineWidth = circleData.weight;
  }

  /* Set circle halign - center by default (Spec says left but that is not the logic) */
  if (circleData.halign) {
    if (circleData.halign.toLowerCase() == "left") {
      locX = locX + radius;
    }
    else if (circleData.halign.toLowerCase() == "right") {
      locX = locX - radius;
    }
    else if (circleData.halign.toLowerCase() == "center") {
      // Do nothing
    }
  }

  /* Set circle valign - middle by default (Spec says bottom but that is not the logic) */
  if (circleData.valign) {
    if (circleData.valign.toLowerCase() == "top") {
      locY = locY + radius;
    }
    else if (circleData.valign.toLowerCase() == "bottom") {
      locY = locY - radius;
    }
    else if (circleData.valign.toLowerCase() == "middle") {
      // Do Nothing
    }
  }

  /* Draw Circle */
  ctx.arc(parseInt(locX), parseInt(locY), radius, 0 * Math.PI, 2 * Math.PI); /* (x,y,r,startAngle,endAngle,counterclockwise) */

  /* Set circle fill color - Needs to go after ARC */
  if (typeof circleData.fillColor !== 'undefined') {
    if(!isNaN(circleData.fillColor) || circleData.fillColor.indexOf('#') == -1) {
      ctx.fillStyle = translateColor(parseInt(circleData.fillColor));
    }
    else {
      ctx.fillStyle = circleData.fillColor;
    }
    ctx.fill();
  }
  
  ctx.stroke();
}

function draw2d_DrawRectangle(rectData) {
  var ctx = document.getElementById(current_canvas).getContext("2d");

  var locX, locY, rectWidth, rectHeight;
  var cornerRadius = 0;

  /* Start Path */
  ctx.beginPath();

  /* Set Rectangle Location */
  locX = rectData.locate[0];
  locY = draw2d_invertY(rectData.locate[1]);

  /* Set Rectangle color */
  ctx.strokeStyle = forecolor;
  if (typeof rectData.color !== 'undefined') {
    if(!isNaN(rectData.color) || rectData.color.indexOf('#') == -1) {
      ctx.strokeStyle = translateColor(parseInt(rectData.color));
    }
    else {
      ctx.strokeStyle = rectData.color;
    }
  }

  /* Set Rectangle height */
  if (rectData.height) {
    rectHeight = rectData.height;
  }

  /* Set Rectangle width */
  if (rectData.width) {
    rectWidth = rectData.width;
  }
  
  /* Set Rectangle weight */
  if (rectData.weight) {
    ctx.lineWidth = rectData.weight;
  }

  /* Set Rectangle corner radius */
  if (rectData.radius) {
    cornerRadius = rectData.radius;
  }

  locX = locX - (rectWidth / 2);
  locY = locY - (rectHeight / 2);

  /* Set Rectangle halign - center by default (Spec says left but that is not the logic) */
  if (rectData.halign) {
    if (rectData.halign.toLowerCase() == "left") {
      locX = locX + (rectWidth / 2);
    }
    else if (rectData.halign.toLowerCase() == "right") {
      locX = locX - (rectWidth / 2);
    }
    else if (rectData.halign.toLowerCase() == "center") {
      // Do Nothing
    }
  }

  /* Set Rectangle valign - middle by default (Spec says bottom but that is not the logic) */
  if (rectData.valign) {
    if (rectData.valign.toLowerCase() == "top") {
      locY = locY + (rectHeight / 2);
    }
    else if (rectData.valign.toLowerCase() == "bottom") {
      locY = locY - (rectHeight / 2);
    }
    else if (rectData.valign.toLowerCase() == "middle") {
      // Do Nothing
    }
  }
  
  /* Set Rectangle corner radius */
  if (rectData.radius) {
    cornerRadius = rectData.radius;
    var r = locX + rectWidth;
    var b = locY + rectHeight;
    ctx.moveTo(locX + cornerRadius, locY);
    ctx.lineTo(r - cornerRadius, locY);
    ctx.quadraticCurveTo(r, locY, r, locY + cornerRadius);
    ctx.lineTo(r, locY + rectHeight - cornerRadius);
    ctx.quadraticCurveTo(r, b, r - cornerRadius, b);
    ctx.lineTo(locX + cornerRadius, b);
    ctx.quadraticCurveTo(locX, b, locX, b - cornerRadius);
    ctx.lineTo(locX, locY + cornerRadius);
    ctx.quadraticCurveTo(locX, locY, locX + cornerRadius, locY);
  }
  else {
    ctx.rect(locX, locY, rectWidth, rectHeight);
  }

  /* Set Rectangle fill color - Needs to go after the rect is drawn */
  if (typeof rectData.fillColor !== 'undefined') {
    if(!isNaN(rectData.fillColor) || rectData.fillColor.indexOf('#') == -1) {
      ctx.fillStyle = translateColor(parseInt(rectData.fillColor));
    }
    else {
      ctx.fillStyle = rectData.fillColor;
    }
    ctx.fill();
  }

  ctx.stroke();
}

function draw2d_DrawDiamond(diamData) {
  /* Basically we need to convert the diamond shape to 4 lines and pass to draw line */
  var ctx = document.getElementById(current_canvas).getContext("2d");
  var locX, locY, diamWidth, diamHeight;

  /* Set Diamond Location */
  locX = diamData.locate[0];
  locY = draw2d_invertY(diamData.locate[1]);

  /* Set Diamond height */
  if (diamData.height) {
    diamHeight = diamData.height;
  }

  /* Set Diamond width */
  if (diamData.width) {
    diamWidth = diamData.width;
  }
  
  /* Set Diamond weight */
  if (diamData.weight) {
    ctx.lineWidth = diamData.weight;
  }

  /* Set Diamond halign - center by default (Spec says left but that is not the logic) */
  if (diamData.halign) {
    if (diamData.halign.toLowerCase() == "left") {
      locX = locX + (diamWidth / 2);
    }
    else if (diamData.halign.toLowerCase() == "right") {
      locX = locX - (diamWidth / 2);
    }
    else if (diamData.halign.toLowerCase() == "center") {
      // Do Nothing
    }
  }

  /* Set Diamond valign - middle by default (Spec says bottom but that is not the logic) */
  if (diamData.valign) {
    if (diamData.valign.toLowerCase() == "top") {
      locY = locY + (diamHeight / 2);
    }
    else if (diamData.valign.toLowerCase() == "bottom") {
      locY = locY - (diamHeight / 2);
    }
    else if (diamData.valign.toLowerCase() == "middle") {
      // Do Nothing
    }
  }

  diamData.points = [];
  diamData.points[0] = locX;
  diamData.points[1] = locY - (diamHeight / 2);
  diamData.points[2] = locX + (diamWidth / 2);
  diamData.points[3] = locY;
  diamData.points[4] = locX;
  diamData.points[5] = locY + (diamHeight / 2);
  diamData.points[6] = locX - (diamWidth / 2);
  diamData.points[7] = locY;
  //diamData.points[8] = locX;
  //diamData.points[9] = locY - (diamHeight / 2);

  draw2d_DrawLine(diamData);
  ctx.closePath();
  ctx.stroke();
  
  /* Set Diamond fill color - Needs to go after the rect is drawn */
  if (typeof diamData.fillColor !== 'undefined') {
    if(!isNaN(diamData.fillColor) || diamData.fillColor.indexOf('#') == -1) {
      ctx.fillStyle = translateColor(parseInt(diamData.fillColor));
    }
    else {
      ctx.fillStyle = diamData.fillColor;
    }
    ctx.fill();
  }
}

function draw2d_DrawImage(imageData) {
  var ctx = document.getElementById(current_canvas).getContext("2d");
  var imageObj = new Image;

  var locX, locY, imagWidth, imagHeight;

  imageObj.onload = function () {
  
  /* Set Circle Location */
  locX = imageData.locate[0];
  locY = draw2d_invertY(imageData.locate[1]);

  /* Default height to size of image */
  imagHeight = this.height;
  imagWidth = this.width;
  
  /* Set Image height */
  if (imageData.height) {
    imagHeight = imageData.height;
  }

  /* Set Image width */
  if (imageData.width) {
    imagWidth = imageData.width;
  }

  locX = locX - (imagWidth/2); /* Origin center of image */
  /* Set Image halign - center by default (Spec says left but that is not the logic) */
  if (imageData.halign) {
    if (imageData.halign.toLowerCase() == "left") {
      locX = locX + (imagWidth / 2);
    }
    else if (imageData.halign.toLowerCase() == "right") {
      locX = locX - (imagWidth / 2);
    }
    else if (imageData.halign.toLowerCase() == "center") {
      // Do Nothing
    }
  }

  locY = locY - (imagHeight/2); /* Origin center of image */
  /* Set Image valign - middle by default (Spec says bottom but that is not the logic) */
  if (imageData.valign) {
    if (imageData.valign.toLowerCase() == "top") {
      locY = locY + (imagHeight / 2);
    }
    else if (imageData.valign.toLowerCase() == "bottom") {
      locY = locY - (imagHeight / 2);
    }
    else if (imageData.valign.toLowerCase() == "middle") {
      // Do Nothing
    }
  }
    
    ctx.drawImage(imageObj, locX, locY, imagWidth, imagHeight);
  };

  imageObj.src = imageData.imagename;
  //imageObj.src = "/mc/done.gif";
}

function draw2d_DeleteEntitieID(entitieID) 
{
    for (var i = 0; i < drawEntities.text.length; i++) {
      if(drawEntities.text[i].id == entitieID) {
        drawEntities.text.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.line.length; i++) {
      if (drawEntities.line[i].id == entitieID) {
        drawEntities.line.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.path.length; i++) {
      if (drawEntities.path[i].id == entitieID) {
        drawEntities.path.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.circ.length; i++) {
      if (drawEntities.circ[i].id == entitieID) {
        drawEntities.circ.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.rect.length; i++) {
      if (drawEntities.rect[i].id == entitieID) {
        drawEntities.rect.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.diam.length; i++) {
      if (drawEntities.diam[i].id == entitieID) {
        drawEntities.diam.splice(i, 1);
      }
    }
    for (var i = 0; i < drawEntities.imag.length; i++) {
      if (drawEntities.imag[i].id == entitieID) {
        drawEntities.imag.splice(i, 1);
      }
    }
}

function draw2d_DrawEntities() {

  var ctx = document.getElementById(current_canvas).getContext("2d");
  /* clear the canvas */
  ctx.fillStyle = backcolor;
  ctx.fillRect(0, 0, ctx.canvas.width, ctx.canvas.height);

  /* Sort the layers from lowest to highest */
  enabledLayers.sort(function (a, b) {
    return a - b
  })

  for (var idx = 0; idx < enabledLayers.length; idx++) {
    for (var i = 0; i < drawEntities.text.length; i++) {
      if(drawEntities.text[i].layer == enabledLayers[idx]) {
        draw2d_DrawText(drawEntities.text[i]);
      }
    }
    for (var i = 0; i < drawEntities.line.length; i++) {
      if (drawEntities.line[i].layer == enabledLayers[idx]) {
        draw2d_DrawLine(drawEntities.line[i]);
      }
    }
    for (var i = 0; i < drawEntities.path.length; i++) {
      if (drawEntities.path[i].layer == enabledLayers[idx]) {
        draw2d_DrawLine(drawEntities.path[i]);
      }
    }
    for (var i = 0; i < drawEntities.circ.length; i++) {
      if (drawEntities.circ[i].layer == enabledLayers[idx]) {
        draw2d_DrawCircle(drawEntities.circ[i]);
      }
    }
    for (var i = 0; i < drawEntities.rect.length; i++) {
      if (drawEntities.rect[i].layer == enabledLayers[idx]) {
        draw2d_DrawRectangle(drawEntities.rect[i]);
      }
    }
    for (var i = 0; i < drawEntities.diam.length; i++) {
      if (drawEntities.diam[i].layer == enabledLayers[idx]) {
        draw2d_DrawDiamond(drawEntities.diam[i]);
      }
    }
    for (var i = 0; i < drawEntities.imag.length; i++) {
      if (drawEntities.imag[i].layer == enabledLayers[idx]) {
        draw2d_DrawImage(drawEntities.imag[i]);
      }
    }
  }

}

function draw2d_InitDraw(data) {

  if (data) {
    if(data.Id) {
      canvasIDs.push(data.Id);
      current_canvas = data.Id;
      drawEntities_array.push(Object.assign({}, drawEntities_temp));
      drawEntities = drawEntities_array[drawEntities_array.length - 1];
      var test = drawEntities_temp;
      var objCopy = Object.assign({}, drawEntities_temp);
    }
    /* Create Canvas for the draw area */
    var out = '<canvas id="' + current_canvas + '"></canvas>';
    data.$this.html(out);
    var ctx = document.getElementById(current_canvas).getContext("2d");
    ctx.canvas.height = data.height;
    ctx.canvas.width = data.width;
    
    /* Set Invert if set true */
    if (data.InvertY !== undefined && data.InvertY === "1") {
      invertY = ctx.canvas.height; /* change origin to bottom */
    }
  }
  else {
    var ctx = document.getElementById(current_canvas).getContext("2d");
    /* clear the canvas */
    ctx.fillStyle = backcolor;
    ctx.fillRect(0, 0, ctx.canvas.width, ctx.canvas.height);
  }
  
  drawEntities.text = [];
  drawEntities.line = [];
  drawEntities.path = [];
  drawEntities.circ = [];
  drawEntities.rect = [];
  drawEntities.diam = [];
  drawEntities.imag = [];
  //enabledLayers = []; /* Following old logic - Clear only clears screen, active lavers stay active */

  if (data && data.Data) {
    draw2d_ParseData(data.Data)
  }

}

function draw2d_ParseData(drawData) {

  var dataString = drawData.trim();
  //dataString = dataString.replace(/\s\s+/g, ' '); /* Replace whitespace with single space to split later */
  //dataString = dataString.replace(/'/g, '"');
  //var splitData = dataString.match(/(?:[^\s"]+|"[^"]*")+/g) /* Split data at spaces, but not if space is inside quotes */

  /* Replace single quote with double quote */
  dataString = dataString.replace(/'/g, '"');
  /* remove whitespace before and after equal sign */
  dataString = dataString.replace(/\s*=\s*/g, '=');
  /* remove invalid characters '?' */
  dataString = dataString.replace(/\uFFFD/g, '')
  var splitData = dataString.match(/(?:[^\s"]+|"[^"]*")+/g)

  var currentEntity, entityData;

  entityData = []
  entityData[0] = currentEntity = "none";

  var idx = 0;
  var bNewType = false;

  for (var i = 0; i < splitData.length; i++) {
    if (splitData[i]) {
      var tempLineData = splitData[i].trim();
      /* DEBUG */
      //console.log(tempLineData);
      if (tempLineData.toLocaleLowerCase() == "clear") {
        draw2d_InitDraw();
        continue;
      }

      /* Split data at equal, but not if equal is inside quotes */
      var subData = tempLineData.match(/(?:[^="]+|"[^"]*")+/g)
      if (subData[0].toLowerCase() === "text") {
        currentEntity = "text";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "line") {
        currentEntity = "line";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "path") {
        currentEntity = "path";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "circ") {
        currentEntity = "circ";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "rect") {
        currentEntity = "rect";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "diam") {
        currentEntity = "diam";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "imag") {
        currentEntity = "imag";
        bNewType = true;
      }
      else if (subData[0].toLowerCase() === "inverty") {
        invertY = 0;
        if (subData[1] == 1) {
          var ctx = document.getElementById(current_canvas).getContext("2d");
          invertY = ctx.canvas.height; /* change origin to bottom */
        }
        continue;
      }
      else if (subData[0].toLowerCase() === "delete") {
        draw2d_DeleteEntitieID(subData[1].toLowerCase());
        continue;
      }
      else if (subData[0].toLowerCase() === "verbose") {
        /* Not sure what this is */
        continue;
      }
      else if (subData[0].toLowerCase() === "layeron") {
        var bAddLayer;
        var tempLayers = subData[1].split(',');

        /* If enabled layers is empty just set the values */
        if (enabledLayers.length == 0) {
          enabledLayers = tempLayers;
          continue;
        }

        for (var idx1 = 0; idx1 < tempLayers.length; idx1++) {
          bAddLayer = true;
          for (var idx2 = 0; idx2 < enabledLayers.length; idx2++) {
            if (tempLayers[idx1] == enabledLayers[idx2]) {
              bAddLayer = false;
              break
            }
          }
          /* If layer is not already on the list, add layer */
          if (bAddLayer) {
            enabledLayers.push(parseInt(tempLayers[idx1]));
          }
        }
        continue;
      }
      else if (subData[0].toLowerCase() === "layeroff") {
        var bDelLayer;
        var tempLayers = subData[1].split(',');

        for (var idx1 = 0; idx1 < tempLayers.length; idx1++) {
          bDelLayer = false;
          for (var idx2 = 0; idx2 < enabledLayers.length; idx2++) {
            if (tempLayers[idx1] == enabledLayers[idx2]) {
              bAddLayer = true;
              break
            }
          }
          /* if layer was on the list, remove layer */
          if (bAddLayer) {
            enabledLayers.splice(idx2, 1);
          }
        }
        continue;
      }

      /* Clean up value text */
      if (subData[1] !== undefined) {
        /* remove quotes - They will be added in later if needed */
        subData[1] = subData[1].replace(/"/g, "");
        /* Remove leading and training whitespace */
        subData[1] = subData[1].trim();
      }

      /* Setup default color and font */
      if (subData[0].toLowerCase() === "fontsize") {
        if (subData[1] !== undefined) {
          fontsize = subData[1];
        }
      }
      else if (subData[0].toLowerCase() === "forecolor" || subData[0].toLowerCase() === "backcolor" || subData[0].toLowerCase() === "fontsize") { /* Setup default colors */
        if (subData[1] !== undefined) {
          var curColor = subData[1];
          /* Convert to hex and put in proper format */
          if (!isNaN(curColor) || curColor.indexOf('#') == -1) {
            curColor = translateColor(parseInt(curColor));
          }
          eval(subData[0].toLowerCase() + " = \"" + curColor + "\"");
        }
      }
      else if (currentEntity != "none") {
        if (bNewType) {
          bNewType = false;
          /* get index for entity type */
          idx = -1
          var tempID;
          var tempLength = eval("drawEntities." + currentEntity + ".length");
          if (tempLength > 0) {  //Need to check and see if new ID or not
            for (var idIdx = 0; idIdx < tempLength && idx == -1; idIdx++) {
              tempID = eval("drawEntities." + currentEntity + "[" + idIdx + "].id");
              if (subData[1] !== undefined && subData[1].toLowerCase() === tempID)
                idx = idIdx;
            }
          }
          if (idx == -1) { // No match found, make new index
            eval("drawEntities." + currentEntity + ".push([])");
            tempLength = eval("drawEntities." + currentEntity + ".length");
            idx = tempLength - 1;
            if (subData[1] !== undefined) {
              tempID = eval("drawEntities." + currentEntity + "[" + idx + "].id = \"" + subData[1].toLowerCase() + "\"");
            }
            eval("drawEntities." + currentEntity + "[" + idx + "].layer = 0");
          }
        }
        else {
          if (subData[0].toLowerCase() === "points" || subData[0].toLowerCase() === "locate") {
            var pointData = subData[1].replace(/[^\d.]+/g, ",");
            pointData = pointData.split(',');
            /* Initilize array if it does not exist or it is not a path */
            if (eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " === undefined") || currentEntity != "path") {
              eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " = []");
            }
            for (var numPoint = 0; numPoint < pointData.length; numPoint++) {
              //eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + "[" + numPoint + "] = " + pointData[numPoint]);
              eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + ".push(" + pointData[numPoint] + ")");
            }
          }
          else if (subData[0].toLowerCase() === "halign" || subData[0].toLowerCase() === "valign" || subData[0].toLowerCase() === "imagename" || subData[0].toLowerCase() === "string") {
            eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " = \"" + subData[1] + "\"");
          }
          else if (subData[0].toLowerCase() === "fill") { /* Since fill is a function the name fill was causing issues */
            /* Default to unfill by setting to background color */
            eval("drawEntities." + currentEntity + "[" + idx + "].fillColor = \"" + backcolor + "\"");
            if (subData[1] !== undefined) {
              var curColor = subData[1];
              /* Convert to hex and put in proper format */
              if (!isNaN(curColor) || curColor.indexOf('#') == -1) {
                curColor = translateColor(parseInt(curColor));
              }
              eval("drawEntities." + currentEntity + "[" + idx + "].fillColor = \"" + curColor + "\"");
            }
          }
          else if (subData[0].toLowerCase() === "color") { /* Since color can be multiple types it was causing issues */
            /* Default to forecolor color */
            eval("drawEntities." + currentEntity + "[" + idx + "].color = \"" + forecolor + "\"");
            if (subData[1] !== undefined) {
              var curColor = subData[1];
              /* Convert to hex and put in proper format */
              if (!isNaN(curColor) || curColor.indexOf('#') == -1) {
                curColor = translateColor(parseInt(curColor));
              }
              eval("drawEntities." + currentEntity + "[" + idx + "].color = \"" + curColor + "\"");
            }
          }
          else if (colorTable[subData[0].toLowerCase()]) { /* If the color just comes in as a name */
            eval("drawEntities." + currentEntity + "[" + idx + "].color = \"" + colorTable[subData[0].toLowerCase()] + "\"");
          }
          /* font */
          else if (subData[0].toLowerCase() === "font") {
            /* Sometimes there was junk after the size, remove junk after number */
            subData[1] = subData[1].replace(/[^\d]+$/,'')
            if (subData[1].substring(0, 1) == '+' || subData[1].substring(0, 1) == '-') {
              if (eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " === undefined")) {
                eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " = 0");
              }
              eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " " + subData[1].substring(0, 1) + "= " + subData[1].substring(1, subData[1].length));
            }
            else {
              eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " = " + subData[1]);
            }
          }
          else {
              eval("drawEntities." + currentEntity + "[" + idx + "]." + subData[0].toLowerCase() + " = " + subData[1]);
          }
        }
      }
      else {
        /* Do not know command */
      }
    }
  }

  draw2d_DrawEntities();

}