/*
 *  This material is the joint property of FANUC America Corporation and
 *  FANUC LTD Japan, and must be returned to either FANUC America Corporation
 *  or FANUC LTD Japan immediately upon request.  This material and
 *  the information illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without the
 *  prior written consent of both FANUC America Corporation and FANUC LTD Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2017
 *           FANUC America Corporation
 *           FANUC LTD Japan
 *  +
 *  Module: chart.js
 *  
 *  Description:
 *    Chart plugin
 *
 *  Author: Michael Dow
 *          FANUC America Corporation
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  24-AUG-2017 DOWMW    pr50376 - Initial version
 *  18-SEP-2017 EVANSJA  pr50432 - Move common javascript to util.js
 *  25-SEP-2017 DOWMW    pr50376a - Added SYSVAR, PROGRAM and IO events
 *  15-FEB-2018 EVANSJA  pr50687b - Support for chart on dialog boxes.
 *  03-MAY-2019 DOWMW    pr50376b - Commonized pipes, names now support pipes, horizontal bar chart
 *  03-MAY-2019 DOWMW    pr50376c - Bug fixes, data value, missing features added
 *  24-MAY-2019 DOWMW    v930pr50376b - Change logic for End of Transmission to be Key Release event
 *  20-JUN-2019 DOWMW    pr51640 - Updated canvas name and fix color issues
 *  25-JUN-2020 DOWMW    pr51640b - Fixed multiple charts on same page.
 *  -
*/

var canvasIDs = [];
var current_canvas = "canvas";

/* Default Chart Data */
var ChartConfig;
var ChartConfig_array = [];
var ChartConfig_Default = {
  type: "",
  data: {
    labels: [],
    datasets: [],
    lineAtIndex: [0, 0, 0, 0],
    lineAtIndexC: ['#ff0000', '#ff0000', '#ff0000', '#ff0000']
  },
  options: {
    title: {},
    scales: {
      xAxes: [{
        ticks: {
          //maxTicksLimit: 15
          autoSkip: true,
          autoSkipPadding: 7
        },
        scaleLabel: {
          display: false,
          labelString: ""
        }
      }],
      yAxes: [{
        ticks: {
          //suggestedMax: 0,
          //suggestedMin: 0,
          //max: 0,
          //min: 0,
          autoSkip: true,
          autoSkipPadding: 7
        }
      }]
    },
    legend: {
      labels: {
        filter: ""
      },
    },
    elements: {
      point:{
        radius: 2 //Points size on line chart
      }
    },
    zoom: {
      enabled: true,
      mode: 'y',
    },
    pan: {
      enabled: true,
      mode: 'xy',
      rangeMin: {
        x: 0,
      },
      rangeMax: {
        x: 20
      }
    },
    responsive: true,
    maintainAspectRatio: false
  },
  combinedBuffer: "",
  xStepSize: 1,
  thisChart: undefined
};

var ChartSource = {
  chNum: [],
  varType: [],
  sourceType: [],
  sourceName: []
};

var ChartName = {
  chNum: [],
  varType: [],
  sourceType: [],
  sourceName: []
};

var maxNumCh = 8;
var maxScaleX;
var dataRate = 0;
var lastEventTime = 0;
var maxValue = 0;
var minValue = 0;
var bindEvents = false;
var barHorizontal = false;
var showValues = false;
var paused = false;
var timeoutVar;

var IOmap = {
  "ALL"   : "0",
  "DIN"   : "1",
  "DOUT"  : "2",
  "ANIN"  : "3",
  "ANOUT" : "4",
  "TOOL"  : "5",
  "PLCIN" : "6",
  "PLCOUT": "7",
  "RDI"   : "8",
  "RDO"   : "9",
  "BRAKE" : "10",
  "SOPIN" : "11",
  "SOPOUT": "12",
  "ESTOP" : "13",
  "TPIN"  : "14",
  "TPOUT" : "15",
  "WDI"   : "16",
  "WDO"   : "17",
  "GPIN"  : "18",
  "GPOUT" : "19",
  "UOPIN" : "20",
  "UOPOUT": "21",
  "LDIN"  : "22",
  "LDOUT" : "23",
  "LANIN" : "24",
  "LANOUT": "25",
  "WSIN"  : "26",
  "WSOUT" : "27",
  "F"     : "35",
  "M"     : "36"
};

/* chNum is 1 Indexed */
/* chIdx is 0 Indexed */

// Private functions 
// Create control only when PMON Server is ready
function chart_ServerReady(data) {
  // Initialize control data.
  chart_InitCtlData(data);

  // Create control.
  chart_CreateCtl(data);

  // Initialize events.
  chart_InitRobotEvent(true, data);

} // chart_ServerReady

// New instance of control.
function chart_NPP_New(data) {

  if (undefined != data.PipeMonRate) {
    data.PipeMonRate = Number(data.PipeMonRate);
  }
  data.DataType = parseInt(data.DataType);

  SetCommonParams(data);
  if (data.IsDialog) {
    chart_ServerReady(data);
  }
  else { 
    top.rpcmc_getTaskIdx(data.fDeviceId, function(taskIdx) { 
      data.fTaskIdx = taskIdx;

      // Complete after top.rpcmc_getTaskIdx completes.
      chart_ServerReady(data);
    });
  }

} // chart_NPP_New

// Destroy instance of control.
function chart_NPP_Destroy(data) {

  // Uninitialize events.
  chart_InitRobotEvent(false, data);

  // Delete control.
  chart_DeleteCtl(data);

} // chart_NPP_Destroy

// Private functions

// Prepare all the necessary data to access the value from the controller.
/* TODO: THIS DOES NOT SEEM TO DO ANYTHING - SHOULD IT BE REMOVED???? */
function chart_PrepareCtl(p_DataType, p_DataIndex, data) {

  switch (p_DataType) {   
    case CGRSTYPE_REG: /* *NUMREG* */
      data.prognam = NUMREG_C;
      data.progvar = NUM_REGISTER_C + "[" + p_DataIndex + "]";
      break;
    case CGRSTYPE_SYSVAR: /* *SYSTEM* */
      data.prognam = SYSNAME_C;
      data.progvar = p_DataIndex;
      break;
    
  }
}

// Set the border.
function chart_SetBorder(data) {

  var border = parseInt(data.Border);
  if (border > 0) {
    data.$this.css({"border-style":"solid", "border-width":"1px", "border-color":"gray"});
  }
  else {
    data.$this.css("border-style", "none");
  }

} // chart_SetBorder

// Update Control.
function chart_UpdateCtl(data) {


} // chart_UpdateCtl

// Initialize or uninitialize pipe event.
function chart_InitPipeEvent(init, data) {

  /* Get pipe data rate - Default 1sec */
  if (data.PipeMonRate === undefined || data.PipeMonRate < 0) {
    data.PipeMonRate = 1000;
  }

  // Notify PMON to start/stop monitoring our pipe
  if (init) {
    top.jQuery.filelis.bind('PipeEvent', data, chart_HandlePipeEvent); // Attach handler for PipeEvent
    top.rpcmc_startPipeMonitor(data.Pipe, data.PipeMonRate); // Start PMON monitor for our pipe
    // Attach handler for KeyReleasedEvent - This is used to listen to end of pipe transmission
    top.jQuery.keylis.bind('KeyReleasedEvent', chart_HandleKeyEvent);

  }
  else {
    top.rpcmc_stopPipeMonitor(data.Pipe); // Stop PMON monitor for our pipe
    top.jQuery.filelis.unbind('PipeEvent', chart_HandlePipeEvent); // Detach handler for PipeEvent.
    // Detach handler for KeyReleasedEvent.
    top.jQuery.keylis.unbind('KeyReleasedEvent', chart_HandleKeyEvent);
  }
  
} // chart_InitPipeEvent

function shadeColor1(color, percent) {  // deprecated. See below.
  var num = parseInt(color.slice(1), 16), amt = Math.round(2.55 * percent), R = (num >> 16) + amt, G = (num >> 8 & 0x00FF) + amt, B = (num & 0x0000FF) + amt;
  return "#" + (0x1000000 + (R < 255 ? R < 1 ? 0 : R : 255) * 0x10000 + (G < 255 ? G < 1 ? 0 : G : 255) * 0x100 + (B < 255 ? B < 1 ? 0 : B : 255)).toString(16).slice(1);
}

function shadeColor2(color, percent) {
  var f = parseInt(color.slice(1), 16), t = percent < 0 ? 0 : 255, p = percent < 0 ? percent * -1 : percent, R = f >> 16, G = f >> 8 & 0x00FF, B = f & 0x0000FF;
  return "#" + (0x1000000 + (Math.round((t - R) * p) + R) * 0x10000 + (Math.round((t - G) * p) + G) * 0x100 + (Math.round((t - B) * p) + B)).toString(16).slice(1);
}

// Replace any indirection with actual value.
function chart_IndirectReplace(data) {

  var l_ind;
  var pos;

 $.each(data, function( argn, argv ) { 
    if (typeof argv !== 'string') {
      return;
    }
    // Value contain !PaneId?
    if ((pos = argv.toLowerCase().indexOf("!paneid")) >= 0) {
      argv = argv.substring(0, pos) + data.fTaskIdx + argv.substring(pos+7);
      data[argn] = argv;
    }
  });
} // chart_IndirectReplace

// Initialize Control Data.
function chart_InitCtlData(data) {

  // Process parameters.
  // Name not supported
  // Border
  // Pipe
  // Scene
  // SubPane
  // Verbose not supported
  // ExecConnectId not required (only for ActiveX Controls)

  data.Pipe = data.Pipe.toUpperCase();
  chart_IndirectReplace(data);

  data.SubPane = parseInt(data.SubPane);

  if (data.BackColor == "") {
    data.BackColor = data.InvisibleColor;
  }
  else {
    data.BackColor = translateColor(data.BackColor);
  }
  if (data.ForeColor == "") {
    data.ForeColor = data.TextColor;
  }
  else {
    data.ForeColor = translateColor(data.ForeColor);
  }

} // chart_InitCtlData

/* This is used to set up PMON for variable data */
function chart_SetupSource(chNum, tempName) {
  var enableUpdates = false;
  var sourceType;
  var sourceName
  
  /* Remove Quotes */
  tempName = tempName.replace(/"/g, '');

  sourceType = '';
  sourceName  = '';
  varEvent = false;

  /* Check if *NUMREG* */
  if (tempName.indexOf('numreg') >= 0) {
    varEvent = true;
    sourceType = NUMREG_C;
    sourceName  = "$" + tempName.toUpperCase();
  }
  /* Check if SYSVAR */
  else if (tempName[0] == '$') {
    varEvent = true;
    sourceType = SYSNAME_C;
    sourceName  = tempName.toUpperCase();
  }
  /* Check if PROGRAM Variable */
  else if (tempName[0] == '[') {
    varEvent = true;
    sourceType = tempName.substr(0, tempName.indexOf(']') + 1);
    sourceName  = tempName.substr(tempName.indexOf(']') + 1);
  }
  /* Assume everything else is IO */
  else if (tempName.length > 0) {
    varEvent = false;
    sourceType = tempName.substr(0, tempName.indexOf('['));
    if (IOmap[sourceType.toUpperCase()]) {
      sourceType = IOmap[sourceType.toUpperCase()];
      sourceName = tempName.substr(tempName.indexOf('[') + 1, (tempName.indexOf(']') - tempName.indexOf('[')) - 1);
    }
  }
  else {
    /* DO NOTHING */
  }
  
  ChartSource.chNum.push(chNum);
  ChartSource.varType.push(varEvent);
  ChartSource.sourceType.push(sourceType.toUpperCase());
  ChartSource.sourceName.push(sourceName.toUpperCase());

  if (sourceType && sourceName) {
    enableUpdates = true;
    if(varEvent) {
      top.rpcmc_startVarMonitor(sourceType.toUpperCase(), sourceName.toUpperCase(), dataRate); // Start PMON monitors for vars.
    }
    else {
      top.rpcmc_startIOMonitor(sourceType.toUpperCase(), sourceName.toUpperCase()); // Start PMON monitor for the I/O port.
    }
  }

  /* if it is a Var Event we need to enable updating the chart based on the data rate */
  if (bindEvents == false) {
    bindEvents = true;
    /* Only perform the binds and unbinds once */
    top.jQuery.varlis.bind('VarEvent', null, chart_HandleVarEvent); // Attach handler for VarEvent.
    top.jQuery.iolis.bind('IOBooleanEvent', null, chart_HandleIOEvent); // Attach handler for IOBooleanEvent.
    
    var axis;
    if(barHorizontal) {
      axis = ChartConfig.options.scales.yAxes[0];
    }
    else {
      axis = ChartConfig.options.scales.xAxes[0];
    }
    axis.scaleLabel.display = true;
    axis.scaleLabel.labelString = "Time (" + dataRate + " ms Intervals)";

    updateChart();
  }

}

/* This is used to set up PMON for variable data */
function chart_StopSource() {
  var sourceType;
  var sourceName;
  var varType;
  
  for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
    sourceType = ChartSource.sourceType[chNum-1];
    sourceName = ChartSource.sourceName[chNum-1];
    varType = ChartSource.varType[chNum-1];

    if (sourceType && sourceName) {
      if(varType) {
        top.rpcmc_stopVarMonitor(sourceType.toUpperCase(), sourceName.toUpperCase()); // Stop PMON monitors for vars.
      }
      else {
        top.rpcmc_stopIOMonitor(sourceType.toUpperCase(), sourceName.toUpperCase()); // Stop PMON monitor for the I/O port.
      }
    }
  }
  
  /* Clear ChartSource */
  ChartSource = {
    chNum: [],
    varType: [],
    progName: [],
    varName: []
  };

  /* Only perform the binds and unbinds once */
  if (bindEvents == true) {
    bindEvents = false;
    top.jQuery.varlis.unbind('VarEvent', chart_HandleVarEvent); // Detach handler for VarEvent.
    top.jQuery.iolis.unbind('IOBooleanEvent', chart_HandleIOEvent); // Detach handler for IOBooleanEvent.
  }
}

/* This is used to set up PMON for variable data */
function chart_SetupName(chNum, tempName) {
  var sourceType;
  var sourceName
  
  /* Remove Quotes */
  tempName = tempName.replace(/"/g, '');

  sourceType = '';
  sourceName  = '';

  if (tempName[0] == '$') {
    sourceType = SYSNAME_C;
    sourceName  = tempName.toUpperCase();
  }
  /* Check if PROGRAM Variable */
  else if (tempName[0] == '[') {
    sourceType = tempName.substr(0, tempName.indexOf(']') + 1);
    sourceName  = tempName.substr(tempName.indexOf(']') + 1);
  }
  else {
    ChartConfig.data.datasets[chNum - 1].label = tempName;
    return;
  }
  
  ChartName.chNum.push(chNum);
  ChartName.sourceType.push(sourceType.toUpperCase());
  ChartName.sourceName.push(sourceName.toUpperCase());

  if (sourceType && sourceName) {
      top.rpcmc_startVarMonitor(sourceType.toUpperCase(), sourceName.toUpperCase(), dataRate); // Start PMON monitors for vars.
  }

  /* if it is a Var Event we need to enable updating the chart based on the data rate */
  if (bindEvents == false) {
    bindEvents = true;
    /* Only perform the binds and unbinds once */
    top.jQuery.varlis.bind('VarEvent', null, chart_HandleVarEvent); // Attach handler for VarEvent.
    top.jQuery.iolis.bind('IOBooleanEvent', null, chart_HandleIOEvent); // Attach handler for IOBooleanEvent.
    
    var axis;
    if(barHorizontal) {
      axis = ChartConfig.options.scales.yAxes[0];
    }
    else {
      axis = ChartConfig.options.scales.xAxes[0];
    }
    axis.scaleLabel.display = true;
    axis.scaleLabel.labelString = "Time (" + dataRate + " ms Intervals)";

    updateChart();
  }

}

/* This is used to set up PMON for variable data */
function chart_StopName() {
  var sourceType;
  var sourceName;
  
  for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
    sourceType = ChartName.sourceType[chNum-1];
    sourceName = ChartName.sourceName[chNum-1];

    if (sourceType && sourceName) {
        top.rpcmc_stopVarMonitor(sourceType.toUpperCase(), sourceName.toUpperCase()); // Stop PMON monitors for vars.
    }
  }
  
  /* Clear ChartName */
  ChartName = {
    chNum: [],
    progName: [],
    varName: []
  };

  /* Only perform the binds and unbinds once */
  if (bindEvents == true) {
    bindEvents = false;
    top.jQuery.varlis.unbind('VarEvent', chart_HandleVarEvent); // Detach handler for VarEvent.
    top.jQuery.iolis.unbind('IOBooleanEvent', chart_HandleIOEvent); // Detach handler for IOBooleanEvent.
  }
}

// Initialize or uninitialize events for each type.
function chart_InitRobotEvent(init, data) {
  var enableUpdates = false;
  var varEvent      = false;
  var ioEvent       = false;
  var progName;
  var varName

  // Start/stop the Pipe Event.
  if (data.Pipe != "") {
    chart_InitPipeEvent(init, data);
  }

  /* Get the loweset data rate otherwise default to 100ms */
  for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
    var tempName;
    /* ch#_Rate */
    eval("tempName = data.ch" + chNum + "_Rate");
    if (dataRate == 0 || dataRate > parseInt(tempName)) {
      dataRate = parseInt(tempName);
    }
  }
  if (dataRate < 100) {
    dataRate = 100;
  }
  
  if(init) {
    for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
      var tempName;
      /* ch#_Source */
      eval("tempName = data.ch" + chNum + "_Source");
      if(tempName) {
        tempName = tempName.toLowerCase();
        chart_SetupSource(chNum, tempName)
      }
    }
  }
  else{
    chart_StopSource();
  }
} // chart_InitRobotEvent

// Create Control.
function chart_CreateCtl(data) {
  
  /* TODO: THIS DOES NOT SEEM TO DO ANYTHING - SHOULD IT BE REMOVED???? */
  if (data.DataIndex != "") {
    // Prepare all the necessary data to access DataType/DataIndex from the controller.
    chart_PrepareCtl(data.DataType, data.DataIndex, data);
  }

  data.$this.css("display", "inline-block");
  if (data.width.indexOf("%") >= 0) {
    data.$this.css("width", data.width);
  }
  else {
    data.$this.css("width", data.width + "px");
  }
  if (data.height.indexOf("%") >= 0) {
    data.$this.css("height", data.height);
  }
  else {
    data.$this.css("height", data.height + "px");
  }
  data.$this.css("vertical-align", "top");
  chart_SetBorder(data);
  SetColors(data);
  SetFont(data);

  setupChartDefaults(data);
  chart_InitChart(data);

  // Attach handler for click event.
  data.$this.bind("click", data, chart_HandleClickEvent);

} // chart_CreateCtl

// Handle Control events.
function chart_CtlEvent(data) {

  var sb_state;

  if (((data.ReadOnly != "") && (data.ReadOnly != "0")) ||
      (data.IsEcho)) {
    return;  
  }
  if (data.dlgDismiss == "1") {
    // Dismiss dialog box.
    top.rpcmc_sendKeyCode(tk_cancel_c);
  }

} // chart_CtlEvent

// Delete Control Data.
function chart_DeleteCtl(data) {

} // chart_DeleteCtl

// Private functions

/* This is used to put PIPE data in the correct format */
function chart_FormatDataStructure(dataName) {

  /* Single command */
  if(!(Array.isArray(dataName))) {
    for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
      var tempName;
      /* ch#_clear */
      eval("tempName = \"ch" + chNum + "_clear\"");
      if (dataName.toLowerCase() === tempName.toLowerCase()) {
        chart_ClearChannelData(chNum);
        return;
      }
      if (dataName.toLowerCase() === "pause") {
        paused = true;
        clearTimeout(timeoutVar);
        return;
      }
      if (dataName.toLowerCase() === "resume") {
        paused = false;
        updateChart();
        return;
      }
      if (dataName.toLowerCase() === "chartclear") {
        ChartConfig.data.labels = [0];
        chart_ClearChannelData(chNum);
      }
    }
    return;
  }

  /* Name Value pair calls */
  if(dataName[0]) {
    /* Remove quotes from data - This was causing issues */
    dataName[1] = dataName[1].replace(/"/g, '');
    if(dataName[0].toLowerCase() === "caption") {
      ChartConfig.options.title.text = dataName[1].replace(/['"]+/g, '');;
      ChartConfig.options.title.display = true;
      return;
    }
    if (dataName[0].toLowerCase() === "linescaleactive") {
      chart_setYMaxMin(dataName[1]);
      return;
    }
    if (dataName[0].toLowerCase() === "datashowvalues") {
      showValues = !!parseInt(dataName[1]);
      return;
    }
    if (dataName[0].toLowerCase() === "samplescale") {
      var subData = dataName[1];
      subData = subData.trim();
      subData = subData.split(',');
      ChartConfig.data.labels = [parseInt(subData[0])];
      maxScaleX = parseInt(subData[1]) - parseInt(subData[0]);
      return;
    }
    if (dataName[0].toLowerCase() === "samplescaleaspect") {
      ChartConfig.xStepSize = parseFloat(dataName[1]);
    }
    if (dataName[0].toLowerCase() === "datascale") {
      var axis;
      if(barHorizontal) {
        axis = ChartConfig.options.scales.xAxes[0];
      }
      else {
        axis = ChartConfig.options.scales.yAxes[0];
      }
      var subData = dataName[1];
      subData = subData.trim();
      subData = subData.split(',');
      axis.ticks.min = parseInt(subData[0]);
      axis.ticks.max = parseInt(subData[1]);
      return;
    }
    
    /* Channel specific settings */
    for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
      var tempName;
      /* Sometimes chart data was sent before a chart was initilized */
      if (typeof ChartConfig.data.datasets[chNum - 1] === 'undefined') {
        continue;
      }
        
      /* ch#_Name */
      eval("tempName = \"ch" + chNum + "_name\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        chart_SetupName(chNum, dataName[1])
        return;
      }
      /* ch#_Color */
      eval("tempName = \"ch" + chNum + "_color\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        if (dataName[1] !== undefined) {
          var curColor = dataName[1];
          /* Convert to hex and put in proper format */
          if(curColor.indexOf('#') == -1) {
            curColor = translateColor(parseInt(curColor));
          }
          ChartConfig.data.datasets[chNum - 1].backgroundColor = curColor;
          ChartConfig.data.datasets[chNum - 1].borderColor = curColor;
          ChartConfig.data.datasets[chNum - 1].lineAtValueC = [curColor, curColor, curColor, curColor];
        }
        return;
      }
      /* ch#_State */
      eval("tempName = \"ch" + chNum + "_state\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        if(parseInt(dataName[1]) == true) {
          chart_ActivateChannel(chNum);
        }
        else {
          chart_DeactivateChannel(chNum);
        }
        return;
      }
      /* ch#_Data */
      eval("tempName = \"ch" + chNum + "_data\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        chart_PopulateData(chNum, false, dataName[1], null)
        return;
      }
      /* ch#_Digital */
      eval("tempName = \"ch" + chNum + "_digital\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        ChartConfig.data.datasets[chNum - 1].steppedLine = Boolean(parseInt(dataName[1]));
        return;
      }
      /* ch#_Source */
      eval("tempName = \"ch" + chNum + "_source\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        chart_SetupSource(chNum, dataName[1])
        return;
      }
      /* Interleaved data overrides ch#_Data - Re-Populate Data */
      eval("tempName = \"ch_Data_" + chNum + "\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        chart_PopulateData(chNum, true, dataName[1], null)
        return;
      }
    }
    for (var markIdx = 1; markIdx <= 4; ++markIdx) {
      var tempName;
      /* SampleMarker# */
      eval("tempName = \"SampleMarker" + markIdx + "\"");
      if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
        chart_CreateVertLineMarkers(markIdx, dataName[1])
        return;
      }
      for (var chNum = 1; chNum <= maxNumCh; ++chNum) {
        var tempName;
        /* ch#_DataMarker# */
        eval("tempName = \"ch" + chNum + "_DataMarker" + markIdx + "\"");
        if (dataName[0].toLowerCase() === tempName.toLowerCase()) {
          chart_CreateHorizLineMarkers(chNum, markIdx, dataName[1])
          return;
        }
      }
    }
  }
}

function chart_HandleKeyEvent(event, key) {
  // These keys came from the controller.
  /* Indicates the pipe transmission is complete */
  if (key == 4) {
    if (ChartConfig.combinedBuffer.length > 0) {
      /* Replace single quote with double quote */
      ChartConfig.combinedBuffer = ChartConfig.combinedBuffer.replace(/'/g, '"');
      /* remove whitespace before and after equal sign */
      ChartConfig.combinedBuffer = ChartConfig.combinedBuffer.replace(/\s*=\s*/g, '=');
      var lineData = ChartConfig.combinedBuffer.match(/(?:[^\s"]+|"[^"]*")+/g)
        for(var i = 0; i < lineData.length; i++) {
        if(lineData[i]) {
          var tempLineData = lineData[i].trim();
          if (tempLineData.indexOf('=') > 0) {
            var subData = tempLineData.split('=');
            chart_FormatDataStructure(subData);
          }
          else {
            chart_FormatDataStructure(lineData[i]);
          }
        }
      }
      ChartConfig.combinedBuffer = '';
      chart_CreateLabels();
      ChartConfig.thisChart.update();
    }
  }
}

function chart_HandlePipeEvent(event, file, buffer) {
  event.preventDefault();
  var data = event.data || event;
  
  if (file == data.Pipe) {
    if(data.Id) {
      current_canvas = data.Id;
      for (var idx = 0; idx < canvasIDs.length; idx++) {
        if(canvasIDs[idx] == data.Id) {
          ChartConfig = ChartConfig_array[idx];
          current_canvas = canvasIDs[idx];
        }
      }
    }
    if (buffer.length > 0) {
      ChartConfig.combinedBuffer = ChartConfig.combinedBuffer.concat(buffer);
    }
  }
  
  return true;
} // chart_HandlePipeEvent

/* This handles data that is coming through as a VAR */
/* Current assumption is only one value comes through at once */
function chart_HandleVarEvent(event, prog_name, var_name, type_code, val_str) {
  /* Remove Quotes */
  var_name = var_name.replace(/"/g, '');
  /* Loop over all ChartSource to find the right channel */
  for (var idx = 0; idx < ChartSource.chNum.length; ++idx) {
    if (prog_name.toUpperCase() == ChartSource.sourceType[idx] && var_name.toUpperCase() == ChartSource.sourceName[idx]) {
      chart_PopulateData(ChartSource.chNum[idx], false, null, parseFloat(val_str));
      /* TODO: What if the var in the same for two different channels? */
      /* break; */
    }
  }
  for (var idx = 0; idx < ChartName.chNum.length; ++idx) {
    if (prog_name.toUpperCase() == ChartName.sourceType[idx] && var_name.toUpperCase() == ChartName.sourceName[idx]) {
      ChartConfig.data.datasets[ChartSource.chNum[idx] - 1].label = val_str;
      /* TODO: What if the var in the same for two different channels? */
      /* break; */
    }
  }
  return true;
} // chart_HandleVarEvent

/* This handles data that is coming through as a IO */
function chart_HandleIOEvent(event, io_type, io_index, io_value) {
  /* Loop over all ChartSource to find the right channel */
  for (var idx = 0; idx < ChartSource.chNum.length; ++idx) {
    if (io_type.toUpperCase() == ChartSource.sourceType[idx] && io_index == ChartSource.sourceName[idx]) {
      chart_PopulateData(ChartSource.chNum[idx], false, null, parseFloat(io_value));
      /* TODO: What if the var in the same for two different channels? */
      /* break; */
    }
  }
  return true;
} // chart_HandleIOEvent

function chart_HandleClickEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  chart_CtlEvent(data);
  return true;
} // chart_HandleClickEvent

function createChart() {
  var ctx = document.getElementById(current_canvas).getContext("2d");
  
  /* Need to remove the old chart if making a new one */
  if (ChartConfig.thisChart !== undefined)
    ChartConfig.thisChart.destroy();
  ChartConfig.thisChart = new Chart(ctx, ChartConfig);
  ChartConfig.thisChart.update();
}

function randomColor() {
  return ('#'+(Math.random()*0xFFFFFF<<0).toString(16));
}

function setupChartDefaults(data) {
  var chartGlobals = Chart.defaults.global;

  /* Line Defaults */
  /* Curvature of the line Typically: 0 - No Curve or .4 - Slight Cureve*/
  chartGlobals.elements.line.tension = 0;

  /* Animation time */
  chartGlobals.animation.duration = 0;
  
  /* Fonts */
  chartGlobals.defaultFontColor  = '#000000'
  chartGlobals.defaultFontFamily = data.FontName;
  chartGlobals.defaultFontSize   = parseInt(data.FontSize);
  chartGlobals.defaultFontStyle  = 'normal';
  if (parseInt(data.FontBold) > 0) {
    chartGlobals.defaultFontStyle  = 'bold';
  }
  else if (parseInt(data.FontItalic) > 0) {
    chartGlobals.defaultFontStyle  = 'italic';
  }
  else {
    chartGlobals.defaultFontStyle  = 'normal';
  }    

  /* Ledgend */
  chartGlobals.legend.position = 'left';
  chartGlobals.legend.labels.usePointStyle = true; /* Label style will match corresponding point style */
}

function chart_DefaultChannel(data, chNum) {
  /* NOTE: chNum starts at 1, not 0 */

  /* setup default dataset if active channel*/
  eval("tempName = data.ch" + chNum + "_Name");
  var rColor = randomColor();
  ChartConfig.data.datasets[chNum - 1] = {
    label: tempName,
    backgroundColor: rColor,
    borderColor: rColor,
    //borderWidth: 3,
    fillColor: rColor,
    data: [],
    hidden: false,
    legendHidden: false,
    fill: false,
    spanGaps: true,
    steppedLine: false,
    lineAtValue: [0, 0, 0, 0],
    lineAtValueC: [rColor, rColor, rColor, rColor],
    scaleY: "",
    autoScaleY: false,
  };

  /* Populate Color of Data */
  eval("tempVar = data.ch" + chNum + "_Color");
  /* Color of the Data */
  if (tempVar !== undefined) {
    /* Convert to hex and put in proper format */
    var curColor = translateColor(tempVar)
    ChartConfig.data.datasets[chNum - 1].backgroundColor = curColor;
    ChartConfig.data.datasets[chNum - 1].borderColor = curColor;
    ChartConfig.data.datasets[chNum - 1].lineAtValueC = [curColor, curColor, curColor, curColor];
  }
}

function chart_ActivateChannel(chNum) {
  ChartConfig.data.datasets[chNum - 1].hidden = false;
  ChartConfig.data.datasets[chNum - 1].legendHidden = false;
}

function chart_DeactivateChannel(chNum) {
  ChartConfig.data.datasets[chNum - 1].hidden = true;
  ChartConfig.data.datasets[chNum - 1].legendHidden = true;
}

function chart_ClearChannelData(chNum) {
  ChartConfig.data.datasets[chNum - 1].data = [];
}

function chart_PopulateData(chNum, interleaved, dataString, dataPoint) {
  /* NOTE: chNum starts at 1, not 0 */
  
  /* if data is paused, don't update dataset */
  if(paused) {
    return;
  }

  /* is data in string format */
  if (dataString) {
    /* replace double quote with single */
    dataString = dataString.replace(/['"]+/g, '');
    /* remove whitespace from beginning and end */
    dataString = dataString.trim();
    /* replace non-numbers with comma*/
    dataString = dataString.replace(/[^-0-9e+|.]+/g, ',');
    var array = dataString.split(",").map(Number);
    
    /* Not interleaved */
    if(!interleaved) {
      /* Erase current populated data */
      ChartConfig.data.datasets[chNum - 1].data = [];
      for (var i = 0; i < array.length; i++) {
        ChartConfig.data.datasets[chNum - 1].data.push(array[i]);
      }
    }
    /* Interleaved */
    else {
      for (var i = 0; i < array.length;) {
        for (var j = 0; j < chNum; j++) {
          if (ChartConfig.data.datasets[j]) {
            ChartConfig.data.datasets[j].data.push(array[i]);
          }
          i++;
        }
      }
    }
    
  }
  else if (dataPoint !== null) {
    ChartConfig.data.datasets[chNum - 1].data.push(dataPoint);
  }
  else {
    return
  }

}

function chart_CreateLabels() {

  var maxDataSet = 0; /* This is the length of the channel index with the more data points */

  for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
    if (ChartConfig.data.datasets[chIdx]) {
      if (maxDataSet < ChartConfig.data.datasets[chIdx].data.length) {
        maxDataSet = ChartConfig.data.datasets[chIdx].data.length;
      }
    }
    /* Check if we need to auto scale */
    if(!ChartConfig.data.datasets[chIdx].hidden && ChartConfig.data.datasets[chIdx].autoScaleY) {
      var axis;
      if(barHorizontal) {
        axis = ChartConfig.options.scales.xAxes[0];
      }
      else {
        axis = ChartConfig.options.scales.yAxes[0];
      }
      if(axis.ticks.min !== undefined && axis.ticks.max !== undefined) {
        axis.ticks.suggestedMin = axis.ticks.min;
        axis.ticks.suggestedMax = axis.ticks.max;
        /* input zero is auto scale */
        chart_setYMaxMin(0);
      }
    }
  }

  /* Create Labels */
  /* CURRENTLY JUST DEFAULTING TO 1 INDEX STARTING AT 0*/
  if (maxDataSet && maxDataSet > ChartConfig.data.labels.length) {
    for (; maxDataSet > ChartConfig.data.labels.length || maxScaleX > ChartConfig.data.labels.length;) {
      ChartConfig.data.labels.push(parseFloat((ChartConfig.data.labels[ChartConfig.data.labels.length - 1] + ChartConfig.xStepSize).toPrecision(5)));
    }
  }

  /* Strip off data if larger that what user requested */
  if (maxDataSet >= 0 && maxDataSet > maxScaleX) {
    for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
      if (ChartConfig.data.datasets[chIdx]) {
        for (; maxScaleX < ChartConfig.data.datasets[chIdx].data.length;) {
          ChartConfig.data.datasets[chIdx].data.shift();
        }
      }
    }
  }

  /* Strip off labels if larger that what user requested */
  for (; maxScaleX < ChartConfig.data.labels.length;) {
    ChartConfig.data.labels.shift();
  }
}

function chart_setYMaxMin(axisInput) {

  var axis;
  if(barHorizontal) {
    axis = ChartConfig.options.scales.xAxes[0];
  }
  else {
    axis = ChartConfig.options.scales.yAxes[0];
  }

  if (parseInt(axisInput) > 0) {
    var subData = ChartConfig.data.datasets[axisInput - 1].scaleY;
    subData = subData.trim();
    subData = subData.split(',');
    axis.ticks.min = parseInt(subData[0]);
    axis.ticks.max = parseInt(subData[1]);
    return
  }
  if (parseInt(axisInput) == 0) {
    axis.ticks.min = undefined;
    axis.ticks.max = undefined;
    return
  }
  
  axis.ticks.suggestedMax = 0;
  axis.ticks.suggestedMin = 0;

  /* Max Channels supported is 8 */
  for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
    if(ChartConfig.data.datasets[chIdx]) {
      for (var markIdx = 0; markIdx < 4; ++markIdx) {
        if (axis.ticks.suggestedMax < ChartConfig.data.datasets[chIdx].lineAtValue[markIdx]) {
          axis.ticks.suggestedMax = ChartConfig.data.datasets[chIdx].lineAtValue[markIdx] + 1;
        }
        else if (axis.ticks.suggestedMin > ChartConfig.data.datasets[chIdx].lineAtValue[markIdx]) {
          axis.ticks.suggestedMin = ChartConfig.data.datasets[chIdx].lineAtValue[markIdx] - 1;
        }
      }
    }
  }

}

function chart_CreateHorizLineMarkers(chNum, markIdx, location) {
  /* NOTE: Marker Index starts at 1, not 0 */

  location = location.trim();
  var subData = location.split(',');
  /* Line is active */
  if (parseInt(subData[0]) === 1) {
    ChartConfig.data.datasets[chNum - 1].lineAtValue[markIdx - 1] = parseInt(subData[1]);
    /* Change color if it is sent */
    if (parseInt(subData[2])) {
      ChartConfig.data.datasets[chNum - 1].lineAtValueC[markIdx - 1] = translateColor(parseInt(subData[2]));
    }
  }
    /* Line is not active */
  else {
    ChartConfig.data.datasets[chNum - 1].lineAtValue[markIdx - 1] = 0;
  }

  chart_setYMaxMin();

}

function chart_CreateVertLineMarkers(markIdx, location) {
  /* NOTE: Marker Index starts at 1, not 0 */

  location = location.trim();
  var subData = location.split(',');
  /* Line is active */
  if (parseInt(subData[0]) === 1) {
    ChartConfig.data.lineAtIndex[markIdx - 1] = parseInt(subData[1]);
    /* Change color if it is sent */
    if (parseInt(subData[2])) {
      ChartConfig.data.lineAtIndexC[markIdx - 1] = translateColor(parseInt(subData[2]));
    }
  }
  /* Line is not active */ 
  else {
    ChartConfig.data.lineAtIndex[markIdx-1] = 0;
  }

}

function updateChart() {
  ChartConfig.data.labels.push(ChartConfig.data.labels[ChartConfig.data.labels.length - 1] + 1);

  for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
    if (ChartConfig.data.datasets[chIdx]) {
      for (; ChartConfig.data.labels.length > ChartConfig.data.datasets[chIdx].data.length;) {
        ChartConfig.data.datasets[chIdx].data.push(ChartConfig.data.datasets[chIdx].data[ChartConfig.data.datasets[chIdx].data.length - 1]);
      }
    }
  }

  chart_CreateLabels();

  ChartConfig.thisChart.update();
  timeoutVar = setTimeout(function () { updateChart(); }, dataRate);
}

function chart_InitChart(data) {
  /* Create Canvas for the chart */
  if (data) {
    if(data.Id) {
      canvasIDs.push(data.Id);
      current_canvas = data.Id;
      ChartConfig = JSON.parse(JSON.stringify(ChartConfig_Default));
      ChartConfig.options.legend.labels.filter =
        function (legendItem, data) {
          var dataset = data.datasets[legendItem.datasetIndex];
          return !dataset.legendHidden;
        };
      ChartConfig_array.push(ChartConfig);
    }
    /* Create Canvas for the draw area */
    var out = '<canvas id="' + current_canvas + '"></canvas>';
    data.$this.html(out);
    var ctx = document.getElementById(current_canvas).getContext("2d");
    ctx.canvas.height = data.height;
    ctx.canvas.width = data.width;
  }
  else {
    var ctx = document.getElementById(current_canvas).getContext("2d");
    /* clear the canvas */
    ctx.fillStyle = backcolor;
    ctx.fillRect(0, 0, ctx.canvas.width, ctx.canvas.height);
  }
  
  var numCh = 0; /* Num of Active Channels */
  var interleavedIdx = 0;
  
  /* These are used with the EVAL functions */
  var tempState;
  var tempName;
  var tempVar;
  var tempData;

  /* Set Chart Type */
  if (data.ChartType == 1) {
    ChartConfig.type = "bar";
    if(data.Orientation == 1) {
      ChartConfig.type = "horizontalBar";
      barHorizontal = true;
    }
  }
  else if (data.ChartType == 2) {
    ChartConfig.type = "line";
  }
  
  /* Show data values */
  if (data.DataShowValues == 1) {
    showValues = true;
  }

  /* Setup default label array */
  /* Get the Max amount of data to show on X axis */
  var scale = data.SampleScale;
  scale = scale.trim();
  scale = scale.split(',');
  ChartConfig.data.labels = [parseInt(scale[0])];
  maxScaleX = parseInt(scale[1]) - parseInt(scale[0]);
  
  ChartConfig.xStepSize = data.SampleScaleAspect;
  
  /* Max Channels supported is 8 */
  for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
    var chNum = chIdx + 1;

    /* Check if channel is valid */
    eval("tempName = data.ch" + chNum + "_Name");
    if (undefined !== tempName) {  
      chart_DefaultChannel(data, chNum);

      /* Check if Channel is active */
      eval("tempState = data.ch" + chNum + "_State");
      if (tempState && parseInt(tempState)) {
        chart_ActivateChannel(chNum);
        numCh++;
      }
      else {
        chart_DeactivateChannel(chNum);
      }

      /* Check if Channel line is stepped (Digital) */
      eval("tempState = data.ch" + chNum + "_Digital");
      if (tempState) {
        ChartConfig.data.datasets[chIdx].steppedLine = Boolean(parseInt(tempState));
      }

      /* Check if there are horizontal Line markers */
      for (var markIdx = 1; markIdx <= 4; ++markIdx) {
        eval("tempData = data.ch" + chNum + "_DataMarker" + markIdx);
        if (tempData) {
          chart_CreateHorizLineMarkers(chNum, markIdx, tempData)
        }
      }

      /* Get Y Scale */
      eval("tempData = data.ch" + chNum + "_DataScale");
      ChartConfig.data.datasets[chIdx].scaleY = tempData;
      
      /* Get Auto Y Scale */
      eval("tempData = data.ch" + chNum + "_AutoRange");
      ChartConfig.data.datasets[chIdx].autoScaleY = Boolean(parseInt(tempData));
    }

    /* Populate Data */
    eval("tempData = data.ch" + chNum + "_Data");
    if(tempData) {
      chart_PopulateData(chNum, false, tempData, null)
    }
    
    /* Check if interleaved data exists - to be used later*/
    eval("tempData = data.ch_Data_" + chNum);
    if(tempData) {
      interleavedIdx = chNum;
    }
  }

  /* set Y axis */
  if (data.LineScaleActive) {
    chart_setYMaxMin(data.LineScaleActive);
  }
  
  /* Interleaved data overrides ch#_Data - Re-Populate Data */
  if (interleavedIdx) {
    eval("tempData = data.ch_Data_" + interleavedIdx);
    chart_PopulateData(interleavedIdx, true, tempData, null)
  }
    
  /* Set Chart Title */
  if(data.Caption) {
    ChartConfig.options.title.text = data.Caption;
    ChartConfig.options.title.display = true;
  }

  /* There are only 4 Sample Markers */
  for (var markIdx = 1; markIdx <= 4; ++markIdx) {
    eval("tempData = data.SampleMarker" + markIdx);
    if (tempData) {
      if (data.SampleMarkerColor > 0) {
        ChartConfig.data.lineAtIndexC[markIdx - 1] = translateColor(data.SampleMarkerColor);
      }
      chart_CreateVertLineMarkers(markIdx, tempData)
    }
  }

  chart_CreateLabels();
  
  createChart();
} // chart_InitchartFile


function chart_DrawLines(chart) {
  var ctx = chart.chart.ctx;
  var xaxis = chart.scales['x-axis-0'];
  var yaxis = chart.scales['y-axis-0'];

  /* Draw vertical Line - Uses lineAtIndex*/
  for (var markIdx = 0; markIdx < 4; ++markIdx) {
    var index = chart.config.data.lineAtIndex[markIdx];
    if (index) {
      ctx.save();
      ctx.beginPath();
      ctx.moveTo(xaxis.getPixelForValue(undefined, index), yaxis.top);
      ctx.strokeStyle = chart.config.data.lineAtIndexC[markIdx];
      ctx.lineTo(xaxis.getPixelForValue(undefined, index), yaxis.bottom);
      ctx.stroke();
      ctx.restore();
    }
  }


  /* Draw horizontal Line - Uses lineAtValue*/
  /* Max Channels supported is 8 */
  for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
    if (ChartConfig.data.datasets[chIdx]) {
      for (var markIdx = 0; markIdx < 4; ++markIdx) {
        var value = ChartConfig.data.datasets[chIdx].lineAtValue[markIdx];
        if (value) {
          ctx.save();
          ctx.beginPath();
          ctx.moveTo(xaxis.left, yaxis.getPixelForValue(value));
          ctx.strokeStyle = ChartConfig.data.datasets[chIdx].lineAtValueC[markIdx];
          ctx.lineTo(xaxis.right, yaxis.getPixelForValue(value));
          ctx.stroke();
          ctx.restore();
        }
      }
    }
  }
  
  /* Show latest data value on chart */
  if(showValues && ChartConfig.thisChart !== undefined){
    for (var chIdx = 0; chIdx < maxNumCh; ++chIdx) {
      if (ChartConfig.data.datasets[chIdx] && ChartConfig.data.datasets[chIdx].data.length) {
        var meta = ChartConfig.thisChart.getDatasetMeta(chIdx);
        var dataIdx = ChartConfig.data.datasets[chIdx].data.length-1;
        var data = ChartConfig.data.datasets[chIdx].data[dataIdx];
        var x = meta.data[dataIdx]._model.x;
        var y = meta.data[dataIdx]._model.y;

        if(!isNaN(data) && meta.hidden != true && ChartConfig.data.datasets[chIdx].hidden != true) {
          ctx.save();
          ctx.textAlign = 'center';        
          if(ChartConfig.type == "line") {
            ctx.fillStyle = meta.data[dataIdx]._model.backgroundColor;
            ctx.translate(x - ((data.toString().length)*2), y-8);
          }
          else {
            var center = meta.data[dataIdx].getCenterPoint();
            /* Change text to white if black text on black background - default black for Bar*/
            ctx.fillStyle = "#000000";
            if(meta.data[dataIdx]._model.backgroundColor == "#000000") {
              ctx.fillStyle = "#FFFFFF";
            }
            ctx.translate(center.x, center.y);
            /* rotate text for vertical bar cahrts */
            if(!barHorizontal) {
              ctx.rotate(-0.5 * Math.PI);
            }
            /* Shift text over if too close to axis */
            else if (x - center.x < ((data.toString().length)*3)){
              ctx.translate(((data.toString().length)*3), 0);
            }
          }
          ctx.fillText(data, 0, 0);
          ctx.restore();
        }
      }
    }
  }

}

/* Draw Lines on Line chart */
var originalLineDraw = Chart.controllers.line.prototype.draw;
Chart.helpers.extend(Chart.controllers.line.prototype, {
  draw: function () {
    originalLineDraw.apply(this, arguments);
    var chart = this.chart;
    chart_DrawLines(chart);
  }
});

/* Draw Lines on Bar chart */
var originalBarDraw = Chart.controllers.bar.prototype.draw;
Chart.helpers.extend(Chart.controllers.bar.prototype, {
  draw: function () {
    originalBarDraw.apply(this, arguments);
    var chart = this.chart;
    chart_DrawLines(chart);
  }
});