/*
 *  This material is the joint property of FANUC America Corporation and
 *  FANUC LTD Japan, and must be returned to either FANUC America Corporation
 *  or FANUC LTD Japan immediately upon request.  This material and
 *  the information illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without the
 *  prior written consent of both FANUC America Corporation and FANUC LTD Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2017
 *           FANUC America Corporation
 *           FANUC LTD Japan
 *  +
 *  Module: cgtpmain.js
 *  
 *  Description:
 *    Handle events from the controller
 *
 *  Author: Judy Evans
 *          FANUC America Corporation
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  03-DEC-2017 EVANSJA pr50565 - Remove cgtpfrm to allow iPad to scroll properly.
 *  05-DEC-2017 EVANSJA pr50577 - Support for string and numeric input in icon editor.
 *  19-JAN-2018 GRASSCM pr50667 - Support for notification icon.
 *  30-JAN-2018 EVANSJA pr50687 - Support for cgop on dialog boxes.
 *  26-FEB-2018 EVANSJA pr50747 - Add start mode.
 *  05-MAR-2018 DOWMW   pr50762 - Support for Sub and Alpha Menus
 *  12-APR-2018 EVANSJA pr50799 - Support for WebSockets
 *  26-APR-2018 GRASSCM pr50876 - Support for Sub Menu optional parameters.
 *  02-MAY-2018 EVANSJA pr50891 - Support for multiple panes
 *  11-MAY-2018 DOWMW   pr50913 - Support for Override display (COORD)
 *  27-JUN-2018 GRASSCM pr50999 - Cause and remedy for HTML5 mismatching cache.
 *  14-AUG-2018 GRASSCM pr51068 - Update FixJSON to include removal of CR or LF.
 *  30-APR-2018 EVANSJA pr51508 - Turn off heartbeat checking if direct iPendant connection.
 *  08-MAY-2019 EVANSJA pr51529 - Implement pulldown menus for jog menu.
 *  16-MAY-2019 DOWMW   pr51557 - PMEV_WEBPAGE_C - display files as raw text, XML source in iframe no longer support in browsers
 *  21-MAY-2019 DOWMW   pr51557a - removed pr51557 changes - need to revist and fix
 *  28-MAY-2019 EVANSJA pr51583 - If frame is not found, delay 250 ms and try again.
 *  17-JUN-2019 EVANSJA pr51622 - Support for KAREL prompt boxes.
 *  31-JAN-2020 EVANSJA pr52071 - Reconnect to controller automatically if controller is turned off/on
 *  22-JUN-2020 EVANSJA pr52329 - Fix the deletion of alpha entry
 *  14-JUL-2021 KOETSKR pr53221 - Add left/right and enter keys to yellow pop-ups for usability on touch screen
 *  -
*/
// Safari's iPad/iPhone does not support multiple threads
var _nowait = 0;
//if (navigator.userAgent.indexOf("Safari") >= 0) {
//  _nowait = 1;
//}

var heartbeat_start = 0;
var heartbeat_end = 0;
var get_pkt_seq = 0;
var VALUEMIN_C             = -3.402823564E+38;
var VALUEMAX_C             = 3.402823564E+38;
var ovrd_timer;
var buf = '';
var keepbuf = '';
var buflen = 0;
var curr_ver = "";
var has_tabtp_rdy = true;

// Handle WebSocket.
function handleWebSocket(url) {
  try {
    g_wspmonsock = new WebSocket(url, "comet");
    //top.rpcmc_rprintf('WS Status: [' + g_wspmonsock.readyState + '] Connecting to : ' + url);

    g_wspmonsock.onopen = function() { 
      //top.rpcmc_rprintf('WS Status: [' + g_wspmonsock.readyState + '] Handshake done..');

      // Start the events
      setTimeout(getnextevent, 0);
    }

    g_wspmonsock.onmessage = function(msg) {
      if (!top.g_connect_id) {
        // Connection Closed already
        return;
      }
      var pos;
      if (!buflen) {
        // Start a new buffer
        buf = keepbuf += msg.data;
        keepbuf = '';
        pos = buf.indexOf('{');
        if (pos > 0) {
          // Extract the number of bytes before the start of the buffer
          buflen = Number(buf.substring(0, pos));
          //top.rpcmc_rprintf('buflen ' + buf.substring(0, pos));
          buf = buf.substring(pos);
        }
        else {
          buflen = Number(buf);
          //top.rpcmc_rprintf('buflen ' + buf);
          buf = '';
        }
      }
      else {
        buf += msg.data;
      }
      if (buf.length < buflen) {
        return;
      }
      if (buf.length > buflen) {
        keepbuf = buf.substring(buflen+1);
        buf = buf.substring(0, buflen);
      }
      buflen = 0; // start a new buffer next time
      try {
        var json = jQuery.parseJSON(buf);
      } catch (err) {
        var pos = buf.indexOf('{"FANUC"');
        if (pos >= 0) {
          FixJSON(buf.substr(pos));
        }
        else if (buf != "") {
          top.rpcmc_rprintf('[handleWebSocket] err: ' + buf);
        }
        return;
      }
      if (undefined != json.FANUC.RPC) {
        if (parseInt(json.FANUC.RPC[0].status) != 0) {
          // Display "TPIF-215 iPendant login unsuccessful"
          parent.document.location.href = "/frh/diageg/TPIF.htm#TPIF-215";
          return;
        }
        return;
      }
      if (undefined != json.FANUC.PMEV) {
        HandleEvents(json, null);
      } // if PMEV
    }

    g_wspmonsock.onclose = function() {
      if (!get_pkt_seq) {
        // WebSocket never finished the handshake
        g_wspmonsock = 0;

        // Start the events without WebSockets
        setTimeout(getnextevent, 0);
        return;
      }
      //top.rpcmc_rprintf('WS Status: [' + g_wspmonsock.readyState + '] Session closed..');

      // The WebSocket should not close, immediately go to HOME page
      top.document.location.href = 'http://' + location.hostname;
    }			
  } 
  catch(exception) {
    if (!get_pkt_seq) {
      // WebSocket never finished the handshake
      g_wspmonsock = 0;

      // Start the events without WebSockets
      setTimeout(getnextevent, 0);
      return;
    }
    top.rpcmc_rprintf('WS Error:' + exception);

    // The WebSocket should not error, immediately go to HOME page
    top.document.location.href = 'http://' + location.hostname;
  }
} // handleWebSocket

// Request heartbeat and verify PMON_GET_PKT by sending PMON_VERIFY_PKT
function heartbeat() {
  if (!top.g_connect_id) {
    return;
  }
  top.heartbeat_handle = null;
  heartbeat_start = (new Date()).getTime();
  $.getJSON("/COMET/rpc?func=PMON_VERIFY_PKT&connect_id=" + top.g_connect_id, function(json) {
    try {
      if (undefined != json.FANUC.RPC) {
        if (parseInt(json.FANUC.RPC[0].status) != 0) {
          // PMON_GET_PKT got lost
          top.rpcmc_rprintf('[cgtpmain] recovering from lost PMON_GET_PKT');
          setTimeout(getnextevent, 0);
        }
      }
    } catch (err) { /* nop */ };

    heartbeat_end = (new Date()).getTime();
    if (top.heartbeat_timer > 0) {
      top.heartbeat_handle = setTimeout(heartbeat, top.heartbeat_timer);
    }
  });
}

// Check heartbeat
function check_heartbeat() {
  // Heartbeat may never come back if controller is powered down
  if (heartbeat_start > heartbeat_end && heartbeat_end !== 0) {
    if ((new Date()).getTime() - heartbeat_start > top.heartbeat_timer * 2) {
      top.g_recon_proc = true;
      top.dlg_recon_open();
      waitReconnect();
      return;
    }
  }
  if (top.heartcheck_timer) {
    setTimeout(check_heartbeat, top.heartcheck_timer);
  }
}

function find_frame(curr_win, frame_name) {
  var ret_frame = null;
  var len = curr_win.frames.length;
  
  if (frame_name === "prim") {
    if (top.mainfrm && top.mainfrm.prim) {
      return top.mainfrm.prim;
    }
  }
  else if (frame_name === "dual") {
    if (top.mainfrm && top.mainfrm.dual) {
      return top.mainfrm.dual;
    }
  }
  else if (frame_name === "third") {
    if (top.mainfrm && top.mainfrm.third) {
      return top.mainfrm.third;
    }
  }
  
  for (var idxf = 0; idxf < len; idxf++) {
    if (curr_win.frames[idxf].name == frame_name) {
      return curr_win.frames[idxf];
    }
    if (curr_win.frames[idxf].window) {
      // Recursively find the frame
      ret_frame = find_frame(curr_win.frames[idxf].window, frame_name);
      if (ret_frame) {
        return ret_frame;
      }
    }
  }
  return ret_frame;
}

var find_frame_delay = function(curr_win, frame_name, url_flag, url) {
  var ret_frame = find_frame(curr_win, frame_name);
  if (url_flag) {
    if (ret_frame) {
      try {
        ret_frame.document.location.href = url;
      }
      catch (err) {
        ret_frame.location.href = url;
      }
    }
    else {
      top.rpcmc_rprintf("[cgtpmain] frame not found for loading " + url + " into " + frame_name);
    }
  }
  else {
    if (ret_frame) {
      ret_frame.document.location.reload(true);
    }
    else {
      top.rpcmc_rprintf("[cgtpmain] frame not found for refresh " + frame_name);
    }
  }
};

// Takes the MS decimal color in zsx_color and reverses it (BGR -> RGB).
function translateColor(colorStr) {
  var colorNum = Number(colorStr);
  var colorHexStr = (0x1000000 + colorNum).toString(16).slice(1);
  return '#' + colorHexStr.substr(4, 2) + colorHexStr.substr(2, 2) + colorHexStr.substr(0, 2);

} // translateColor

function display_home(webpage) {
  // Logout from controller
  top.rpcmc_logout();

  if (webpage == undefined) {
    parent.document.location.href = 'http://' + location.hostname;
  }
  else {
    parent.document.location.href = webpage;
  }
} // display_home

function getnextevent() {
  if (!top.g_connect_id) {
    return;
  }
  get_pkt_seq = new Date().getTime();
  if (g_wspmonsock) {
    try {
      g_wspmonsock.send("/COMET/rpc?func=PMON_GET_PKT&connect_id=" + top.g_connect_id + "&nowait=" + _nowait + "&_seq=" + get_pkt_seq);
      // If successful, we only need to request this once.
      return;
    }
    catch(exception) {
      top.rpcmc_rprintf('WS Send Error:' + exception + ', Closing..');
      g_wspmonsock.close();
    }
  }
  $.getJSON("/COMET/rpc?func=PMON_GET_PKT&connect_id=" + top.g_connect_id + "&nowait=" + _nowait + "&_seq=" + get_pkt_seq, function(json) {
    //alert("[EVENT] " + JSON.stringify(json));
    try {
      if (undefined != json.FANUC.RPC) {
        if (parseInt(json.FANUC.RPC[0].status) != 0) {
          // Display "TPIF-215 iPendant login unsuccessful"
          parent.document.location.href = "/frh/diageg/TPIF.htm#TPIF-215";
          return;
        }
        setTimeout(getnextevent, 0);
        return;
      }
      if (undefined != json.FANUC.PMEV) {
        HandleEvents(json, null);
      } // if PMEV
    } catch (err) { /* nop */ };

    getnextevent();
  });

} // getnextevent

// Extract all the XML from the json string
// Search for the keyword XMLbuf
function FixJSON(jsontext) {

  // Remove unwanted space
  //jsontext = jsontext.replace(/(\r|\n)/g, '');

  var newjsontext = '';
  var xmlarray = [];  // array of XML strings
  var xmlstart = jsontext.indexOf('XMLbuf');
  while (xmlstart >= 0) {
    xmlstart += 9;   // skip past XMLbuf":"
    var xmlend = jsontext.indexOf('"}',xmlstart);
    if (xmlend >= 0) {
      // Push the XML into the xmlarray
      xmlarray.push(jsontext.substring(xmlstart, xmlend));
      // Strip out the actual XML
      newjsontext = jsontext.substring(0, xmlstart) + jsontext.substr(xmlend);
      jsontext = newjsontext;
    }
    // Search after current xmlstart
    xmlstart = jsontext.indexOf('XMLbuf', xmlstart);
  }  // while
  if (newjsontext == '') {
    newjsontext = jsontext;
  }
  try {
    var json = jQuery.parseJSON(newjsontext);
    if (undefined != json.FANUC.PMEV) {
      HandleEvents(json, xmlarray);
    } // if PMEV
  } catch (err) {
    FixJSON2(newjsontext, xmlarray);
  }
} // FixJSON

// Extract all the buf from the json string
// Search for the keyword buf
function FixJSON2(jsontext, xmlarray) {
  var newjsontext = '';
  var bufarray = [];  // array of buf strings
  var xmlstart = jsontext.indexOf('buf":"');
  while (xmlstart >= 0) {
    xmlstart += 6;   // skip past buf":"
    var xmlend = jsontext.indexOf('"}',xmlstart);
    if (xmlend >= 0) {
      // Push the bad text into the bufarray
      bufarray.push(jsontext.substring(xmlstart, xmlend));
      // Strip out the actual bad text but add in BADBUF
      newjsontext = jsontext.substring(0, xmlstart) + 'BADBUF' + jsontext.substr(xmlend);
      jsontext = newjsontext;
    }
    // Search after current xmlstart
    xmlstart = jsontext.indexOf('buf', xmlstart);
  }  // while
  if (newjsontext == '') {
    newjsontext = jsontext;
  }
  try {
    var json = jQuery.parseJSON(newjsontext);
    if (undefined != json.FANUC.PMEV) {
      // Now go through and replace BADBUF with actual buf
      var bufidx = 0;
      var buflen = 0;
      if (bufarray) {
        buflen = bufarray.length;
      }
      for (var idx = 0; idx < json.FANUC.PMEV.length; idx++) {
        if (undefined != json.FANUC.PMEV[idx].dlg) {
          var tmpdlg = json.FANUC.PMEV[idx].dlg;
          for (var idx2 = 0; idx2 < tmpdlg.length; idx2++) {
            if (tmpdlg[idx2].buf == 'BADBUF') {
              if (bufidx < buflen) {
                tmpdlg[idx2].buf = bufarray[bufidx++];
              }
            }
          }
        }
      }
      HandleEvents(json, xmlarray);
    } // if PMEV
  } catch (err) { /* nop */ };
} // FixJSON2

function FixJSONPipe(jsontext) {
  var newjsontext = '';
  var xmlarray = [];
  var xmlstart = jsontext.indexOf('XMLbuf');
  while (xmlstart >= 0) {
    xmlstart += 9;
    var xmlend = jsontext.indexOf('"}',xmlstart);
    if (xmlend >= 0) {
      xmlarray.push(jsontext.substring(xmlstart, xmlend));
      newjsontext = jsontext.substring(0, xmlstart) + jsontext.substr(xmlend);
      jsontext = newjsontext;
    }
    xmlstart = jsontext.indexOf('XMLbuf', xmlstart);
  }
  if (newjsontext == '') {
    newjsontext = jsontext;
  }
  var json = jQuery.parseJSON(newjsontext);
  if (undefined != json.FANUC.PMEV) {
    json.FANUC.PMEV = json.FANUC.PMEV.filter(function(pmev){return pmev.pmev_type == PMEV_PIPE_C});
    HandleEvents(json, xmlarray);
  }
}

function HandleEvents(json, xmlarray) {
  var xmlidx = 0;
  var xmllen = 0;
  if (xmlarray) {
    xmllen = xmlarray.length; // for PMEV_PIPE_C
  }
  for (var idx = 0; idx < json.FANUC.PMEV.length; idx++) {
  try {
    switch (parseInt(json.FANUC.PMEV[idx].subsys_code)) {
      case SSC_PMON:
        switch (parseInt(json.FANUC.PMEV[idx].pmev_type)) {
          case PMEV_CONNECT_ID_C:
            // Set the language.
            setLang(json.FANUC.PMEV[idx].lang);

            // Trigger a SetLangEvent on the dict event listeners.
            // For instance, jquery.sta.js needs the event to set the font-family
            // because it was loaded before the connect event.
            top.jQuery.dictlis.trigger("SetLangEvent");
            try {
              if (funckeys.location.href.indexOf('blank') < 0) {
                funckeys.location.reload(true);
              }
              if (tpkeys.location.href.indexOf('blank') < 0) {
                tpkeys.location.reload(true);
              }
              if (top.g_irprog) {
                shiftfwd.location.reload(true);
              }
              numpad.location.reload(true);
            } catch (err) { 
              // iframes may not exist
            }
            top.g_version = json.FANUC.PMEV[idx].version;
            top.g_hostname = json.FANUC.PMEV[idx].hostname;
            if (undefined == top.g_hostname) {
              top.g_hostname = '';
            }
            top.g_focusbg = json.FANUC.PMEV[idx].focusbg;
            if (undefined != top.g_focusbg) {
              top.g_focusbg = translateColor(top.g_focusbg);
            }
            top.g_startmd = parseInt(json.FANUC.PMEV[idx].startmd);
            top.g_virtual = parseInt(json.FANUC.PMEV[idx].virtual);
            finish_login_connected(); // in common.hdr

            // Start the heartbeat
            if (top.heartbeat_timer > 0) {
              top.heartbeat_handle = setTimeout(heartbeat, 10000);
              setTimeout(check_heartbeat, 10000);
            }
            break;

          case PMEV_DISCON_C:
            // Disconnect PMON using RPC.
            top.rpcmc_pmon_disconnect();

            // Display home page
            setTimeout(display_home, 100);
            break;

          case PMEV_WEBPAGE_C:
            // Put url into frame.
            if (json.FANUC.PMEV[idx].frame === "_top") {
              top.document.location.href = json.FANUC.PMEV[idx].url;
              break;
            }
            if (json.FANUC.PMEV[idx].frame === "cghide") {
              top.cghide.document.location.href = json.FANUC.PMEV[idx].url;
              break;
            }

            // Recursively find the frame
            var ret_frame = find_frame(window, json.FANUC.PMEV[idx].frame);
            if (ret_frame) {
              try {
                ret_frame.document.location.href = json.FANUC.PMEV[idx].url;
              }
              catch (err) {
                ret_frame.location.href = json.FANUC.PMEV[idx].url;
              }
              //top.rpcmc_rprintf("[cgtpmain] loading " + json.FANUC.PMEV[idx].url + " into " + json.FANUC.PMEV[idx].frame);
            }
            else {
              // Try again after delay
              setTimeout(find_frame_delay, 250, window, json.FANUC.PMEV[idx].frame, true, json.FANUC.PMEV[idx].url);
            }
            break;

          case PMEV_WEBFRAME_REFRESH_C:
            // Find frame.
            if (json.FANUC.PMEV[idx].frame === "_top") {
              top.document.location.reload(true);
              break;
            }
            if (json.FANUC.PMEV[idx].frame === "cghide") {
              top.cghide.document.location.reload(true);
              break;
            }

            // Recursively find the frame
            var ret_frame = find_frame(window, json.FANUC.PMEV[idx].frame);
            if (ret_frame) {
              ret_frame.document.location.reload(true);
            }
            else {
              // Try again after delay
              setTimeout(find_frame_delay, 250, window, json.FANUC.PMEV[idx].frame, false, "");
            }
            break;

          case PMEV_WEBFRAME_FOCUS_C:
            // Give frame focus.
            var device_id;
            if (json.FANUC.PMEV[idx].frame.indexOf('*') == 0) {
              // Special event
              // jquery.fky.js will also get device event so can ignore
              break;
            }
            // Recursively find the frame
            var ret_frame = find_frame(window, json.FANUC.PMEV[idx].frame);
            if (ret_frame) {
              ret_frame.focus();
            }
            else {
              top.rpcmc_rprintf("[cgtpmain] frame not found for focus " + json.FANUC.PMEV[idx].frame);
            }
            if (('dual' == json.FANUC.PMEV[idx].frame) ||
                ('extstat' == json.FANUC.PMEV[idx].frame)) {
              device_id = 1;
            }
            else if ('third' == json.FANUC.PMEV[idx].frame) {
              device_id = 2;
            }
            else {
              device_id = 0;
            }
            if (device_id != top.g_device_id) {
              top.g_device_id = device_id;

              // Trigger a ChangeFocusEvent on the fky event listeners.
              top.jQuery.fkylis.trigger("ChangeFocusEvent");
            }
            break;

          case PMEV_IO_C:
            // Trigger an IOBooleanEvent on the io event listeners. 
            top.jQuery.iolis.trigger("IOBooleanEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index,
              json.FANUC.PMEV[idx].value]);
            break;

          case PMEV_SAFE_IO_C:
            // Trigger an SafetyIOEvent on the io event listeners. 
            top.jQuery.iolis.trigger("SafetyIOEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index,
              json.FANUC.PMEV[idx].value]);
          break;

          case PMEV_IO_MGTH_C:
            // Trigger an HandleIOEvent on the io event listeners. 
            var hdata = {
              type  : null,
              index : null,
              value : null,
              pos   : {1:{}, 2:{}, 3:{}, 4:{}, 5:{}, 6:{}, 7:{}, 8:{}}
            }
            var keys = ['type', 'index', 'value'];
            for (var j = 1; j <= 8; j++) {
              keys.push('jpos' + j);
              keys.push('pos' + j);
            }
            keys.forEach(function(key) {
              if (key.indexOf('pos') > -1) {
                var g = key.replace(/[^0-9]/g, '');
                var rep = key.indexOf('j') > -1 ? 'jpos' : 'cpos';
                hdata.pos[g][rep] = json.FANUC.PMEV[idx][key] ? json.FANUC.PMEV[idx][key] : null;
              } else {
                hdata[key] = json.FANUC.PMEV[idx][key];
              }
            });
            top.jQuery.iolis.trigger("HandleIOEvent", [hdata]);
            break;

          case PMEV_DEVICE_C:
            if (json.FANUC.PMEV[idx].device == 'TPFK') {
              // Trigger a DeviceEvent on the fky event listeners.
              top.jQuery.fkylis.trigger("DeviceEvent", [
                json.FANUC.PMEV[idx].device,
                json.FANUC.PMEV[idx].line]);
            }
            else if (json.FANUC.PMEV[idx].device == 'STAT') {
              // Trigger a DeviceEvent on the device event listeners.
              top.jQuery.devlis.trigger("DeviceEvent", [
                json.FANUC.PMEV[idx].device,
                json.FANUC.PMEV[idx].line,
                json.FANUC.PMEV[idx].stat_tpprog_flag]);
            }
            else {
              // Trigger a DevEvent2 on the dev event2 listeners.
              // New event is more efficient for TP, TP2, TP3
              top.jQuery.devlis2.trigger("DevEvent2", [
                json.FANUC.PMEV[idx].device,
                json.FANUC.PMEV[idx].line]);
            }
            break;

          case PMEV_FILE_C:
            // Trigger a FileEvent on the file event listeners.  
            top.jQuery.filelis.trigger("FileEvent", [
              json.FANUC.PMEV[idx].file,
              json.FANUC.PMEV[idx].buffer]);
            break;

          case PMEV_PIPE_C:
            // Trigger a PipeEvent on the file event listeners.  
            if (undefined == json.FANUC.PMEV[idx].buffer) {
              if (json.FANUC.PMEV[idx].XMLbuf != "") {
                top.jQuery.filelis.trigger("PipeEvent", [
                  json.FANUC.PMEV[idx].file,
                  json.FANUC.PMEV[idx].XMLbuf]);
              }
              else if (xmlidx < xmllen) {
                top.jQuery.filelis.trigger("PipeEvent", [
                  json.FANUC.PMEV[idx].file,
                  xmlarray[xmlidx++]]);
              }
            }
            else {
              top.jQuery.filelis.trigger("PipeEvent", [
                json.FANUC.PMEV[idx].file,
                json.FANUC.PMEV[idx].buffer]);
            }
            break;

          case PMEV_VAR_C:
            // Trigger a VarEvent on the var event listeners.  
            top.jQuery.varlis.trigger("VarEvent", [
              json.FANUC.PMEV[idx].prog_name,
              json.FANUC.PMEV[idx].var_name,
              json.FANUC.PMEV[idx].type_code,
              json.FANUC.PMEV[idx].value]);
            break;

            case PMEV_CURPOS_JOINT_C:
            case PMEV_CURPOS_USER_C:
            case PMEV_CURPOS_WORLD_C:
            // Trigger a CurposEvent on the curpos event listeners.  
            top.jQuery.curposlis.trigger("CurposEvent", [
              parseInt(json.FANUC.PMEV[idx].mon_type),
              parseInt(json.FANUC.PMEV[idx].group_num),
              json.FANUC.PMEV[idx].title,
              json.FANUC.PMEV[idx].line]);
            break;

          case PMEV_MAIN_MENU_C:
          case PMEV_QUIC_MENU_C:
          case PMEV_TYPE_MENU_C:
          case PMEV_FCTN_MENU_C:
          case PMEV_DISP_MENU_C:
          case PMEV_SUB_MENU_C:
          case PMEV_PLDn_MENU_C:
          case PMEV_ALPHA_TYPE_MENU_C:
            // Populate the popup dialog.
            dlg_items_per_page =  10;
            dlg_type = parseInt(json.FANUC.PMEV[idx].pmev_type);
            dlg_items = json.FANUC.PMEV[idx].dlg;
            // Remove empty items from end of array
            for (index = dlg_items.length; index > 0; index--) {
              if (dlg_items[index-1].buf.length) {
                break;
              }
            }
            if (index < dlg_items.length) {
              dlg_items.splice(index - dlg_items.length);
            }
            if (dlg_items.length < dlg_items_per_page) {
              dlg_items_this_page = dlg_items.length;
            }
            else {
              dlg_items_this_page = dlg_items_per_page;
            }
            dlg_flyout = parseInt(json.FANUC.PMEV[idx].flyout);
            dlg_notitle = false;
            dlg_spage = 0;
            dlg_epage = ((dlg_items.length - 1) / (dlg_items_per_page-1)) >> 0; // integer division
            dlg_total = parseInt(json.FANUC.PMEV[idx].total);
            dlg_start = parseInt(json.FANUC.PMEV[idx].start);
            if (undefined != json.FANUC.PMEV[idx].item) {
              dlg_item = parseInt(json.FANUC.PMEV[idx].item);
              if (dlg_item < 1) {
                dlg_item = 1;
              }
            }
            else {
              dlg_item = 1;
            }
            dlg_current = parseInt(json.FANUC.PMEV[idx].current);
            dlg_populate();
            if (!json.FANUC.PMEV[idx].next.length) {
              $('#f10').text('');
            }
            else {
              $('#f10').text('0 ' + json.FANUC.PMEV[idx].next);
            }
            // Open the popup dialog.
            dlg_open(json.FANUC.PMEV[idx].title);
            break;

          case PMEV_WARN_MENU_C:
            top.jQuery.devlis.trigger("WarnEvent", [
              json.FANUC.PMEV[idx].img_src,
              json.FANUC.PMEV[idx].pmev_type,
              json.FANUC.PMEV[idx].url,
              json.FANUC.PMEV[idx].timer]);
            break;

          case PMEV_CANC_WARN_MENU_C:
            top.jQuery.devlis.trigger("WarnEvent", [
              "", json.FANUC.PMEV[idx].pmev_type, "", 0]);
            break;

          case PMEV_CANC_MAIN_MENU_C:
          case PMEV_CANC_QUIC_MENU_C:
          case PMEV_CANC_TYPE_MENU_C:
          case PMEV_CANC_FCTN_MENU_C:
          case PMEV_CANC_DISP_MENU_C:
          case PMEV_CANC_SUB_MENU_C:
          case PMEV_CANC_PLDn_MENU_C:
            // Close the popup dialog.
            dlg_popup_closed = true;
            dlg_close();
            break;

          case PMEV_WSCR_MENU_C:
          case PMEV_PRBX_MENU_C:
            // Populate the warning dialog.
            $("#dlgWSCR").dialog("option", "title", json.FANUC.PMEV[idx].title);
            dlg_items = json.FANUC.PMEV[idx].dlg;
            var index;
            var buffer = "";
            var str;
            for (index = 0; index < dlg_items.length; index++) {
              str = dlg_items[index].buf;
              str = str.replace(/ /g, '&nbsp;');
              buffer += str + "<br />";
            }
            $("#dlgWTEXT").html(buffer);

            // Open the warning dialog.
            dlg_wscr_closed = false;
            $("#dlgWSCR").dialog("open");
            document.onkeydown = wscr_dlg_keydown;
            break;

          case PMEV_CANC_WSCR_MENU_C:
          case PMEV_CANC_PRBX_MENU_C:
            // Close the warning dialog.
            dlg_wscr_closed = true;
            $("#dlgWSCR").dialog("close");
            document.onkeydown = null;
            break;
            
          case PMEV_OVRD_MENU_C:
            clearTimeout(ovrd_timer);
            ovrd_timer = setTimeout(function () { $('#dlgOVRD').dialog("close"); }, 2000); 
            // Populate the over-ride dialog in title.       
            dlg_items = json.FANUC.PMEV[idx].dlg;
            $("#dlgOVRD").dialog( "option", "title", dlg_items[0].buf);
            $("#dlgOVRD").dialog( "option", "height", 0 );
            $("#dlgOVRD").dialog( "option", "min-height", 0 );

            // Open the over-ride dialog.
            dlg_wscr_closed = false;
            $("#dlgOVRD").dialog("open");
            document.onkeydown = wscr_dlg_keydown;
            break;

          case PMEV_CANC_OVRD_MENU_C:
            // Close the over-ride dialog.
            dlg_wscr_closed = true;
            $("#dlgOVRD").dialog("close");
            document.onkeydown = null;
            break;

          case PMEV_NUMINPUT_C:
            if ((undefined == json.FANUC.PMEV[idx].start_str) && 
                (undefined == json.FANUC.PMEV[idx].old_val)) {
              break;
            }
            dlg_input_min_val = null;
            dlg_input_max_val = null;
            dlg_input_real = false;
            var start_str = json.FANUC.PMEV[idx].start_str;
            var value = json.FANUC.PMEV[idx].old_val;
            var start_title = top.srvap_getString("HK_title");
            var start_guide = top.srvap_getString("HK_guide").replace("% g", value);
            if ((undefined != json.FANUC.PMEV[idx].min_val) && 
                (undefined != json.FANUC.PMEV[idx].max_val)) {
              if ((json.FANUC.PMEV[idx].min_val != VALUEMIN_C) && (json.FANUC.PMEV[idx].max_val != VALUEMAX_C)) {
                if (undefined != json.FANUC.PMEV[idx].real_val) {
                  dlg_input_real = json.FANUC.PMEV[idx].real_val;
                }
                if (dlg_input_real) {
                  dlg_input_min_val = parseFloat(json.FANUC.PMEV[idx].min_val);
                  dlg_input_max_val = parseFloat(json.FANUC.PMEV[idx].max_val);
                }
                else {
                  dlg_input_min_val = parseInt(json.FANUC.PMEV[idx].min_val);
                  dlg_input_max_val = parseInt(json.FANUC.PMEV[idx].max_val);
                }          
                var res = top.srvap_getString("HK_guide2").split("% g");
                start_guide += "<br />" + res[0] + dlg_input_min_val + "<br />" + res[1] + dlg_input_max_val;
              }
            }
            dlg_input_listener = $('body');
            dlg_input_listener.bind('InputEvent', InputEvent); // Attach handler for InputEvent.
            dlg_numeric_open(start_title, start_guide, start_str);
            break;
 
          case PMEV_STRINPUT_C:
            var start_title = getString("Str_title");
            var start_guide = getString("Str_guide");
            var start_str = json.FANUC.PMEV[idx].start_str;
            dlg_input_listener = $('body');
            dlg_input_listener.bind('InputEvent', InputEvent); // Attach handler for InputEvent.
            dlg_input_open(start_title, start_guide, start_str);
            break;
 
          case PMEV_END_STRINPUT_C:
            break;

          case PMEV_CANC_STRINPUT_C:
            if (!dlg_input_closed) {
              dlg_input_close();
            }
            else if (!dlg_numeric_closed) {
              dlg_numeric_close();
            }
            break;

          case PMEV_DIALOG_BOX_C:
            var url = json.FANUC.PMEV[idx].url;
            if (url.toUpperCase().indexOf("DIALOG_TERM") >= 0) {
              // Close dialogs that may be open first
              if (!dlg_input_closed) {
                dlg_input_close();
              }
              else if (!dlg_numeric_closed) {
                dlg_numeric_close();
              }
              // Close the dialog.
              dlg_dialog_close();
            }
            else {
              // Open the dialog
              dlg_dialog_open(parseInt(json.FANUC.PMEV[idx].device_id),
                json.FANUC.PMEV[idx].title,
                parseInt(json.FANUC.PMEV[idx].width),
                parseInt(json.FANUC.PMEV[idx].height),
                parseInt(json.FANUC.PMEV[idx].posx),
                parseInt(json.FANUC.PMEV[idx].posy),
                url);
            }
            break;

          default:
            top.rpcmc_rprintf("[cgtpmain] PMON event not handled " + JSON.stringify(json.FANUC.PMEV[idx]));
            break;
        } // switch(pmev_type)
        break;

      case SSC_DP:
        if (undefined != json.FANUC.PMEV[idx].lang) {
          // Set the language.
          setLang(json.FANUC.PMEV[idx].lang);

          // Trigger a SetLangEvent on the dict event listeners.
          top.jQuery.dictlis.trigger("SetLangEvent");
          try {
            if (funckeys.location.href.indexOf('blank') < 0) {
              funckeys.location.reload(true);
            }
            if (tpkeys.location.href.indexOf('blank') < 0) {
              tpkeys.location.reload(true);
            }
            if (top.g_irprog) {
              shiftfwd.location.reload(true);
            }
            numpad.location.reload(true);
          } catch (err) { 
            // iframes may not exist
          }
        }
        else {
          top.rpcmc_rprintf("[cgtpmain] DP event not handled " + JSON.stringify(json.FANUC.PMEV[idx]));
        }
        break;

      case SSC_OS:
        switch (parseInt(json.FANUC.PMEV[idx].request_code)) {
          case F_DISCON:
            // Disconnect PMON using RPC.
            top.rpcmc_pmon_disconnect();

            // Display home page
            setTimeout(display_home, 100);
            break;
          case F_KEYC:
            if (undefined != json.FANUC.PMEV[idx].key) {
              // Trigger a KeyPressedEvent on the key event listeners.
              if (json.FANUC.PMEV[idx].keytype == "KeyPressedEvent") {
                top.jQuery.keylis.trigger("KeyPressedEvent", [json.FANUC.PMEV[idx].key]);
              }
              else {
                top.jQuery.keylis.trigger("KeyReleasedEvent", [json.FANUC.PMEV[idx].key]);
              }
            }
            break;
          default:
            top.rpcmc_rprintf("[cgtpmain] OS event not handled " + JSON.stringify(json.FANUC.PMEV[idx]));
            break;
        } // switch(request_code)
        break;

      case SSC_IO:
        switch (parseInt(json.FANUC.PMEV[idx].request_code)) {
          case io_sim_c:
            // Trigger an IOSimEvent on the io event listeners. 
            top.jQuery.iolis.trigger("IOSimEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index, 1]);
            break;
          case io_unsim_c:
            // Trigger an IOSimEvent on the io event listeners. 
            top.jQuery.iolis.trigger("IOSimEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index, 0]);
            break;
          case io_sim_all_c:
            // Trigger an IOSimAllEvent on the io event listeners. 
            top.jQuery.iolis.trigger("IOSimAllEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index, 1]);
            break;
          case io_unsim_all_c:
            // Trigger an IOSimAllEvent on the io event listeners. 
            top.jQuery.iolis.trigger("IOSimAllEvent", [
              json.FANUC.PMEV[idx].type,
              json.FANUC.PMEV[idx].index, 0]);
            break;
          default:
            top.rpcmc_rprintf("[cgtpmain] IO event not handled " + JSON.stringify(json.FANUC.PMEV[idx]));
            break;
        } // switch(request_code)
        break;

      default:
        top.rpcmc_rprintf("[cgtpmain] event not handled " + JSON.stringify(json.FANUC.PMEV[idx]));
        break;
    } // switch(subsys_code)
  } catch (err) {
    // top.rpcmc_rprintf("[cgtpmain] PMON event got an error " + JSON.stringify(json.FANUC.PMEV[idx]));
  }
  } // for each event
} // HandleEvents

function InputEvent(event, val_str) {
  dlg_input_listener.unbind('InputEvent', InputEvent); // Detach handler for InputEvent.
  dlg_input_listener = null;
  if (val_str == null) {
    top.rpcmc_sendKeyCode(tk_cancel_c);
    return;
  }
  // Send the string to the controller
  top.rpcmc_sendTXCMND('B', val_str);

} // InputEvent

function setLang(
    lang) // (in)  language
{
  var lang_suffix = "";  // English is default
  switch (lang) {
    case dplang_japanese_c:
    case dplang_kanji_c:
      lang_suffix = "ja";
      break;
    case dplang_french_c:
      lang_suffix = "fr";
      break;
    case dplang_german_c:
      lang_suffix = "gr";
      break;
    case dplang_spanish_c:
      lang_suffix = "sp";
      break;
    case dplang_chinese_c:
      lang_suffix = "ch";
      break;
    case dplang_taiwanese_c:
      lang_suffix = "tw";
      break;
  } // switch(lang)

  // Load cgtp_xx.res file for the current language.
  jQuery.fn.res({ name: 'cgtp', path: '', mode: 'map', language: lang_suffix });


  // Set top.g_lang_suffix
  if (lang_suffix == "") {
    top.g_lang_suffix = "eg";
  }
  else {
    top.g_lang_suffix = lang_suffix;
  }

  // Clear the fonts so they will be reinit.
  top.g_fixfont = "";
  top.g_deffont = "";

} // setLang

// Get localized string
function getString(
    key) // (in)  key
{
  var str = jQuery.res.text(key);
  if (str) {
    str = str.replace(/\\n/g, '\n');
  }
  return (str);
} // getString

// attachEvent is supported in IE only
// addEventHandler is supported in compliant browsers
function addEventHandler(target, event, handler) {
  if (target.attachEvent) {
    target.attachEvent('on' + event, handler);
  }
  else {
    target.addEventListener(event, handler, false);
  }
}
function initZoom(frameobj, new_zoom) {
  var org_zoom = zoompane(frameobj.name, new_zoom);
  if (org_zoom) {
    addEventHandler(frameobj, "unload", function() {
      zoompane(frameobj.name, org_zoom);
    });
  }
}

function zoompane(frame_name, new_zoom) {
  // Zoom prim, dual, third, or extstat
  // Return original zoom if successful, otherwise 0
  var fele = mainfrm.document.getElementById(frame_name);
  var org_zoom = 0;
  if (fele !== undefined) {
    if (fele != null) {
      if (fele.style.zoom != new_zoom) {
        org_zoom = fele.style.zoom;
        if (org_zoom == "") {
          org_zoom = "100%";
        }
        fele.style.zoom = new_zoom;
      }
    }
  }
  return org_zoom;
}

function rebooted() { 
  top.dlg_ping_progress(top.getString("CONNECTING") + " " + location.host);
  setTimeout(function () { top.document.location.reload(true); }, 3000);
}

function cbTabtpRdy(io_type, io_index, io_value, cbOK, chk_status) {
  if (chk_status == 0) {
    if (io_value == "TRUE") {
      // Controller is ready, call rebooted()
      setTimeout(function () { cbOK(); }, 1000);
      return;
    }
  } else {
    // If $PWRUP_DELAY.$TABTP_RDY does not exist, the monitoring target will be $PWRUP_DELAY.$SY_READY thereafter.
    has_tabtp_rdy = false;
    top.rpcmc_chkkey_rtryabl(null, mor_ss_c, SYSNAME_C,'$PWRUP_DELAY.$SY_READY', cbSyReady, cbOK);
    return;
  }
  // Keep trying every 3 seconds
  setTimeout(function() {
    waitController(cbOK, 1);
  } , 3000 );
}

function cbSyReady(io_type, io_index, io_value, cbOK, chk_status) {
  if (chk_status == 0) {
    if (io_value == "TRUE") {
      // Controller is ready, call rebooted()
      setTimeout(function () { cbOK(); }, 1000);
      return;
    }
  }
  // Keep trying every 3 seconds
  setTimeout(function() {
    waitController(cbOK, 1);
  } , 3000 );
}

function waitController(cbOK, state) {
  top.dlg_ping_progress("*");
  setTimeout(function() {
    if (state == 0) {
      // Ping controller
      ret = ping("http://" + location.host);
      if (ret) {
        top.dlg_ping_progress(top.getString("WAITSYREADY"));
        state++;  // advance to the next state
        waitController(cbOK, state);
      }
      else {
        waitController(cbOK, state);
      }
    }
    else if (top.g_virtual) {
      // Virtual is always ready, call rebooted()
      cbOK();
    }
    else if (has_tabtp_rdy) {
      // Wait for controller with $PWRUP_DELAY.$TABTP_RDY to be ready.
      // Use rpcmc_chkkey_rtryabl, because it returns even if bad status and retries in case of communication failure.
      top.rpcmc_chkkey_rtryabl(null, mor_ss_c, SYSNAME_C,'$PWRUP_DELAY.$TABTP_RDY', cbTabtpRdy, cbOK);
    }
    else {
      // Wait for controller to be ready.
      // Use rpcmc_chkkey_rtryabl, because it returns even if bad status and retries in case of communication failure.
      top.rpcmc_chkkey_rtryabl(null, mor_ss_c, SYSNAME_C,'$PWRUP_DELAY.$SY_READY', cbSyReady, cbOK);
    }
  } , 1000 );
}

function ping(url) {
  var ret = false;
  $.ajax(
    {
      async: false,
      url: url,
      success: function(result) {ret = true;},     
      error: function(result) {ret = false;}
    }
  );
  return ret;
}

function waitReconnect(){
  if (top.g_recon_proc) {
    if (heartbeat_start > heartbeat_end && heartbeat_end !== 0) {
      setTimeout(function(){waitReconnect();}, 1000);
    }
    else {
      top.dlg_recon_close();
      top.g_recon_proc = false;
    }
  }
}
