/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2015
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: cgop.js
 *  
 *  Description:
 *    CGOP plugins
 *
 *  Author: Dan Merchant
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  09-JUN-2015 EVANSJA Merge changes by Dan Merchant.
 *  14-AUG-2015 EVANSJA Rewrite based on cgop.c.
 *  18-SEP-2017 EVANSJA pr50432 - Move common javascript to util.js
 *  09-OCT-2017 EVANSJA pr50451 - Support indirection for incremental jog
 *  03-DEC-2017 EVANSJA pr50565 - Remove cgtpfrm to allow iPad to scroll properly.
 *  05-DEC-2017 EVANSJA pr50577 - Support for checking min and max in dialog input.
 *  16-JAN-2018 EVANSJA pr50656 - Support for function key viewtype.
 *  30-JAN-2018 EVANSJA pr50687 - Support for cgop on dialog boxes.
 *  09-MAR-2018 GRASSCM pr50779 - Support for Execution Control.
 *  09-APR-2018 GRASSCM pr50835 - Support for Checkbox viewtype.
 *  26-APR-2018 GRASSCM pr50876 - Support for tpxensub_ext.
 *  17-MAY-2018 GRASSCM pr50928 - Replace `j_popup_close_c` with `prev_c`.
 *  30-JUL-2018 GRASSCM pr51068 - Support various submenu call types for combobox.
 *  02-MAY-2019 GRASSCM pr51496 - Support check control and MessageBox.
 *  03-MAY-2019 DOWMW   pr50376b - Added Pipe Function calls to not hang PMON
 *  09-MAY-2019 GRASSCM pr51532 - Support indirection
 *  17-MAY-2019 GRASSCM pr51562 - Support commmonly used pipe parameters
 *  12-JUN-2019 GRASSCM pr51613 - Caption enhancements for support of %n
 *  07-AUG-2019 GRASSCM pr51784 - Enhancements for v9.30 second release.
 *  15-OCT-2019 GRASSCM pr51920 - Fix for PIPE Caption and DataIndex properties.
 *  25-NOV-2019 EVANSJA pr51981 - Support for password protected dialog items.
 *  22-JUN-2020 EVANSJA pr52329 - Fix the deletion of alpha entry
 *  06-AUG-2020 GRASSCM pr52431 - Support for indirect dictionary types.
 *  04-DEC-2020 EVANSJA pr52714 - Skip over the format specifiers
 *  21-DEC-2020 EVANSJA pr52764 - Distinguish empty strings from uninit
 *  13-SEP-2020 LEESY   pr53684 - Add connect_id for GENLINK																								 
 *  03-MAY-2023 LEESY   pr53938 - Fixed UpdateValue issue: 
 *                      should not use top.srvap_getString to map the strings to the specific language  
 *  -
*/


// from cgop.h

var mCgop_paintNone       = 0x0000;
var mCgop_paintResize     = 0x0001;
var mCgop_paintUpdate     = 0x0002;
var mCgop_paintReinit     = 0x0004;
var mCgop_paintButton     = 0x0008;
var mCgop_paintRecreate   = 0x0010;
var mCgop_paintDataSource = 0x0020;
var mCgop_paintTextType   = 0x0040;
var mCgop_paintImages     = 0x0080;
var mCgop_paintColors     = 0x0100;

// for CGButton definition
var CGBUTTON_NORMAL       =  0;
var CGBUTTON_PANEL        =  1;
var CGBUTTON_CIRCLE       =  2;
var CGBUTTON_FIXEDCIRCLE  =  3;
var CGBUTTON_BITMAP       =  4;
var CGBUTTON_CHECKBOX     =  5;
var CGBUTTON_LABEL        =  6;
var CGBUTTON_TOGGLE       =  7;
var CGBUTTON_MULTIBITMAP  =  8;
var CGBUTTON_F2           =  9;
var CGBUTTON_F3           = 10;
var CGBUTTON_F4           = 11;
var CGBUTTON_F5           = 12;
var CGBUTTON_F7           = 13;
var CGBUTTON_F8           = 14;
var CGBUTTON_F9           = 15;
var CGBUTTON_F10          = 16;
var CGBUTTON_POSITION     = 20;

// CGDataOperator Property definition
var CGOPERATOR_EQ      = 0;
var CGOPERATOR_NE      = 1;
var CGOPERATOR_LT      = 2;
var CGOPERATOR_LE      = 3;
var CGOPERATOR_GT      = 4;
var CGOPERATOR_GE      = 5;

// CGBorder Property definition
var CGBORDER_3D         = 0;
var CGBORDER_NONE       = 1;
var CGBORDER_BLACK      = 2;
var CGBORDER_FORECOLOR  = 3;
var CGBORDER_EMPHASIS3D = 4;

// for CGEdBox definition
var  CGEDBOX_CHANGED       = 101;
var  CGEDBOX_NUMERICAL     = 0;
var  CGEDBOX_FULLKEY       = 1;
var  CGEDBOX_NUMERICINC    = 2;
var  CGEDBOX_NORMAL        = 0;
var  CGEDBOX_F2            = 1;
var  CGEDBOX_F3            = 2;
var  CGEDBOX_F4            = 3;
var  CGEDBOX_F5            = 4;
var  CGEDBOX_F7            = 5;
var  CGEDBOX_F8            = 6;
var  CGEDBOX_F9            = 7;
var  CGEDBOX_F10           = 8;

// for CGTgLmp definition
var  CGTGLMP_BOOL          = 0;
var  CGTGLMP_NUMERIC       = 1;
var  CGTGLMP_STRING        = 2;
var  CGTGLMP_FULL          = 0;
var  CGTGLMP_CIRCLE        = 1;
var  CGTGLMP_FIXEDCIRCLE   = 2;
var  CGTGLMP_BITMAP        = 3;
var  CGTGLMP_FALSE         = 0;
var  CGTGLMP_TRUE          = 1;

// for CGOsBtn definition
var  CGOSBTN_BOOL          = 0;
var  CGOSBTN_NUMERIC       = 1;
var  CGOSBTN_STRING        = 2;
var  CGOSBTN_NORMALBUTTON  = 0;
var  CGOSBTN_BITMAP        = 1;
var  CGOSBTN_F2            = 2;
var  CGOSBTN_F3            = 3;
var  CGOSBTN_F4            = 4;
var  CGOSBTN_F5            = 5;
var  CGOSBTN_F7            = 6;
var  CGOSBTN_F8            = 7;
var  CGOSBTN_F9            = 8;
var  CGOSBTN_F10           = 9;

// for CGTgBtn definition
var  CGTGBTN_BOOL          = 0;
 
var  CGTGBTN_NUMERIC       = 1;
var  CGTGBTN_STRING        = 2;
var  CGTGBTN_PUSHBUTTON    = 0;
var  CGTGBTN_CHECKBOX      = 1;
var  CGTGBTN_BITMAP        = 2;
var  CGTGBTN_F2            = 3;
var  CGTGBTN_F3            = 4;
var  CGTGBTN_F4            = 5;
var  CGTGBTN_F5            = 6;
var  CGTGBTN_F7            = 7;
var  CGTGBTN_F8            = 8;
var  CGTGBTN_F9            = 9;
var  CGTGBTN_F10           = 10;
var  CGTGBTN_CIRCLE        = 11;

// for CGMulti definition
var  CGMULTI_MAX_NUM       = 10;
var  CGMULTI_DEFAULT       = -1;
var  CGMULTI_DATA01        = 0;
var  CGMULTI_DATA02        = 1;
var  CGMULTI_DATA03        = 2;
var  CGMULTI_DATA04        = 3;
var  CGMULTI_DATA05        = 4;
var  CGMULTI_DATA06        = 5;
var  CGMULTI_DATA07        = 6;
var  CGMULTI_DATA08        = 7;
var  CGMULTI_DATA09        = 8;
var  CGMULTI_DATA10        = 9;
var  CGMULTI_LABEL         = 0;
var  CGMULTI_BITMAP        = 1;

// for Form definition
var  MAX_FORM_NUM        = 10;

// for CGAtChg definition
var  CGATCHG_MAX_NUM       =  MAX_FORM_NUM;
var  CGATCHG_FORMNAME01    =   0;
var  CGATCHG_FORMNAME02    =   1;
var  CGATCHG_FORMNAME03    =   2;
var  CGATCHG_FORMNAME04    =   3;
var  CGATCHG_FORMNAME05    =   4;
var  CGATCHG_FORMNAME06    =   5;
var  CGATCHG_FORMNAME07    =   6;
var  CGATCHG_FORMNAME08    =   7;
var  CGATCHG_FORMNAME09    =   8;
var  CGATCHG_FORMNAME10    =   9;

// for CGMnChg definition
var  CGMNCHG_MAX_NUM       =  MAX_FORM_NUM;
var  CGMNCHG_FORMNAME01    =   0;
var  CGMNCHG_FORMNAME02    =   1;
var  CGMNCHG_FORMNAME03    =   2;
var  CGMNCHG_FORMNAME04    =   3;
var  CGMNCHG_FORMNAME05    =   4;
var  CGMNCHG_FORMNAME06    =   5;
var  CGMNCHG_FORMNAME07    =   6;
var  CGMNCHG_FORMNAME08    =   7;
var  CGMNCHG_FORMNAME09    =   8;
var  CGMNCHG_FORMNAME10    =   9;
var  CGMNCHG_NORMALBUTTON  =   CGOSBTN_NORMALBUTTON;
var  CGMNCHG_BITMAP        =   CGOSBTN_BITMAP;

// for CGBtChg definition
var  CGBTCHG_NORMALBUTTON  =   CGOSBTN_NORMALBUTTON;
var  CGBTCHG_BITMAP        =   CGOSBTN_BITMAP;

// for MnChgBtn definition
var  MAX_FORM_NUM          =  10;
var  MNCHG_BUTTONUP        = 201;
var  TGBTN_BUTTONUP        = 202;
var  BTCHG_BUTTONUP        = 203;
var  OSBTN_BUTTONUP        = 204;
var  BTCHGCTL_PUSHED       = 205;
var  ATCHGCTL_SPECIFIED    = 206;
var  MNCHGCTL_SELECTED     = 207;

// for ComboBox definition
var  COMBOBOX_LISTTYPE     =    0;
var  COMBOBOX_PROGRAMTYPE  =    1;
var  COMBOBOX_VARIABLETYPE =    2;
var  COMBOBOX_DICTIONARYTYPE=   3;
var  COMBOBOX_FILETYPE     =    4;
var  COMBOBOX_STRREGTYPE   =    5;

// for Execution definition
var  CGEXECUTION_KAREL     =    0;
var  CGEXECUTION_SOFTPART  =    1;
var  CGEXECUTION_MAXPARM   =    8;

// for parameter definition (check with lowercase)
var ANZ_HEIGHT             = "height";
var ANZ_LEFT               = "left";
var ANZ_TOP                = "top";
var ANZ_WIDTH              = "width";
var ANZ_CAPTION            = "caption";
var ANZ_TRUECAPTION        = "truecaption";
var ANZ_HALIGN             = "halign";
var ANZ_VALIGN             = "valign";
var ANZ_FORECOLOR          = "forecolor";
var ANZ_BACKCOLOR          = "backcolor";
var ANZ_FONT_NAME          = "fontname";
var ANZ_FONT_SIZE          = "fontsize";
var ANZ_FONT_BOLD          = "fontbold";
var ANZ_FONT_ITALIC        = "fontitalic";
var ANZ_FONT_WEIGHT        = "fontweight";
var ANZ_FONT_UNDERLINE     = "fontunderline";
var ANZ_FONT_STRIKETHROUGH = "fontstrikethrough";
var ANZ_TAB_INDEX          = "tabindex";
var ANZ_INDEX              = "index";
var ANZ_CGDATATYPE         = "datatype";
var ANZ_CGDATAINDEX        = "dataindex";
var ANZ_CGBORDER           = "border";
var ANZ_CGTYPE             = "type";
var ANZ_CGVIEWTYPE         = "viewtype";
var ANZ_CGTRUEBMP          = "truebmp";
var ANZ_CGTRUEIMAGE        = "trueimage";
var ANZ_CGFALSEBMP         = "falsebmp";
var ANZ_CGFALSEIMAGE       = "falseimage";
var ANZ_CGTRUECOLOR        = "truecolor";
var ANZ_CGFALSECOLOR       = "falsecolor";
var ANZ_CGTRUESTRCOLOR     = "truestrcolor";
var ANZ_CGFALSESTRCOLOR    = "falsestrcolor";
var ANZ_CGTRUEVALUE        = "truevalue";
var ANZ_CGFALSEVALUE       = "falsevalue";
var ANZ_CGSETVALUE         = "setvalue";
var ANZ_CGOTHERPHASE       = "otherphase";
var ANZ_CGCMPVALUE         = "cmpvalue";
var ANZ_CGCMPOPERATOR      = "cmpoperator";
var ANZ_CGFORMNAME         = "formname"; 
var ANZ_CGFRAME            = "frame";
var ANZ_CGFORMNAME01       = "formname01";
var ANZ_CGFORMNAME02       = "formname02";
var ANZ_CGFORMNAME03       = "formname03";
var ANZ_CGFORMNAME04       = "formname04";
var ANZ_CGFORMNAME05       = "formname05";
var ANZ_CGFORMNAME06       = "formname06";
var ANZ_CGFORMNAME07       = "formname07";
var ANZ_CGFORMNAME08       = "formname08";
var ANZ_CGFORMNAME09       = "formname09";
var ANZ_CGFORMNAME10       = "formname10";
var ANZ_CGFORMCAPTION01    = "formcaption01";
var ANZ_CGFORMCAPTION02    = "formcaption02";
var ANZ_CGFORMCAPTION03    = "formcaption03";
var ANZ_CGFORMCAPTION04    = "formcaption04";
var ANZ_CGFORMCAPTION05    = "formcaption05";
var ANZ_CGFORMCAPTION06    = "formcaption06";
var ANZ_CGFORMCAPTION07    = "formcaption07";
var ANZ_CGFORMCAPTION08    = "formcaption08";
var ANZ_CGFORMCAPTION09    = "formcaption09";
var ANZ_CGFORMCAPTION10    = "formcaption10";
var ANZ_CGDATADEFAULT      = "datadefault";
var ANZ_CGDATA01           = "data01";
var ANZ_CGDATA02           = "data02";
var ANZ_CGDATA03           = "data03";
var ANZ_CGDATA04           = "data04";
var ANZ_CGDATA05           = "data05";
var ANZ_CGDATA06           = "data06";
var ANZ_CGDATA07           = "data07";
var ANZ_CGDATA08           = "data08";
var ANZ_CGDATA09           = "data09";
var ANZ_CGDATA10           = "data10";
var ANZ_CGVALUEMIN01       = "valuemin01";
var ANZ_CGVALUEMIN02       = "valuemin02";
var ANZ_CGVALUEMIN03       = "valuemin03";
var ANZ_CGVALUEMIN04       = "valuemin04";
var ANZ_CGVALUEMIN05       = "valuemin05";
var ANZ_CGVALUEMIN06       = "valuemin06";
var ANZ_CGVALUEMIN07       = "valuemin07";
var ANZ_CGVALUEMIN08       = "valuemin08";
var ANZ_CGVALUEMIN09       = "valuemin09";
var ANZ_CGVALUEMIN10       = "valuemin10";
var ANZ_CGVALUEMAX01       = "valuemax01";
var ANZ_CGVALUEMAX02       = "valuemax02";
var ANZ_CGVALUEMAX03       = "valuemax03";
var ANZ_CGVALUEMAX04       = "valuemax04";
var ANZ_CGVALUEMAX05       = "valuemax05";
var ANZ_CGVALUEMAX06       = "valuemax06";
var ANZ_CGVALUEMAX07       = "valuemax07";
var ANZ_CGVALUEMAX08       = "valuemax08";
var ANZ_CGVALUEMAX09       = "valuemax09";
var ANZ_CGVALUEMAX10       = "valuemax10";
var ANZ_INTERVAL           = "interval";
var ANZ_PERIODIC           = "periodic";
var ANZ_PULSE              = "pulse";
var ANZ_PULSETIME          = "pulsetime";
var ANZ_NAME               = "name";
var ANZ_PARAMETER1         = "parameter1";
var ANZ_PARAMETER2         = "parameter2";
var ANZ_PARAMETER3         = "parameter3";
var ANZ_PARAMETER4         = "parameter4";
var ANZ_PARAMETER5         = "parameter5";
var ANZ_PARAMETER6         = "parameter6";
var ANZ_PARAMETER7         = "parameter7";
var ANZ_PARAMETER8         = "parameter8";
var ANZ_PROGTYPE           = "progtype";
var ANZ_DATASOURCE         = "datasource";
var ANZ_TEXTTYPE           = "texttype";
var ANZ_TEXTINDEX          = "textindex";
var ANZ_CLICKEVENTS        = "clickevents";
var ANZ_CHANGEEVENTS       = "changeevents";
var ANZ_FOCUSEVENTS        = "focusevents";
var ANZ_IGNOREKEYSEVENTS   = "ignorekeysevents";
var ANZ_IOSIM              = "iosim";
var ANZ_PIPE               = "pipe";
var ANZ_READONLY           = "readonly";
var ANZ_INVISIBLE          = "invisible";
var ANZ_DISABLE            = "disable";
var ANZ_POSALIGN           = "posalign";
var ANZ_GROUPNUM           = "groupnum";
var ANZ_OVERRIDE_RO        = "overridero";
var ANZ_RWACCESS           = "rwaccess";
var ANZ_NOCONTENT          = "nocontent"
var ANZ_MOMENTARY          = "momentary"
var ANZ_VALUESTR           = "valuestr";
var ANZ_TRUEVALUESTR       = "truevaluestr";
var ANZ_INCRVALUE          = "incrvalue";
var ANZ_CONFIRM            = "confirm";
var ANZ_INVERT             = "invert";
var ANZ_SETFOCUS           = "setfocus";
var ANZ_CLICKME            = "clickme";
var ANZ_CHKDEADMAN         = "chkdeadman";
var ANZ_CHKTPENABLE        = "chktpenable";
var ANZ_CHKSHIFT           = "chkshift";
var ANZ_CHKIO              = "chkio";
var ANZ_CHKIOTYPE          = "chkiotype";
var ANZ_CHKIOIDX           = "chkioidx";
var ANZ_TRUEFONT           = "truefont";
var ANZ_CAPTIONONIMG       = "captiononimg";
var ANZ_NOSCALE            = "noscale";
var ANZ_EXECCONNECTID      = "execconnectid";
var ANZ_FEXECCONNECTID     = ("f" + ANZ_EXECCONNECTID);

var MAX_VARINFO_C          =  3;
var MAX_INDIRECT_C         =  5;
var DIRECT_C               = -1;
var PREP_NORMAL_C          =  0;
var PREP_VARNAME_C         = -1;

// cgop.js helper functions
// Alternative to case-sensitive hasOwnProperty() method.
// We cannot enforce that pipe data case matches javascript structures.
function hasOwnProperty_noCase(data, target_property) {
  var l_property;
  for (l_property in data) {
    if (l_property.toLowerCase() == target_property.toLowerCase()) {
      return true;
    }
  }
  return false;
}

// Format and open check control MessageBox
function chkkey_message(data, message) {
  var msg_string = "";
  var caption = data.Caption;
  if (caption == "") {
    caption = data.ButtonText;
  }
  msg_string = caption + "\n\n" + "Message:\n" + message + "\n";
  top.dlg_msgbox_open("Check Condition", msg_string);
}

// Initialize chkkey results to assist with async. processing
function chkkey_result_init(data) {
  data.ChkDeadmanEstpResult = false;
  data.ChkDeadmanMorssResult = false;
  data.ChkDeadmanResult = false;
  data.ChkTpEnableResult = false;
  data.ChkShiftResult = false;
  data.ChkIoResult = false;
}

// Initialize chkkey results if control supports chkkey
function chkkey_init(data) {
  if (chkkey_supported(data) &&
      ((data.ChkDeadman != "0") || (data.ChkTpEnable != "0") ||
      ((data.ChkShift != "0") || (data.ChkIo != "0")))) {
    chkkey_result_init(data);
  }
}

// Attach handler for click events (toggle or momentary)
function SetTgbtnEvent(data) {
  data.Momentary = parseInt(data.Momentary);
  if ((data.CtlType == GMCT_TGBTN) && (data.Momentary !== 0)) {
    data.$this.bind("mousedown", data, HandleMouseDownEvent);
    data.$this.bind("mouseup", data, HandleMouseUpEvent);
    data.$this.bind("mouseout", data, HandleMouseUpEvent);
  }
  else {
    data.$this.bind("click", data, HandleClickEvent);
  }
}

function HandleVarCallbackErrors(data_type, val_str, data_caption, ret_stat) {
  // Check for rpc error
  if (val_str == null ||
      val_str == "Uninitialized" ||
      val_str == UNINIT_STR_C) {

    switch (data_type) {
    case CGRSTYPE_FIX:
      break;

    case CGRSTYPE_REG:
    case CGRSTYPE_STRREG:
      ret_stat = "REGERR_C";
      break;

    case CGRSTYPE_SYSVAR:
      ret_stat = "SVERR_C";
      break;

    case CGRSTYPE_KARELVAR:
      ret_stat = "KRLERR_C";
      break;

    case CGRSTYPE_DICTELE:
      ret_stat = "ASCERR_C";
      break;

    case CGRSTYPE_DI:
    case CGRSTYPE_DO:
    case CGRSTYPE_AI:
    case CGRSTYPE_AO:
    case CGRSTYPE_PI:
    case CGRSTYPE_PO:
    case CGRSTYPE_RI:
    case CGRSTYPE_RO:
    case CGRSTYPE_SI:
    case CGRSTYPE_SO:
    case CGRSTYPE_TPIN:
    case CGRSTYPE_TPOUT:
    case CGRSTYPE_WI:
    case CGRSTYPE_WO:
    case CGRSTYPE_GI:
    case CGRSTYPE_GO:
    case CGRSTYPE_UI:
    case CGRSTYPE_UO:
    case CGRSTYPE_LDI:
    case CGRSTYPE_LDO:
    case CGRSTYPE_LAI:
    case CGRSTYPE_LAO:
    case CGRSTYPE_WSI:
    case CGRSTYPE_WSO:
    case CGRSTYPE_MIB:
    case CGRSTYPE_MIN:
    case CGRSTYPE_FLAG:
    case CGRSTYPE_MARKER:
      ret_stat = "IOERR_C";
      break;

    default:
      ret_stat = "ASCERR_C";
      break;
    }
    top.dlg_msgbox_open(top.srvap_getString(ret_stat), data_caption);
  }
  else {
    ret_stat == "";
  }
} // HandleVarCallbackErrors

// Populate curpos table
function PopulateCurpos(data, i_title, i_buffer) {
  var l_start_idx;
  var l_idx;
  var l_n_axes;
  var l_buffer;


  // Offset unused row for JOINT
  if (data.DataIndex == (top.srvap_getString("JOINT_C"))) {
    l_start_idx = 1;
  }
  else {
    l_start_idx = 0;
  }

  if (i_buffer === UNINIT_STR_C) {
    // Erase curpos table
    $('#title', data.$this).html(UNINIT_STR_C);
    for (l_idx = 0; l_idx < MAX_GRP_AXES; l_idx++) {
      $('#label' + l_idx, data.$this).html(UNINIT_STR_C);
      $('#unit' + l_idx, data.$this).html('&nbsp;' + UNINIT_STR_C);
      $('#value' + l_idx, data.$this).html(UNINIT_STR_C);
    }
    data.NumAxes = 0; // Uninit to repopulate labels and units
  }
  else {
    // Populate curpos table
    $('#title', data.$this).html(i_title);
    l_n_axes = i_buffer.length;
    if (l_n_axes != data.NumAxes) {
      // Show the labels and units
      data.NumAxes = l_n_axes;
      for (l_idx = 0; l_idx < l_n_axes; l_idx++) {
        $('#label' + (l_idx + l_start_idx), data.$this).html(i_buffer[l_idx].label);
        $('#unit' + (l_idx + l_start_idx), data.$this).html('&nbsp;' + i_buffer[l_idx].unit);
      }
      if (l_start_idx > 0) {
        // Clear unused rows
        for (l_idx = 0; l_idx < l_start_idx; l_idx++) {
          $('#label' + l_idx, data.$this).html("");
          $('#unit' + l_idx, data.$this).html("");
          $('#value' + l_idx, data.$this).html("");
        }
      }
    }
    // Show the values
    for (l_idx = 0; l_idx < l_n_axes; l_idx++) {
      // Some browsers do not support white-space: pre
      l_buffer = i_buffer[l_idx].value;
      l_buffer = l_buffer.replace(/ /g, '&nbsp;');
      l_buffer = l_buffer.replace(/span&nbsp;/g, 'span ');
      $('#value' + (l_idx + l_start_idx), data.$this).html(l_buffer);
    }
  }
} // PopulateCurpos

// Convert indirect tag to integer
function GetIndirectIdx(i_property_val) {

  var l_indirect_obj = null;
  var l_ind_idx = DIRECT_C;

  if (typeof(i_property_val) == "string") {
    l_indirect_obj = i_property_val.toLowerCase().match(/!indirect([0-5]{1})/g);
    if (l_indirect_obj !== null) {
      l_ind_idx = Number(l_indirect_obj[0].substring(9,10));
      if ((isNaN(l_ind_idx)) || (l_ind_idx <= 0) || (l_ind_idx > MAX_INDIRECT_C)) {
        l_ind_idx = DIRECT_C;
      }
    }
  }
  return l_ind_idx;
} // GetIndirectIdx

// Helper function to check if name/type and var/index contain indirect tag.
// returns:
//   'true' if either string contains indirect tag
//   'false' if neither string contains indirect tag
function ContainsIndirectTag(i_prognam, i_progvar) {
  if ((DIRECT_C == GetIndirectIdx(i_prognam)) &&
      (DIRECT_C == GetIndirectIdx(i_progvar))) {
    return false;
  }
  else {
    return true;
  }
} // ContainsIndirectTag

// Replace all occurences of '!indirectN' in a string with IValueStr[N]
function ReplaceIndirectValues(data, i_argIdx, i_argVal) {
  var l_loop_idx;
  if (data.hasOwnProperty(i_argIdx)) {
    data[i_argIdx] = i_argVal;
    for (l_loop_idx = 0; l_loop_idx < MAX_INDIRECT_C; l_loop_idx++) {
      var l_regex = new RegExp('!indirect' + (l_loop_idx + 1), 'gi');
      data[i_argIdx] = data[i_argIdx].replace(l_regex, data.IValueStr[l_loop_idx]);
    }
  }
} // ReplaceIndirectValues

// Reset monitor for updated indirection value
function UpdateIndirectMonitor(data, data_type, ind_idx, arg_value) {
  switch (data_type) {
  case CGRSTYPE_REG:
  case CGRSTYPE_STRREG:
  case CGRSTYPE_SYSVAR:
  case CGRSTYPE_KARELVAR:
  case CGRSTYPE_KARELPOS:
  case CGRSTYPE_POSREG:
  case CGRSTYPE_TFRAME:
  case CGRSTYPE_JFRAME:
  case CGRSTYPE_UFRAME:
    var l_prev_prognam = "";
    var l_prev_progvar = "";

    if (ind_idx == DIRECT_C) {
      l_prev_prognam = data.varinfo[0].prognam;
      l_prev_progvar = data.varinfo[0].progvar;
    }
    else {
      l_prev_prognam = data.Istr1[ind_idx];
      l_prev_progvar = data.Istr2[ind_idx];
    }

    // Update data object
    PrepareCtl(data_type, arg_value, data, PREP_NORMAL_C, ind_idx, data.varinfo[0]);

    // Update PMON
    if (ind_idx == DIRECT_C) {

      // Only update PMON if values have changed
      if ((l_prev_prognam != data.varinfo[0].prognam) ||
          (l_prev_progvar != data.varinfo[0].progvar)) {

        // Stop PMON monitors for previous data
        if (!ContainsIndirectTag(l_prev_prognam, l_prev_progvar)) {
          data.VarEvent--;
          top.rpcmc_stopVarMonitor(l_prev_prognam, l_prev_progvar);
        }
 
        // Start PMON monitors for new data
        if (!ContainsIndirectTag(data.varinfo[0].prognam, data.varinfo[0].progvar)) {
          data.VarEvent++;
          top.rpcmc_startVarMonitor(data.varinfo[0].prognam, data.varinfo[0].progvar, data.Interval);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
      else {
        if (!ContainsIndirectTag(data.varinfo[0].prognam, data.varinfo[0].progvar)) {
          top.rpcmc_getVar(data.varinfo[0].prognam, data.varinfo[0].progvar, HandleVarCallback, DIRECT_C, data);
        }
      }
    }
    else {
      // Only update PMON if values have changed
      if ((l_prev_prognam != data.Istr1[ind_idx]) ||
          (l_prev_progvar != data.Istr2[ind_idx])) {

        // Stop PMON monitors for previous data
        if (!ContainsIndirectTag(l_prev_prognam, l_prev_progvar)) {
          data.VarEvent--;
          top.rpcmc_stopVarMonitor(l_prev_prognam, l_prev_progvar);
        }
 
        // Start PMON monitors for new data
        if (!ContainsIndirectTag(data.Istr1[ind_idx], data.Istr2[ind_idx])) {
          data.VarEvent++;
          top.rpcmc_startVarMonitor(data.Istr1[ind_idx], data.Istr2[ind_idx], data.Interval);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
      else {
        top.rpcmc_getVar(data.Istr1[ind_idx], data.Istr2[ind_idx], HandleVarCallback, ind_idx, data);
      }
    }
    break;

  case CGRSTYPE_DICTELE:
    var l_prev_dictnam = "";
    var l_prev_dictele = "";

    if (ind_idx == DIRECT_C) {
      l_prev_dictnam = data.DictName;
      l_prev_dictele = data.DictEle;
    }
    else {
      l_prev_dictnam = data.Istr1[ind_idx];
      l_prev_dictele = data.Istr2[ind_idx];
    }

    // Update data object
    PrepareCtl(data_type, arg_value, data, PREP_NORMAL_C, ind_idx, data.varinfo[0]);

    // Read dictionary element
    if (ind_idx == DIRECT_C) {
      // Only read if the value is new
      if ((l_prev_dictnam != data.DictName) ||
          (l_prev_dictele != data.DictEle)) {
        if (!ContainsIndirectTag(data.varinfo[0].prognam, data.varinfo[0].progvar)) {
          top.rpcmc_dpread2(data.DictName, data.DictEle, HandleDictCallback, data);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
    }
    else {
      // Only read if the value is new
      if ((l_prev_dictnam != data.Istr1[ind_idx]) ||
          (l_prev_dictele != data.Istr2[ind_idx])) {
        if (!ContainsIndirectTag(data.Istr1[ind_idx], data.Istr2[ind_idx])) {
          top.rpcmc_dpread3(data.Istr1[ind_idx], data.Istr2[ind_idx], HandleDictCallback, ind_idx, data);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
    }
    break;

  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_AI:
  case CGRSTYPE_AO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_GI:
  case CGRSTYPE_GO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_MIN:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:

    var l_prev_iotype = 0;
    var l_prev_ioindex = 0;

    // Copy current values
    if (ind_idx == DIRECT_C) {
      l_prev_iotype = data.IOType;
      l_prev_ioindex = data.IOIndex;
    }
    else {
      l_prev_iotype = data.Istr1[ind_idx];
      l_prev_ioindex = data.Istr2[ind_idx];
    }

    // Update data object
    PrepareCtl(data_type, arg_value, data, PREP_NORMAL_C, ind_idx, null);

    // Update PMON
    if (ind_idx == DIRECT_C) {

      // Only update PMON if values have changed
      if ((l_prev_iotype != data.IOType) ||
          (l_prev_ioindex != data.IOIndex)) {

        // Stop PMON monitors for previous data
        if (!ContainsIndirectTag(l_prev_iotype, l_prev_ioindex)) {
          top.rpcmc_stopIOMonitor(l_prev_iotype, l_prev_ioindex);
        }
 
        // Start PMON monitors for new data
        if (!ContainsIndirectTag(data.IOType, data.IOIndex)) {
          top.rpcmc_startIOMonitor(data.IOType, data.IOIndex);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
    }
    else {
      // Only update PMON if values have changed
      if ((l_prev_iotype != data.Istr1[ind_idx]) ||
          (l_prev_ioindex != data.Istr2[ind_idx])) {

        // Stop PMON monitors for previous data
        if (!ContainsIndirectTag(l_prev_iotype, l_prev_ioindex)) {
          top.rpcmc_stopIOMonitor(l_prev_iotype, l_prev_ioindex);
        }
 
        // Start PMON monitors for new data
        if (!ContainsIndirectTag(data.Istr1[ind_idx], data.Istr2[ind_idx])) {
          top.rpcmc_startIOMonitor(data.Istr1[ind_idx], data.Istr2[ind_idx]);
          data.ValueText = UNINIT_STR_C;
          data.ValueNumber = 0;
        }
      }
    }
    break;

  case CGRSTYPE_CURPOS:

    if (!ContainsIndirectTag(data.DataIndex, "")) { // MonType is derived from DataIndex for CURPOS
      switch (data.DataIndex.toUpperCase()) {
        case top.srvap_getString("WORLD_C"):
          data.MonType = PMON_MON_CURPOS_WORLD_C;
          break;

        case top.srvap_getString("USER_C"):
          data.MonType = PMON_MON_CURPOS_USER_C;
          break;

        case top.srvap_getString("JOINT_C"):
        default:
          data.MonType = PMON_MON_CURPOS_JOINT_C;
          break;
      }

      if (!ContainsIndirectTag(data.GroupNum, data.MonType)) { // MonType is derived from DataIndex for CURPOS type
        // Stop current monitor
        top.rpcmc_stopCurposMonitor(data.MonType);

        // Clear the control's text values
        PopulateCurpos(data, UNINIT_STR_C, UNINIT_STR_C);

        // Start new montype monitor
        top.rpcmc_startCurposMonitor(data.MonType, data.Interval);
      }
    }
    break;

  default:
    break;
  }
} // UpdateIndirectMonitor

// Private functions (same as pmcompib.c)
function FAbButtonCompo_SetText(data, text) {
  if (data.fKey) {
    top.gPaneFKey[data.fDeviceId].FKeyText[data.fKey-1] = text;
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    return;
  }
  var innerObj = data.$this.children();
  innerObj = innerObj.not("img, canvas"); // Exclusions do not get caption
  if (innerObj) {
    innerObj.text(text);
    SetAlignment(data);
  }
} // FAbButtonCompo_SetText

function FAbButtonCompo_SetTextColor(data, textColor) {
  if (data.fKey) {
    top.gPaneFKey[data.fDeviceId].FKeyTextColor[data.fKey-1] = textColor;
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    return;
  }
  data.$this.css("color", textColor);
} // FAbButtonCompo_SetTextColor

function FAbButtonCompo_SetBackColor(data, backColor) {
  if (data.fKey) {
    top.gPaneFKey[data.fDeviceId].FKeyBackColor[data.fKey-1] = backColor;
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    return;
  }
  data.$this.css("background-color", backColor);
} // FAbButtonCompo_SetBackColor

function FAbButtonCompo_SetImageBackColor(data, backColor) {
  if (data.fKey) {
    // supposed to just set the backcolor of the image
    top.gPaneFKey[data.fDeviceId].FKeyBackColor[data.fKey-1] = backColor;
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    return;
  }
  var innerObj = data.$this.children('img');
  if (innerObj) {
    innerObj.css("background-color", backColor);
  }
} // FAbButtonCompo_SetBackColor

function FAbButtonCompo_SetFImage(data, image) {
  if ((image == "") || (image == data.Image)) {
    return;
  }
  data.Image = image;
  if (data.fKey) {
    top.gPaneFKey[data.fDeviceId].FKeyImgIcon[data.fKey-1] = image;
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    return;
  }
  var innerObj = data.$this.children('img');
  if (innerObj) {
    innerObj.attr("src", image);
  }
} // FAbButtonCompo_SetFImage

function FAbButtonCompo_SetValue(data, sb_state) {
  if (data.fKey) {
    return;
  }
  switch (parseInt(data.Border)) {
  case CGBORDER_3D:
    if (sb_state) {
      data.$this.css({"border-style":"inset", "border-width":"2px", "border-color":"white"});
    }
    else {
      data.$this.css({"border-style":"outset", "border-width":"2px", "border-color":"white"});
    }
    break;
  case CGBORDER_EMPHASIS3D:
    if (sb_state) {
      data.$this.css({"border-style":"inset", "border-width":"4px", "border-color":"white"});
    }
    else {
      data.$this.css({"border-style":"outset", "border-width":"4px", "border-color":"white"});
    }
    break;
  } // switch(data.Border)
} // FAbButtonCompo_SetValue

function FAbButtonCompo_SetCaptionOnImage(data, caption) {
} // FAbButtonCompo_SetCaptionOnImage

// Create button for checkbox and circle view types
function FAbButtonCompo_CreateTableButton(data) {
  var innerObj = data.$this.children();
  if (innerObj) {
    innerObj.each( function() {
      if (($(this).is("img")) || ($(this).is("canvas"))) {
        $(this).css({"display":"inline-block"});
        ComputeVerticalAlignment($(this), data);
      }
      else if ($(this).is("span")) {
        // Set Alignment.
        var horizontal = "center";
        if (data.HAlign == 0) {
          horizontal = "left";
        }
        else if (data.HAlign == 2) {
          horizontal = "right";
        }
        $(this).css({"display":"table-cell", "vertical-align":"middle",
                      "text-align":horizontal});
      }
    });
    CheckChildrenWidth(data, innerObj);
  }
} // FAbButtonCompo_CreateTableButton

function FAbButtonCompo_RenderCircle(data, fillColor) {
  var canvasObj = data.$this.children('canvas').get(0);
  
  if (canvasObj.getContext) {
    var canvasCtx = canvasObj.getContext('2d');
    var canvasWidth = data.$this.children("canvas").width();
    var circleArc = (canvasWidth / 2);
    var xOffset = ((data.$this.width() - canvasWidth -
                    data.$this.children("span").width()) / 2);
  
    if (xOffset && (typeof xOffset === 'number')) {
      if ((xOffset <= data.$this.width()) && xOffset >= 0) {
        data.$this.children("canvas").css({"left": "" + xOffset + "px"});
      }
    }
  
    // Draw Circle
    canvasCtx.clearRect(0, 0, canvasWidth, canvasWidth);
    canvasCtx.arc(circleArc, circleArc, circleArc, 0, 2 * Math.PI);
    canvasCtx.fillStyle = fillColor;
    canvasCtx.fill();
  }
} // FAbButtonCompo_RenderCircle

// Parse HTML entities from jquery control strings
function cgop_FixStrings(data) {
  $.each(data, function (argn, argv) {
    if (typeof(argv) !== 'string') {
      return;
    }
    else {
      // Clean HTML entities from control
      if (argv.match("&gt;")) {
        data[argn] = argv.replace("&gt;", '>');
      }
      if (argv.match("&lt;")) {
        data[argn] = argv.replace("&lt;", '<');
      }
    }
  });
}

// Create control only when PMON Server is ready
function cgop_ServerReady(data) {
  // Initialize control data.
  cgop_InitCtlData(data);

  // Initialize Control Data again using indirection.
  cgop_InitCtlData2(data);

  // Create control.
  cgop_CreateCtl(data);

  // Initialize events.
  cgop_InitRobotEvent(true, data);

  if (data.CtlType == GMCT_HELP) {
    SetHelp(data);
  }

} // cgop_ServerReady

// Common routine to parse dictionary name and element number
function cgop_DictParam(data) {
  var pos = data.DataSource.indexOf("[");
  if (pos > -1) {
    data.DictName = data.DataSource.substring(0, pos);
    data.DictEle = parseInt(data.DataSource.substring(
      pos + 1, data.DataSource.length - 1));
  }
  else {
    data.DictName = "";
  }
} // cgop_DictParam

// New instance of control.
function NPP_New(data) {

  data.repaint = mCgop_paintNone;
  data.MenuUp = false;
  data.inPulse = false;
  data.ValueText = "";
  data.ValueNumber = 0;

  data.Image = "";
  if (undefined != data.Interval) {
    data.Interval = Number(data.Interval);
  }
  if (undefined != data.GroupNum) {
    if (!isNaN(data.GroupNum)) {
    data.GroupNum = Number(data.GroupNum);
  }
  }
  data.VarEvent = 0;
  data.IDataType = [];
  data.IDataIndex = [];
  data.IValueStr = [];
  data.IOIndex = 0;
  data.Istr1 = [];
  data.Istr2 = [];
  data.Istr3 = [];
  for (var si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
    data.IDataType.push(0);
    data.IDataIndex.push("");
    data.IValueStr.push("!indirect" + (si_ind + 1));
    data.Istr1.push("");
    data.Istr2.push("");
    data.Istr3.push("");
  }
  data.Iargr = {}
  data.varinfo = [];
  for (si_ind = 0; si_ind < MAX_VARINFO_C; si_ind++) {
    data.varinfo.push({prognam:"", progvar:""});
  }

  data.ViewType = GeneralViewType(data);

  if (data.ViewType == CGBUTTON_POSITION) {
    // TrueFont should not be used.
    data.TrueFont = "0";
    data.FontName = top.getFixFont();
  }

  SetCommonParams(data);
  if (data.IsDialog) {
    cgop_ServerReady(data);
  }
  else { 
    top.rpcmc_getTaskIdx(data.fDeviceId, function(taskIdx) { 
      data.fTaskIdx = taskIdx;

      // Complete after top.rpcmc_getTaskIdx completes.
      cgop_ServerReady(data);
    });
  }
} // NPP_New

// Destroy instance of control.
function NPP_Destroy(data) {

  // Uninitialize events.
  cgop_InitRobotEvent(false, data);

  // Delete control.
  cgop_DeleteCtl(data);

} // NPP_Destroy

// Submenu rpc callback. Prevents rpc error from cauing dialog box to get out
// of sync with the actual state of the control.
function DialogOpen(error, data) {
  if (error == 0) {
    // Success, put up the menu
    top.dlg_listener = data.$this;
    top.dlg_listener.bind('MenuEvent', data, HandleMenuEvent);
    data.MenuUp = true;
  }
  else {
    // Error, reset control
    FAbButtonCompo_SetValue(data, false); // shows button released
    if (top.dlg_listener == data.$this) {
      // Unbind this listener event handler
      top.dlg_listener.unbind('MenuEvent', HandleMenuEvent);
      top.dlg_listener = null;
    }
    data.MenuUp = false;
  }
} // DialogOpen

// Private functions (same as cgop.c)

/*
  Prepare all the necessary data to access the value from the controller.

  NOTE: This function processes two types of data descriptors
        (1) The primary control's data properties.
          EX: PrepareCtl(data.DataType, data.DataIndex,,, DIRECT_C,);

        (2) The indirect data properties (0 to (MAX_INDIRECT_C - 1)) for the control
          EX: PrepareCtl(data.IDataType[si_ind], data.IDataIndex[si_ind],,, si_ind,);
*/
function PrepareCtl(p_DataType, p_DataIndex, data, p_NameEvent, p_Indirect, p_VarInfo) {

  switch (p_DataType) {
  case CGRSTYPE_REG:
    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = NUMREG_C;
      p_VarInfo.progvar = NUM_REGISTER_C + "[" + p_DataIndex + "]";
    }
    else {
      data.Istr1[p_Indirect] = NUMREG_C;
      data.Istr2[p_Indirect] = NUM_REGISTER_C + "[" + p_DataIndex + "]";
    }
    break;

  case CGRSTYPE_STRREG:
    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = STRREG_C;
      p_VarInfo.progvar = STR_REGISTER_C + "[" + p_DataIndex + "]";
    }
    else {
      data.Istr1[p_Indirect] = STRREG_C;
      data.Istr2[p_Indirect] = STR_REGISTER_C + "[" + p_DataIndex + "]";
    }
    break;

  case CGRSTYPE_SYSVAR:
    p_DataIndex = p_DataIndex.toUpperCase();
    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = SYSNAME_C;
      p_VarInfo.progvar = p_DataIndex;
    }
    else {
      data.Istr1[p_Indirect] = SYSNAME_C;
      data.Istr2[p_Indirect] = p_DataIndex;
    }
    break;

  case CGRSTYPE_KARELVAR:
  case CGRSTYPE_KARELPOS:
    p_DataIndex = p_DataIndex.toUpperCase();
    var pos = p_DataIndex.indexOf("]");
    if (pos > -1) {
      if (p_Indirect == DIRECT_C) {
        p_VarInfo.prognam = p_DataIndex.substring(1, pos);
        p_VarInfo.progvar = p_DataIndex.substr(pos + 1);
      }
      else {
        data.Istr1[p_Indirect] = p_DataIndex.substring(1, pos);
        data.Istr2[p_Indirect] = p_DataIndex.substr(pos + 1);
      }
    }
    else {
      if (p_Indirect == DIRECT_C) {
        p_VarInfo.prognam = "NOTSPECIFIED";
        p_VarInfo.progvar = p_DataIndex;
      }
      else {
        data.Istr1[p_Indirect] = "NOTSPECIFIED";
        data.Istr2[p_Indirect] = p_DataIndex;
      }
    }
    break;

  case CGRSTYPE_POSREG:
    var l_prog_str = POSREG_C;
    var l_var_str = POS_REGISTER_C + "[" + data.GroupNum + "," + data.DataIndex + "]";

    if (!ContainsIndirectTag(data.DataIndex, "")) {
      data.MonType = PMON_MON_SYS_C;
      data.MonSysType = PMON_SYS_POSREG_C;
    }
    else {
      data.MonType = -1;
    }

    if ((data.GroupNum < 1) || (data.GroupNum > MAX_NUM_GRPS)) {
      data.GroupNum = 1;
    }

    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = l_prog_str;
      p_VarInfo.progvar = l_var_str;
    }
    else {
      data.Istr1[p_Indirect] = l_prog_str;
      data.Istr2[p_Indirect] = l_var_str;
    }
    break;

  case CGRSTYPE_CURPOS:
    p_DataIndex = p_DataIndex.toUpperCase();

    var mon_type = -1;

    if (!ContainsIndirectTag(p_DataIndex)) {
      switch (p_DataIndex) {
      case top.srvap_getString("USER_C"):
      mon_type = PMON_MON_CURPOS_USER_C;
        break;

      case top.srvap_getString("WORLD_C"):
      mon_type = PMON_MON_CURPOS_WORLD_C;
        break;

      case top.srvap_getString("JOINT_C"):
      default:
        mon_type = PMON_MON_CURPOS_JOINT_C;
        break;
      }
    }

    if (p_Indirect == DIRECT_C) {
      data.MonType = mon_type;
    }
    else {
      data.Istr1[p_Indirect] = mon_type;
      data.Istr2[p_Indirect] = "";
    }
    data.NumAxes = 0; // uninit value
    break;

  case CGRSTYPE_TFRAME:
    var l_prog_str = SYSNAME_C;
    var l_var_str = PG_MNUTOOL_SYSN_C + "[" + data.GroupNum + "," + data.DataIndex + "]";

    if (!ContainsIndirectTag(data.DataIndex, "")) {
      data.MonType = PMON_MON_SYS_C;
      data.MonSysType = PMON_SYS_TFRAME_C;
    }
    else {
      data.MonType = -1;
    }

    if ((data.GroupNum < 1) || (data.GroupNum > MAX_NUM_GRPS)) {
      data.GroupNum = 1;
    }

    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = l_prog_str;
      p_VarInfo.progvar = l_var_str;
    }
    else {
      data.Istr1[p_Indirect] = l_prog_str;
      data.Istr2[p_Indirect] = l_var_str;
    }
    break;

  case CGRSTYPE_JFRAME:
    var l_prog_str = "TPFDEF";
    var l_var_str = "JOGFRAMES[" + data.GroupNum + "," + data.DataIndex + "]";

    if (!ContainsIndirectTag(data.DataIndex, "")) {
      data.MonType = PMON_MON_SYS_C;
      data.MonSysType = PMON_SYS_JFRAME_C;
    }
    else {
      data.MonType = -1;
    }

    if ((data.GroupNum < 1) || (data.GroupNum > MAX_NUM_GRPS)) {
      data.GroupNum = 1;
    }

    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = l_prog_str;
      p_VarInfo.progvar = l_var_str;
    }
    else {
      data.Istr1[p_Indirect] = l_prog_str;
      data.Istr2[p_Indirect] = l_var_str;
    }
    break;

  case CGRSTYPE_UFRAME:
    var l_prog_str = SYSNAME_C;
    var l_var_str = PG_MNUFRAME_SYSN_C + "[" + data.GroupNum + "," + data.DataIndex + "]";

    if (!ContainsIndirectTag(data.DataIndex, "")) {
      data.MonType = PMON_MON_SYS_C;
      data.MonSysType = PMON_SYS_UFRAME_C;
    }
    else {
      data.MonType = -1;
    }

    if ((data.GroupNum < 1) || (data.GroupNum > MAX_NUM_GRPS)) {
      data.GroupNum = 1;
    }

    if (p_Indirect == DIRECT_C) {
      p_VarInfo.prognam = l_prog_str;
      p_VarInfo.progvar = l_var_str;
    }
    else {
      data.Istr1[p_Indirect] = l_prog_str;
      data.Istr2[p_Indirect] = l_var_str;
    }
    break;

  case CGRSTYPE_DICTELE:
    var pos = p_DataIndex.indexOf("[");
    if (pos > -1) {
      if (p_Indirect == DIRECT_C) {
        data.DictName = p_DataIndex.substring(0, pos);
        data.DictEle = p_DataIndex.substring(pos + 1, p_DataIndex.length - 1);
      }
      else {
        data.Istr1[p_Indirect] = p_DataIndex.substring(0, pos);
        data.Istr2[p_Indirect] = p_DataIndex.substring(pos + 1, p_DataIndex.length - 1);
      }
    }
    else {
      if (p_Indirect == DIRECT_C) {
        data.DictName = "";
      }
      else {
        data.Istr1[p_Indirect] = "";
      }
    }
    break;

  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:
    if (p_Indirect == DIRECT_C) {
      data.IOType = p_DataType;
      data.IOIndex = p_DataIndex;
      data.IOBoolean = 1;
    }
    else {
      data.Istr1[p_Indirect] = p_DataType;
      data.Istr2[p_Indirect] = p_DataIndex;
      data.Istr3[p_Indirect] = 1;
    }
    break;

  case CGRSTYPE_AI:
  case CGRSTYPE_AO:
  case CGRSTYPE_GI:
  case CGRSTYPE_GO:
  case CGRSTYPE_MIN:
    if (p_Indirect == DIRECT_C) {
      data.IOType = p_DataType;
      data.IOIndex = p_DataIndex;
      data.IOBoolean = 0;
    }
    else {
      data.Istr1[p_Indirect] = p_DataType;
      data.Istr2[p_Indirect] = p_DataIndex;
      data.Istr3[p_Indirect] = 0;
    }
    break;

  } // switch (DataType)
} // PrepareCtl

// GetValueFromController
// Get value from controller.
// zsi_ind > 0 uses IDataType/IDataIndex
// zsi_ind -1  uses DataType/DataIndex
// zsi_ind -2  uses ComboBox's DataSource DictionaryType
// zsi_ind -3  uses ComboBox's DataSource StrRegType
// 
// NOTE: Position types are not supported
function GetValueFromController(p_ind, data) {

  var l_DataType;
  var l_DataIndex;
  var si_ind = 0;
  var si_status = 0;
  var l_VarInfo = new Object();
  l_VarInfo.prognam = '';
  l_VarInfo.progvar = '';

  if (p_ind >= 0) {
    if ((p_ind >= MAX_INDIRECT_C) || (data.IDataIndex[p_ind] == "")) {
      return;
    }
    l_DataType = data.IDataType[p_ind];
    l_DataIndex = data.IDataIndex[p_ind];
  }
  else if (p_ind == -1) {
    l_DataType = data.DataType;
    l_DataIndex = data.DataIndex;
  }
  else if (p_ind == -2) {
    l_DataType = CGRSTYPE_DICTELE;
    l_DataIndex = data.DataSource;
  }
  else if (p_ind == -3) {
    // COMBOBOX_STRREGTYPE
    // Change DataSource to use the value as the index into StrReg.
    l_DataType = CGRSTYPE_STRREG;
    data.DataSource = data.ValueNumber;
    l_DataIndex = data.DataSource;
  }
  else {
    return;
  }

  // Prepare all the necessary data to access the value from the
  // controller.
  PrepareCtl(l_DataType, l_DataIndex, data, PREP_NORMAL_C, DIRECT_C, l_VarInfo);

  if (ContainsIndirectTag(l_VarInfo.prognam, l_VarInfo.progvar)) {
    // Don't process indirect assignments for indirect parameters yet
    return;
  }

  switch (l_DataType) {
  case CGRSTYPE_FIX:
    break;

  case CGRSTYPE_STRREG:
    data.varinfo[1] = l_VarInfo;
    top.rpcmc_getVar(data.varinfo[1].prognam, data.varinfo[1].progvar, HandleVarCallback, p_ind, data);
    break;

  case CGRSTYPE_REG:
  case CGRSTYPE_SYSVAR:
  case CGRSTYPE_KARELVAR:
    top.rpcmc_getVar(l_VarInfo.prognam, l_VarInfo.progvar, HandleVarCallback, p_ind, data);
    break;

  case CGRSTYPE_DICTELE:
    si_status = -1;

    if (data.DictName != "") {
      si_status = 0;

      if (p_ind == -2) {

        cgop_DictParam(data);

        // Get text value selected from tpxensub_ext.
        // DataSource contains the starting dictionary element.
        // Need to increase by selected enumeration value.
        if (data.BoolType === "1") { // enum values start at 1
          si_idx = (data.DictEle + data.ValueNumber);
        }
        else {
          si_idx = (data.DictEle + (data.ValueNumber - 1));
        }
      }
      else {
        si_idx = data.DictEle;
      }

      if ((top.g_connect_id === "") || (!top.g_connect_id)) {
        si_status = -1;
      }
      if (!si_status) {
        top.rpcmc_dpread3(data.DictName, si_idx, HandleDictCallback2, p_ind, data);
      }
    }
    break;

  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_AI:
  case CGRSTYPE_AO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_GI:
  case CGRSTYPE_GO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_MIN:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:
    if ((data.IOSim) && (p_ind == -1)) {
      // Only support I/O simulation for DataType/DataIndex
      // Get the I/O simulation status
      top.rpcmc_iocksim(data.IOType, data.IOIndex, HandleIOSimCallback, p_ind, data);
    }
    else {
      // Get the I/O point.
      top.rpcmc_iovalrd(data.IOType, data.IOIndex, HandleIOCallback, p_ind, data);
    }
    break;

  default:
    break;

  } // switch (l_DataType)

} // GetValueFromController

// Send the execution event to controller.
function SendEventToController(zpc_fc, data) {
  if (top.g_echo) {
    return;
  }
  // Verify Execution Control
  if (top.gPaneExec[data.fDeviceId].fTaskIdx &&
      top.gPaneExec[data.fDeviceId].fExecConnectId &&
      (!(top.gPaneExec[data.fDeviceId].fExecName === ''))) {
    // Send to controller based on Type
    if (top.gPaneExec[data.fDeviceId].fExecType == CGEXECUTION_SOFTPART) {
      $.get("/SOFTPART/" + top.gPaneExec[data.fDeviceId].fExecName +
            "?fc=" + zpc_fc +
            "&PANEID=" + top.gPaneExec[data.fDeviceId].fTaskIdx +
            "&CONTROLID=" + data.Id +
            "&CONNECTID=" + top.gPaneExec[data.fDeviceId].fExecConnectId,
            function() { });
    }
    else {
      $.get("/SOFTPART/WEBEXEC?fc=" + zpc_fc +
            "&KAREL_NAME=" + top.gPaneExec[data.fDeviceId].fExecName +
            "&PANEID=" + top.gPaneExec[data.fDeviceId].fTaskIdx +
            "&CONTROLID=" + data.Id +
            "&CONNECTID=" + top.gPaneExec[data.fDeviceId].fExecConnectId,
            function() { });
    }
  }
  else {
    return;
  }
} // SendEventToController

// Verify value is within given limits.
function CheckLimits(val_str, data) {

  if (isNaN(val_str)) {
    return 0;
  }
  var val_num = Number(val_str);
  if (((undefined != data.ValueMin) && (val_num < data.ValueMin)) ||
      ((undefined != data.ValueMax) && (val_num > data.ValueMax))) {
    top.dlg_msgbox_open(top.srvap_getString("LIMITERR_C"),
                        ('(' + data.ValueMin + ',' + data.ValueMax + ')'));
    return -2;
  }
  return 0;

} // CheckLimits

// UpdateValueToController
// Update value to controller.
// p_ind -1  uses DataType/DataIndex
// p_ind -2  uses ComboBox's TextType/TextIndex
//  
// NOTE: Position types are not supported
function UpdateValueToController(p_ind, val_str, data) {

  var l_DataType;
  var l_DataIndex;
  var p_VarInfo = data.varinfo[0];

  if (p_ind == -1) {
    l_DataType = data.DataType;
    l_DataIndex = data.DataIndex;
  }
  else if (p_ind == -2) {
    l_DataType = data.TextType;
    l_DataIndex = data.TextIndex;
    p_VarInfo = data.varinfo[2];
    if (!l_DataType) {
      p_VarInfo = null;
      return 0;  // no TextType specified
    }
  }
  else {
    p_VarInfo = null;
    return 0;
  }

  switch (l_DataType) {
  case CGRSTYPE_FIX:
    break;

  case CGRSTYPE_DICTELE:
    break;

  case CGRSTYPE_REG:
  case CGRSTYPE_STRREG:
  case CGRSTYPE_SYSVAR:
  case CGRSTYPE_KARELVAR:
    // Check limits.
    var l_status = CheckLimits(val_str, data);
    if (l_status) {
      p_VarInfo = null;
      return l_status;
    }
    top.rpcmc_setVar(p_VarInfo.prognam, p_VarInfo.progvar, val_str);
    break;

  case CGRSTYPE_KARELPOS:
  case CGRSTYPE_POSREG:
    var kcl_cmd_str = "RECORD ";

    if (l_DataType == CGRSTYPE_KARELPOS) {
      kcl_cmd_str += "[" + p_VarInfo.prognam + "]" + p_VarInfo.progvar;
      top.rpcmc_cpkcl(kcl_cmd_str, cgop_KclRecord_HandleCallback, data);
    }

    if (l_DataType == CGRSTYPE_POSREG) {
      kcl_cmd_str += "PR[" + data.GroupNum + "," + data.DataIndex + "]";
      top.rpcmc_cpkcl(kcl_cmd_str, cgop_KclRecord_HandleCallback, data);
    }
    break;
  
  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_AI:
  case CGRSTYPE_AO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_GI:
  case CGRSTYPE_GO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_MIN:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:
    var val_num;
    if (data.IOSim) {
      if (val_str == top.srvap_getString("SIM")) {
        val_num = 1;
      }
      else if (val_str == top.srvap_getString("UNSIM")) {
        val_num = 0;
      }
      else {
        if (val_str == "") {
          p_VarInfo = null;
          return -1;
        }
        val_num = Number(val_str);
      }
      if (val_num) {
        // Simulate the I/O point.
        top.rpcmc_iosim(data.IOType, data.IOIndex);
      }
      else {
        // Unsimulate the I/O point.
        top.rpcmc_iounsim(data.IOType, data.IOIndex);
      }
    }
    else {
      if (val_str == top.srvap_getString("ON")) {
        val_num = 1;
      }
      else if (val_str == top.srvap_getString("OFF")) {
        val_num = 0;
      }
      else {
        if (val_str == "") {
          p_VarInfo = null;
          return -1;
        }
        val_num = Number(val_str);
      }

      // Check limits.
      var status = CheckLimits(val_str, data);
      if (status) {
        p_VarInfo = null;
        return status;
      }

      // Set the I/O point.
      top.rpcmc_iovalset(data.IOType, data.IOIndex, val_num);
    }
    break;

  default:
    p_VarInfo = null;
    return -1;
    break;
  } // switch (ss_DataType) 

  if ((p_ind == -1) &&
      (top.gPaneExec[data.fDeviceId].fFlags & PANE_FLAG_CHANGE_EVENTS_C)) {
    SendEventToController("0x430014", data);
  }
  p_VarInfo = null;
  return 0;

} // UpdateValueToController

// Replace any indirection with actual value.
function IndirectReplace(data) {

  var si_ind;
  var pos;
  var arg_match;

 $.each(data, function( argn, argv ) { 
    if (typeof argv !== 'string') {
      return;
    }

    // Value contain !PaneId?
    if ((pos = argv.toLowerCase().indexOf("!paneid")) >= 0) {
      argv = argv.substring(0, pos) + data.fTaskIdx + argv.substring(pos + 7); // Replace Immediately
      data[argn] = argv;
    }

    // Value contain !Indirect?
    if ((pos = argv.toLowerCase().indexOf("!indirect")) >= 0) {
      for (si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
        if (Number(argv.substring(pos+9, pos+10)) == si_ind+1) {

          arg_match = false;

          $.each(data.Iargr, function(argIdx, argObject) {
            if (argIdx == argn) {
              arg_match = true; // Don't process the same argument twice
              return false; // break from the loop
            }
          });

          if (!arg_match) {
            data.Iargr[argn] = argv;
          }

          GetValueFromController(si_ind, data); // Replace with callback
          break;
        }
      }
    }
  });
} // IndirectReplace

// Returns true if value is between min and max.
function CompareMultiValue(val_num, ValueMin, ValueMax) {

  if ((val_num >= Number(ValueMin)) && (val_num <= Number(ValueMax))) {
    return true;
  }
  return false;

} // CompareMultiValue

// Return appropriate string based on value.
function GetMultiString(data) {

  var val_num;
  if ((data.DataType == CGRSTYPE_STRREG) && (data.ValueText)) {
    // Assume string register holds a numeric value.
    val_num = Number(data.ValueText);
  }
  else {
    val_num = data.ValueNumber;
  }

  if (CompareMultiValue(val_num, data.ValueMin01, data.ValueMax01)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName01;
    }
    else {
      return data.Data01;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin02, data.ValueMax02)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName02;
    }
    else {
      return data.Data02;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin03, data.ValueMax03)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName03;
    }
    else {
      return data.Data03;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin04, data.ValueMax04)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName04;
    }
    else {
      return data.Data04;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin05, data.ValueMax05)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName05;
    }
    else {
      return data.Data05;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin06, data.ValueMax06)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName06;
    }
    else {
      return data.Data06;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin07, data.ValueMax07)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName07;
    }
    else {
      return data.Data07;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin08, data.ValueMax08)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName08;
    }
    else {
      return data.Data08;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin09, data.ValueMax09)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName09;
    }
    else {
      return data.Data09;
    }
  }
  if (CompareMultiValue(val_num, data.ValueMin10, data.ValueMax10)) {
    if (data.CtlType == GMCT_ATCHG) {
      return data.PageName10;
    }
    else {
      return data.Data10;
    }
  }
  return data.DataDefault;

} // GetMultiString

// Return General View Type.
function GeneralViewType(data) {
  var ret = CGBUTTON_NORMAL;

  data.ViewType = parseInt(data.ViewType);
  data.DataType = parseInt(data.DataType);
  data.fKey = 0;
  switch (data.CtlType) {
  case GMCT_TGLMP:
    switch (data.ViewType) {
    case CGTGLMP_FULL:
      ret = CGBUTTON_PANEL;
      break;
    case CGTGLMP_CIRCLE:
      ret = CGBUTTON_CIRCLE;
      break;
    case CGTGLMP_FIXEDCIRCLE:
      ret = CGBUTTON_FIXEDCIRCLE;
      break;
    case CGTGLMP_BITMAP:
      ret = CGBUTTON_BITMAP;
      break;
    default:
      ret = CGBUTTON_PANEL;
      break;
    }
    break;
  case GMCT_TGBTN:
    switch (data.ViewType) {
    case CGTGBTN_PUSHBUTTON:
      ret = CGBUTTON_TOGGLE;
      break;
    case CGTGBTN_CHECKBOX:
      ret = CGBUTTON_CHECKBOX;
      break;
    case CGTGBTN_BITMAP:
      ret = CGBUTTON_BITMAP;
      break;
    case CGTGBTN_F2:
      data.fKey = 2;
      ret = CGBUTTON_F2;
      break;
    case CGTGBTN_F3:
      data.fKey = 3;
      ret = CGBUTTON_F3;
      break;
    case CGTGBTN_F4:
      data.fKey = 4;
      ret = CGBUTTON_F4;
      break;
    case CGTGBTN_F5:
      data.fKey = 5;
      ret = CGBUTTON_F5;
      break;
    case CGTGBTN_F7:
      data.fKey = 7;
      ret = CGBUTTON_F7;
      break;
    case CGTGBTN_F8:
      data.fKey = 8;
      ret = CGBUTTON_F8;
      break;
    case CGTGBTN_F9:
      data.fKey = 9;
      ret = CGBUTTON_F9;
      break;
    case CGTGBTN_F10:
      data.fKey = 10;
      ret = CGBUTTON_F10;
      break;
    case CGTGBTN_CIRCLE:
      ret = CGBUTTON_FIXEDCIRCLE;
      break;
    default:
      ret = CGBUTTON_TOGGLE;
      break;
    }
    break;
  case GMCT_MULTI:
    switch (data.ViewType) {
    case CGMULTI_LABEL:
      ret = CGBUTTON_LABEL;
      break;
    case CGMULTI_BITMAP:
      ret = CGBUTTON_MULTIBITMAP;
      break;
    default:
      ret = CGBUTTON_LABEL;
      break;
    }
    break;
  case GMCT_OSBTN:
  case GMCT_MNCHG:
  case GMCT_BTCHG:
    switch (data.ViewType) {
    case CGOSBTN_NORMALBUTTON:
      ret = CGBUTTON_TOGGLE;
      break;
    case CGOSBTN_BITMAP:
      ret = CGBUTTON_BITMAP;
      break;
    case CGOSBTN_F2:
      data.fKey = 2;
      ret = CGBUTTON_F2;
      break;
    case CGOSBTN_F3:
      data.fKey = 3;
      ret = CGBUTTON_F3;
      break;
    case CGOSBTN_F4:
      data.fKey = 4;
      ret = CGBUTTON_F4;
      break;
    case CGOSBTN_F5:
      data.fKey = 5;
      ret = CGBUTTON_F5;
      break;
    case CGOSBTN_F7:
      data.fKey = 7;
      ret = CGBUTTON_F7;
      break;
    case CGOSBTN_F8:
      data.fKey = 8;
      ret = CGBUTTON_F8;
      break;
    case CGOSBTN_F9:
      data.fKey = 9;
      ret = CGBUTTON_F9;
      break;
    case CGOSBTN_F10:
      data.fKey = 10;
      ret = CGBUTTON_F10;
      break;
    default:
      ret = CGBUTTON_TOGGLE;
      break;
    }
    break;
  case GMCT_EDBOX:
  case GMCT_COMBOBOX:
    switch (data.ViewType) {
    case CGEDBOX_F2:
      data.fKey = 2;
      ret = CGBUTTON_F2;
      break;
    case CGEDBOX_F3:
      data.fKey = 3;
      ret = CGBUTTON_F3;
      break;
    case CGEDBOX_F4:
      data.fKey = 4;
      ret = CGBUTTON_F4;
      break;
    case CGEDBOX_F5:
      data.fKey = 5;
      ret = CGBUTTON_F5;
      break;
    case CGEDBOX_F7:
      data.fKey = 7;
      ret = CGBUTTON_F7;
      break;
    case CGEDBOX_F8:
      data.fKey = 8;
      ret = CGBUTTON_F8;
      break;
    case CGEDBOX_F9:
      data.fKey = 9;
      ret = CGBUTTON_F9;
      break;
    case CGEDBOX_F10:
      data.fKey = 10;
      ret = CGBUTTON_F10;
      break;
    default:
      break;
    }
    break;
  case GMCT_LABEL:
    switch (data.DataType) {
    case CGRSTYPE_POSREG:
    case CGRSTYPE_KARELPOS:
    case CGRSTYPE_CURPOS:
    case CGRSTYPE_TFRAME:
    case CGRSTYPE_JFRAME:
    case CGRSTYPE_UFRAME:
    case CGRSTYPE_TPPOS:
      ret = CGBUTTON_POSITION;
      break;
    default:
      break;
    } // switch (data.DataType)
    break;
  default:
    break;
  } // switch (data.CtlType)

  return ret;

} // GeneralViewType

// Get images based on value.
function SetImages(data) {

  switch (data.ViewType) {
  case CGBUTTON_BITMAP:
  case CGBUTTON_CHECKBOX:
    var trueImg = "";
    var falseImg = "";
    var sb_state = GetButtonState(data);

    if (data.ViewType == CGBUTTON_BITMAP) {
      trueImg = data.TrueImage;
      falseImg = data.FalseImage;
    } else {
      trueImg = data.TrueChkImg;
      falseImg = data.FalseChkImg;
    }
    
    // Change the bitmap based on value.
    if ((sb_state) && (trueImg != "")) {
      FAbButtonCompo_SetFImage(data, trueImg);
      FAbButtonCompo_SetImageBackColor(data, data.TrueColor);
    }
    else {
      if (falseImg != "") {
        FAbButtonCompo_SetFImage(data, falseImg);
        FAbButtonCompo_SetImageBackColor(data, data.FalseColor);
      }
    }
    if (data.CaptionOnImage != "0") {
      FAbButtonCompo_SetText(data, "");
    }
    data.dispNoImageStr = false;
    break;

  case CGBUTTON_MULTIBITMAP:
    // Change the bitmap based on value.
    FAbButtonCompo_SetFImage(data, GetMultiString(data));
    break;

  case CGBUTTON_F2:
  case CGBUTTON_F3:
  case CGBUTTON_F4:
  case CGBUTTON_F5:
  case CGBUTTON_F7:
  case CGBUTTON_F8:
  case CGBUTTON_F9:
  case CGBUTTON_F10:
    // Change the image based on value.
    var sb_state = GetButtonState(data);
    if ((sb_state) && (data.TrueImage != "")) {
      FAbButtonCompo_SetFImage(data, data.TrueImage);
      FAbButtonCompo_SetImageBackColor(data, data.TrueColor);
    }
    else {
      FAbButtonCompo_SetFImage(data, data.FalseImage);
      FAbButtonCompo_SetImageBackColor(data, data.FalseColor);
    }
    break;
  } // switch view_type

} // SetImages

// Set the border.
function cgop_SetBorder(data) {

  switch (parseInt(data.Border)) {
  case CGBORDER_3D:
    data.$this.css({"border-style":"outset", "border-width":"2px", "border-color":"white"});
    break;
  case CGBORDER_NONE:
    data.$this.css("border-style", "none");
    break;
  case CGBORDER_BLACK:
    data.$this.css({"border-style":"solid", "border-width":"2px", "border-color":"black"});
    break;
  case CGBORDER_FORECOLOR:
    data.$this.css({"border-style":"solid", "border-width":"2px", "border-color":" + data.ForeColor + "});
    break;
  case CGBORDER_EMPHASIS3D:
    data.$this.css({"border-style":"outset", "border-width":"4px", "border-color":"white"});
    break;
  } // switch(data.Border)

} // cgop_SetBorder

// Return state of button.
function GetButtonState(data) {

  var sb_flag = false;
  var val_num;

  if ((data.CtlType == GMCT_MNCHG) || (data.CtlType == GMCT_EDBOX)) {
    val_num = Number(data.MenuUp);
  }
  else {
    val_num = data.ValueNumber;
  }

  switch (data.CtlType) {
  case GMCT_TGLMP:
    if (data.Type == CGTGLMP_STRING) {
      if ((data.ValueStr != "") && (data.ValueText != "")) {
        switch (data.CmpOperator) {
        case CGOPERATOR_NE:
          if (data.ValueText != data.CmpValue) {
            sb_flag = true;
          }
          break;
        case CGOPERATOR_LT:
          if (data.ValueText < data.CmpValue) {
            sb_flag = true;
          }
          break;
        case CGOPERATOR_LE:
          if (data.ValueText <= data.CmpValue) {
            sb_flag = true;
          }
          break;
        case CGOPERATOR_GT:
          if (data.ValueText > data.CmpValue) {
            sb_flag = true;
          }
          break;
        case CGOPERATOR_GE:
          if (data.ValueText >= data.CmpValue) {
            sb_flag = true;
          }
          break;
        case CGOPERATOR_EQ:
        default:
          if (data.ValueText == data.CmpValue) {
            sb_flag = true;
          }
          break;
        }
      }
    }
    else if (data.Type == CGTGLMP_BOOL) {
      if ((undefined == data.Invert) || (data.Invert == "0")) {
        sb_flag = val_num;
      }
      else {
        sb_flag = !val_num;
      }
    }
    else {
      var sf_cmpValue = Number(data.CmpValue);
      switch (data.CmpOperator) {
      case CGOPERATOR_NE:
        if (val_num != sf_cmpValue)
          sb_flag = true;
        break;
      case CGOPERATOR_LT:
        if (val_num < sf_cmpValue)
          sb_flag = true;
        break;
      case CGOPERATOR_LE:
        if (val_num <= sf_cmpValue)
          sb_flag = true;
        break;
      case CGOPERATOR_GT:
        if (val_num > sf_cmpValue)
          sb_flag = true;
        break;
      case CGOPERATOR_GE:
        if (val_num >= sf_cmpValue)
          sb_flag = true;
        break;
      case CGOPERATOR_EQ:
      default:
        if (val_num == sf_cmpValue)
          sb_flag = true;
        break;
      }
    }
    break;

  case GMCT_TGBTN:
    if (data.Type == CGTGBTN_STRING) {
      if (data.ValueText != "") {
        sb_flag = true;
        if (data.ValueStr != "") {
          if (data.ValueText == data.CmpValue) {
            sb_flag = false;
          }
        }
        if (sb_flag) {
          sb_flag = false;
          if (data.TrueValueStr != "") {
            if (data.ValueText == data.cmpValue) {
              sb_flag = true;
            }
          }
          if (!sb_flag) {
            sb_flag = data.OtherPhase;
          }
        }
      }
    }
    else if (data.Type == CGTGBTN_BOOL) {
      sb_flag = (data.Invert != "0") ? !val_num : val_num;
    }
    else {
      var sf_falseValue = Number(data.FalseValue);
      var sf_trueValue = Number(data.TrueValue);
      if (val_num == sf_falseValue) {
        sb_flag = false;
      }
      else if (val_num == sf_trueValue) {
        sb_flag = true;
      }
      else {
        sb_flag = Number(data.OtherPhase);
      }
    }
    break;

  default:
    sb_flag = val_num;
    break;

  } // switch (data.CtlType)

  return sb_flag;

} // GetButtonState

// Set the background and foreground color.
function cgop_SetColors(data) {

  var si_backColor;
  var si_trueColor;
  var si_falseColor;
  var si_foreColor;
  var si_trueStrColor;
  var si_falseStrColor;
  var sb_state;

  if (data.fKey) {
    // Change White background to DarkGray for function keys.
    data.BackColor = data.BackColor.toUpperCase();
    data.InvisibleColor = data.InvisibleColor.toUpperCase();
    if ((data.BackColor == kWave_ColorLightGray) || 
        (data.BackColor == kWave_ColorWhite) ||
        (data.BackColor == data.InvisibleColor)) {
      data.BackColor = DARK_GRAY;
      data.ForeColor = kWave_ColorWhite;
    }
    if (data.FalseColor != "") {
      data.FalseColor = data.FalseColor.toUpperCase();
      if ((data.FalseColor == kWave_ColorLightGray) || 
          (data.FalseColor == kWave_ColorWhite) ||
          (data.FalseColor == data.InvisibleColor)) {
        data.FalseColor = DARK_GRAY;
        data.ForeColor = kWave_ColorWhite;
        if (data.FalseStrColor != "") {
          data.FalseStrColor = kWave_ColorWhite;
        }
      }
    }
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
  }

  if (((data.CtlType == GMCT_OSBTN) && (data.ViewType != CGBUTTON_BITMAP)) || 
      (data.ViewType == CGBUTTON_CHECKBOX)) {
    // Don't use these colors in CommandButtons or checkbox.
    data.FalseColor = data.BackColor;
    data.TrueColor = data.BackColor;
  }

  if (data.Disable) {
    // Change colors for disabled controls.
    si_backColor = kWave_ColorLightGray;
    si_foreColor = kWave_ColorGray;
    si_trueColor = kWave_ColorLightGray;
    si_falseColor = kWave_ColorLightGray;
    si_trueStrColor = kWave_ColorGray;
    si_falseStrColor = kWave_ColorGray;
  }
  else {
    si_backColor = data.BackColor;
    si_foreColor = data.ForeColor;
    si_trueColor = data.BackColor;
    si_falseColor = data.BackColor;
    si_trueStrColor = data.ForeColor;
    si_falseStrColor = data.ForeColor;
    if (data.TrueColor != "") {
      si_trueColor = data.TrueColor;
    }
    if (data.FalseColor != "") {
      si_falseColor = data.FalseColor;
    }
    if (data.TrueStrColor != "") {
      si_trueStrColor = data.TrueStrColor;
    }
    if (data.FalseStrColor != "") {
      si_falseStrColor = data.FalseStrColor;
    }
  }

  switch (data.ViewType) {
  case CGBUTTON_BITMAP:
    // Set ForeColor.
    if (data.CaptionOnImage != "0") {
      FAbButtonCompo_SetTextColor(data, si_falseStrColor);
    }
    else {
      FAbButtonCompo_SetTextColor(data, si_foreColor);
    }

    // Set BackColors.
    FAbButtonCompo_SetBackColor(data, data.InvisibleColor);
    sb_state = GetButtonState(data);
    if (sb_state) {
      FAbButtonCompo_SetImageBackColor(data, si_trueColor);
    }
    else {
      FAbButtonCompo_SetImageBackColor(data, si_falseColor);
    }
    data.TrueColorImage = si_trueColor;
    data.FalseColorImage = si_falseColor;
    break;

  case CGBUTTON_MULTIBITMAP:
    // Set ForeColor.
    FAbButtonCompo_SetTextColor(data, si_foreColor);

    // Set BackColors.
    FAbButtonCompo_SetBackColor(data, data.InvisibleColor);
    FAbButtonCompo_SetImageBackColor(data, si_backColor);
    break;

  case CGBUTTON_F2:
  case CGBUTTON_F3:
  case CGBUTTON_F4:
  case CGBUTTON_F5:
  case CGBUTTON_F7:
  case CGBUTTON_F8:
  case CGBUTTON_F9:
  case CGBUTTON_F10:
    if (data.CtlType == GMCT_TGBTN) {
      // Set ForeColor/BackColor based on value.
      sb_state = GetButtonState(data);
      if (sb_state) {
        si_foreColor = si_trueStrColor;
        si_backColor = si_trueColor;
      }
      else {
        si_foreColor = si_falseStrColor;
        si_backColor = si_falseColor;
      }
    }
    // Set ForeColor.
    FAbButtonCompo_SetTextColor(data, si_foreColor);

    // Set BackColors.
    FAbButtonCompo_SetBackColor(data, si_backColor);
    FAbButtonCompo_SetImageBackColor(data, si_backColor);
    data.TrueColorImage = si_trueColor;
    data.FalseColorImage = si_falseColor;
    break;

  case CGBUTTON_POSITION:
    /*
      This has to do with the style of the control.
      Currently all position view types are the default styling.
    */
    break;

  default:
    // Set ForeColor.
    if ((data.ViewType == CGBUTTON_CIRCLE) || (data.ViewType == CGBUTTON_FIXEDCIRCLE)) {
      FAbButtonCompo_RenderCircle(data, si_falseColor);
      FAbButtonCompo_SetBackColor(data, si_backColor);
    }
    else {
      if ((data.ViewType == CGBUTTON_TOGGLE) || (data.ViewType == CGBUTTON_PANEL)) {
        FAbButtonCompo_SetTextColor(data, si_falseStrColor);
      }
      else {
        FAbButtonCompo_SetTextColor(data, si_foreColor);
      }

      // Set BackColor.
      if ((data.ViewType == CGBUTTON_TOGGLE) || (data.ViewType == CGBUTTON_PANEL)) {
        FAbButtonCompo_SetBackColor(data, si_falseColor);
      }
      else {
        FAbButtonCompo_SetBackColor(data, si_backColor);
      }
    }
    break;
  } // switch data.ViewType
} // cgop_SetColors

// Set the caption
function SetCaption(data) {

  switch (data.CtlType) {
  case GMCT_COMBOBOX:
  case GMCT_LABEL:
  case GMCT_EDBOX:
    if ((data.DataType == CGRSTYPE_FIX) || (data.ViewType != CGBUTTON_NORMAL)) {
      data.ButtonText = data.Caption;
    }
    else {
      data.ButtonText = "";
    }
    break;

  case GMCT_TGLMP:
  case GMCT_OSBTN:
  case GMCT_TGBTN:
  case GMCT_BTCHG:
  case GMCT_MNCHG:
    if ((data.ViewType != CGBUTTON_BITMAP) || (data.CaptionOnImage != "0")) {
      data.ButtonText = data.Caption;
    }
    else {
      data.ButtonText = "";
    }
    break;

  } // switch (data.CtlType) {

  switch (data.ViewType) {
  case CGBUTTON_NORMAL:
  case CGBUTTON_PANEL:
  case CGBUTTON_CIRCLE:
  case CGBUTTON_FIXEDCIRCLE:
  case CGBUTTON_CHECKBOX:
  case CGBUTTON_LABEL:
  case CGBUTTON_TOGGLE:
  case CGBUTTON_F2:
  case CGBUTTON_F3:
  case CGBUTTON_F4:
  case CGBUTTON_F5:
  case CGBUTTON_F7:
  case CGBUTTON_F8:
  case CGBUTTON_F9:
  case CGBUTTON_F10:
    FAbButtonCompo_SetText(data, data.ButtonText);
    break;
  case CGBUTTON_BITMAP:
    if (data.CaptionOnImage != "0") {
      FAbButtonCompo_SetText(data, data.ButtonText);
    }
    break;
  }
} // SetCaption

// Update the caption based on value.
function UpdateCaption(data) {

  data.ButtonText = "";
  if (data.ValueText != "") {
    if ((data.DataType == CGRSTYPE_FIX) || (data.Caption == "")) {
      data.ButtonText = data.ValueText;
    }
    else {
      // Caption specifies the format string.
      var found = false;
      var lCaption = data.Caption;
      var pos = lCaption.indexOf("%");
      while (!found) {
        if (pos > -1) {
          switch (lCaption.charAt(pos+1)) {
          case "%":
            data.ButtonText += lCaption.substring(0, pos+1);
            lCaption = lCaption.substring(pos + 2); // Advance past extra % but continue looking for format specifier
            pos = lCaption.indexOf("%");
            continue;
          case "n":
            data.ButtonText += lCaption.substring(0, pos) + lCaption.substr(pos + 2);
            found = true;
            break;
          default:
            var formatSpecifier = lCaption.substr(pos);
            var pos2 = formatSpecifier.indexOf(" ");
            if (pos2 > -1) {
              formatSpecifier = lCaption.substring(pos, pos + pos2);
            }

            // @@@ V10TOBEFIXED Actually implement the formatSpecifer
            // For now just add the ValueText and skip over the format Specifier
            data.ButtonText += lCaption.substring(0, pos) + data.ValueText + lCaption.substr(pos + formatSpecifier.length);
            found = true;
            break;
          }
          data.ButtonText = data.ButtonText.replace("%%", "%");
        }
        else {
          // The Caption is ignored
          data.ButtonText = data.ValueText;
        }
        break; // always break out of while loop
      } // while !found
    }
  }
} // UpdateCaption

// Check that children widths are available.
// Must wait for computed properties before manipulating horizontal positions.
function CheckChildrenWidth(data) {
  var innerObj = data.$this.children();
  var ready = true;

  if (innerObj) {
    innerObj.each( function() {
      if (!($(this).width())) {
        ready = false;
      }
    });
  }

  if (ready) {
    switch (data.ViewType) {
      case CGBUTTON_CHECKBOX:
        var xOffset = ((data.$this.width() - 
                       data.$this.children("img").width() - 
                       data.$this.children("span").width()) / 2);
        if (xOffset && (typeof xOffset === 'number')) {
          if ((xOffset <= data.$this.width()) && xOffset >= 0) {
            data.$this.children("img").css({"left": "" + xOffset + "px"});
          }
        }
        break;

      case CGBUTTON_CIRCLE:
      case CGBUTTON_FIXEDCIRCLE:
        cgop_SetColors(data); // This will render the circle
        break;

      default:
        break;
    } 
  }
  else {
    setTimeout(function() { CheckChildrenWidth(data); }, 250);
  }
} // CheckChildrenWidth

// Calculate vertical alignment.
// Required for PanelView which does not know height until later
function ComputeVerticalAlignment(object, data) {
  if (object.height()) {
    object.css('position', 'relative');
    if (data.VAlign == 0) { // Top
      object.css('top', '0px');
    }
    else if (data.VAlign == 2) { // Bottom
      object.css('top', data.$this.height() - object.height() + 'px');
    }
    else {  // Middle
      object.css('top', data.$this.height()/2 - object.height()/2 + 'px');
    }
  }
  else {
    // Try again later when height is known.
    setTimeout(function() { ComputeVerticalAlignment(object, data); }, 250);
  }
} // ComputeVerticalAlignment

// Set the vertical alignment.
function SetVerticalAlignment(data) {
  var innerObj = data.$this.children();
  if ((innerObj) && (innerObj.text() != "")) {
    ComputeVerticalAlignment(innerObj, data);
  }
}

// Set the alignment.
function SetAlignment(data) {

  switch (data.ViewType) {
  case CGBUTTON_CHECKBOX:
  case CGBUTTON_CIRCLE:
  case CGBUTTON_FIXEDCIRCLE:
    FAbButtonCompo_CreateTableButton(data);
    break;

  case CGBUTTON_NORMAL:
  case CGBUTTON_PANEL:
  case CGBUTTON_LABEL:
  case CGBUTTON_TOGGLE:
    // Set Alignment.
    var horizontal = "center";
    if (data.HAlign == 0) {
      horizontal = "left";
    }
    else if (data.HAlign == 2) {
      horizontal = "right";
    }
    data.$this.css("text-align", horizontal);
    SetVerticalAlignment(data);
  }

} // SetAlignment

// Set parameter for the execution control.
function SetParameter(num, parameter, data) {
  if (!data.IsEcho) {
    if (!((parameter === '') || (parameter.length === 0))) {
      // Set $UI_PANEDATA[task_index].$PARAMETERn
      top.rpcmc_setVar(SYSNAME_C, "$UI_PANEDATA[" + data.fTaskIdx + "].$PARAMETER" + num, parameter);
    }
  }

} // SetParameter

function SetHelp(data) {
  if ((!data.IsEcho) && (data.PageName != "")) {
    top.rpcmc_setVar(SYSNAME_C, "$UI_PANEDATA[" + data.fTaskIdx + "].$HELPURL", data.PageName);
  }
} // SetHelp


// Create button
function CreateButton(data) {

  if (data.Invisible) {
    return;
  }
  data.$this.css("display", "inline-block");
  if (data.width.indexOf("%") >= 0) {
    data.$this.css("width", data.width);
  }
  else {
    data.$this.css("width", data.width + "px");
  }
  if (data.height.indexOf("%") >= 0) {
    data.$this.css("height", data.height);
  }
  else {
    data.$this.css("height", data.height + "px");
  }
  data.$this.css("vertical-align", "top");

  switch (data.ViewType) {
  case CGBUTTON_BITMAP:
    // Create a bitmap button.
    var out = '<img id="' + data.Id + '" src="' + data.FalseImage + '" width="' + data.width + '" height="' + data.height + '" />';
    data.$this.html(out);

    SetImages(data);
    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetFont(data);
    SetTgbtnEvent(data);
    break;

  case CGBUTTON_CHECKBOX:
    // Create a checkbox button.
    data.$this.css("display", "inline-table");
    var out = '<img id=\"' + data.Id + '\" src=\"' + data.FalseChkImg + '\" />';
    if (data.Caption) {
      out += '<span id=\"' + data.Id + 'chkCap\"></span>';
    }
    data.$this.html(out);

    SetImages(data);
    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetFont(data);
    SetTgbtnEvent(data);
    break;

  case CGBUTTON_CIRCLE:
  case CGBUTTON_FIXEDCIRCLE:
    var canvasSize = 15;
    if (data.ViewType == CGBUTTON_CIRCLE) {
      var canvasScale = 0.78;
      canvasSize = Math.round(data.height * canvasScale);
    }
    data.$this.css("display", "inline-table");
    var out = '<canvas id=\"' + data.Id + '\" width=\"' + canvasSize + '\" height=\"' + canvasSize + '\" />';
    if (data.Caption) {
      out += '<span id=\"' + data.Id + 'circleCap\"></span>';
    }
    data.$this.html(out);

    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetFont(data);
    SetTgbtnEvent(data);
    break;

  case CGBUTTON_MULTIBITMAP:
    // Create a bitmap button.
    var out = '<img id="' + data.Id + '" src="' + GetMultiString(data) + '" width="' + data.width + '" height="' + data.height + '" />';
    data.$this.html(out);

    SetImages(data);
    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetFont(data);
    break;

  case CGBUTTON_F2:
  case CGBUTTON_F3:
  case CGBUTTON_F4:
  case CGBUTTON_F5:
  case CGBUTTON_F7:
  case CGBUTTON_F8:
  case CGBUTTON_F9:
  case CGBUTTON_F10:
    // Tell FunctionKey plug-in to start our fkey (0 based).
    top.gPaneFKey[data.fDeviceId].FKeyListener[data.fKey-1] = data.$this;
    SetCaption(data);
    SetImages(data);
    cgop_SetColors(data);

    // Trigger a DeviceEvent on the fky event listeners.
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);

    SetTgbtnEvent(data);
    break;

  case CGBUTTON_POSITION:
    var out = null;

    out = '<table style="white-space:pre;">';
    out = out + '<tr><td colspan="3" id="title" text-align:left;"></td></tr>';
    // Add 1 for the config
    for (var idx = 0; idx < MAX_GRP_AXES+1; idx++) {
      out = out + '<tr>';
      out = out + '<td id="label' + idx + '"></td>';
      if (idx == 0) {
        out = out + '<td colspan="2" id="value' + idx + '"></td>';
      }
      else {
        out = out + '<td id="value' + idx + '"></td>';
        out = out + '<td id="unit'  + idx + '"></td>';
      }
      out = out + '</tr>';
    }
    out = out + '</table>';
    data.$this.html(out);
    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetAlignment(data);
    SetFont(data);
    break;

  default:
    // Create a button.
    var tab_enable = '';
    if (top.gPaneExec[data.fDeviceId].fFlags & PANE_FLAG_FOCUS_EVENTS_C) {
      tab_enable = " tabindex=\"0\"";
      data.$this.bind('focusin', data, HandleFocusEvent);
    }
    var out = '<span id="' + data.Id + '"' + tab_enable + '></span>';
    data.$this.html(out);

    if (data.ViewType == CGBUTTON_LABEL) {
      SetCaption(data);
    }

    cgop_SetBorder(data);
    cgop_SetColors(data);
    SetAlignment(data);
    SetFont(data);
    SetTgbtnEvent(data);

  } // switch data.ViewType
} // CreateButton

// Change the page.
function ChangePage(urlstr, data) {

  if (urlstr == "") {
    return;
  }
  var pos = urlstr.toLowerCase().indexOf("tpextreq");
  var funcstr;
  var args;
  if (pos >= 0) {
    // Special command, "tpextreq=request_code,key_code"
    pos = urlstr.indexOf("=");
    funcstr = urlstr.substr(pos + 1);
    args = funcstr.split(',');
    if (args[0] && args[1]) {
      top.rpcmc_tpextreq2(data.fDeviceId, args[0], args[1]);
    }
    return;
  }
  pos = urlstr.toLowerCase().indexOf("javascript:");
  if (pos >= 0) {
    // Execute javascript routine
    funcstr = urlstr.substr(pos + 11);
    args = funcstr.split('(');
    var fn = window[args[0]];
    if (typeof fn === 'function') {
      if (args[1]) {
        setTimeout(eval(funcstr));
      }
      else {
        setTimeout(fn);
      }
    }
    return;
  }

  urlstr = avutil_getRelativeURL(urlstr, data);

  // Show Document.
  if ((data.CtlType == GMCT_BTCHG) && (data.NoContent != "") && (data.NoContent != "0")) {
    // No response is sent back from this request.
    $.get(urlstr, function() { });
  }
  else if (data.FrameName == "cghide") {
    // Use hidden frame.  No response is sent back from this request.
    // showDocument2(urlstr, "cghide", data.PageName);
    top.cghide.window.location.href = urlstr;
  }
  else if (data.FrameName != "") {
    // showDocument2(urlstr, data.FrameName, data.PageName);
    // Recursively find the frame
    var ret_frame = top.find_frame(top, data.FrameName);
    if (ret_frame) {
      ret_frame.document.location.href = urlstr;
    }
    else {
      top.cghide.window.location.href = urlstr;
    }
  }
  else {
    // showDocument2(urlstr, null, data.PageName);
    window.location.href = urlstr;
  }

} // ChangePage

// Update Button based on data.ViewType.
function UpdateButton(data) {

  var sb_state;

  switch (data.ViewType) {
  case CGBUTTON_CIRCLE:
  case CGBUTTON_FIXEDCIRCLE:
  case CGBUTTON_BITMAP:
  case CGBUTTON_CHECKBOX:
    sb_state = GetButtonState(data);
    FAbButtonCompo_SetValue(data, sb_state);
    FAbButtonCompo_SetCaptionOnImage(data, data.CaptionOnImage);    
    if ((CGBUTTON_BITMAP == data.ViewType) || (CGBUTTON_CHECKBOX == data.ViewType)) {
      // Set images based on value.
      SetImages(data);
      if (data.CaptionOnImage != "0") {
        if (!data.Disable) {
          // Change ForeColor/Caption based on value.
          if (sb_state) {
            if (data.TrueStrColor != "") {
              FAbButtonCompo_SetTextColor(data, data.TrueStrColor);
            }
            if (data.TrueCaption != "") {
              FAbButtonCompo_SetText(data, data.TrueCaption);
           }
          } 
          else {
            if (data.FalseStrColor != "") {
              FAbButtonCompo_SetTextColor(data, data.FalseStrColor);
            }
            if (data.TrueCaption != "") {
              // Not actually using TrueCaption, but only have to change text if set.
              FAbButtonCompo_SetText(data, data.ButtonText);
            }
          }
        }
      } // if (data.CaptionOnImage != "0")
    }
    else if ((CGBUTTON_CIRCLE == data.ViewType) || (CGBUTTON_FIXEDCIRCLE == data.ViewType)) {
      if (!data.Disable) {
        var circleColor;
        
        if (sb_state) {
          circleColor = data.TrueColor;
        }
        else {
          circleColor = data.FalseColor;
        }
        
        if (circleColor == "") {
          circleColor = data.BackColor;
        }
        FAbButtonCompo_RenderCircle(data, circleColor);
      }
    }
    break;

  case CGBUTTON_TOGGLE:
    sb_state = GetButtonState(data);
    FAbButtonCompo_SetValue(data, sb_state);

    if (!data.Disable) {
      // Change ForeColor/BackColor/Caption based on value.
      if (sb_state) {
        if (data.TrueStrColor != "") {
          FAbButtonCompo_SetTextColor(data, data.TrueStrColor);
        }
        if (data.TrueColor != "") {
          FAbButtonCompo_SetBackColor(data, data.TrueColor);
        }
        if ((data.CtlType == GMCT_TGBTN) && (data.TrueCaption != "")) {
          FAbButtonCompo_SetText(data, data.TrueCaption);
        }
      }
      else {
        if (data.FalseStrColor != "") {
          FAbButtonCompo_SetTextColor(data, data.FalseStrColor);
        }
        if (data.FalseColor != "") {
          FAbButtonCompo_SetBackColor(data, data.FalseColor);
        }
        if ((data.CtlType == GMCT_TGBTN) && (data.TrueCpation != "")) {
          // Not actually using TrueCaption, but only have to change text if set.
          FAbButtonCompo_SetText(data, data.ButtonText);
        }
      }
    }
    break;

  case CGBUTTON_F2:
  case CGBUTTON_F3:
  case CGBUTTON_F4:
  case CGBUTTON_F5:
  case CGBUTTON_F7:
  case CGBUTTON_F8:
  case CGBUTTON_F9:
  case CGBUTTON_F10:
    switch (data.CtlType) {
    case GMCT_TGBTN:
      var sb_state = GetButtonState(data);
      FAbButtonCompo_SetValue(data, sb_state);

      if (!data.Disable) {
        // Change ForeColor/BackColor/Caption based on value.
        if (sb_state) {
          if (data.TrueStrColor != "") {
            FAbButtonCompo_SetTextColor(data, data.TrueStrColor);
          }
          if (data.TrueColor != "") {
            FAbButtonCompo_SetBackColor(data, data.TrueColor);
          }
          if (data.TrueCaption != "") {
            FAbButtonCompo_SetText(data, data.TrueCaption);
          }
        }
        else {
          if (data.FalseStrColor != "") {
            FAbButtonCompo_SetTextColor(data, data.FalseStrColor);
          }
          if (data.FalseColor != "") {
            FAbButtonCompo_SetBackColor(data, data.FalseColor);
          }
          if (data.TrueCaption != "") {
            // Not actually using TrueCaption, but only have to change text if set.
            FAbButtonCompo_SetText(data, data.ButtonText);
          }
        }
      }
      break;

    case GMCT_EDBOX:
    case GMCT_COMBOBOX:
      UpdateCaption(data);

      // Set label.
      FAbButtonCompo_SetText(data, data.ButtonText);
      break;

    default:
      break;

    } // switch (data.CtlType)

    // Trigger a DeviceEvent on the fky event listeners.
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
    break;

  case CGBUTTON_MULTIBITMAP:
    // Set images based on value.
    SetImages(data);
    break;

  case CGBUTTON_LABEL:
    // Change the button text based on value.
    data.ButtonText = GetMultiString(data);

    // Set label.
    FAbButtonCompo_SetText(data, data.ButtonText);
    break;

  case CGBUTTON_POSITION:
    if ((data.DataType == CGRSTYPE_KARELPOS) ||
        (data.DataType == CGRSTYPE_POSREG) ||
        (data.DataType == CGRSTYPE_TFRAME) ||
        (data.DataType == CGRSTYPE_JFRAME) ||
        (data.DataType == CGRSTYPE_UFRAME)) {
      data.ButtonText = data.ValueText;
      FAbButtonCompo_SetText(data, data.ButtonText);
    }
    break;

  case CGBUTTON_PANEL:
  default:

    if (data.CtlType == GMCT_TGLMP) {
      if (!data.Disable) {
        sb_state = GetButtonState(data);
        // Change ForeColor/BackColor/Caption based on value.
        if (sb_state) {
          if (data.TrueStrColor != "") {
            FAbButtonCompo_SetTextColor(data, data.TrueStrColor);
          }
          if (data.TrueColor != "") {
            FAbButtonCompo_SetBackColor(data, data.TrueColor);
          }
          if ((data.CtlType == GMCT_TGBTN) && (data.TrueCaption != "")) {
            FAbButtonCompo_SetText(data, data.TrueCaption);
          }
        }
        else {
          if (data.FalseStrColor != "") {
            FAbButtonCompo_SetTextColor(data, data.FalseStrColor);
          }
          if (data.FalseColor != "") {
            FAbButtonCompo_SetBackColor(data, data.FalseColor);
          }
          if ((data.CtlType == GMCT_TGBTN) && (data.TrueCaption != "")) {
            // Not actually using TrueCaption, but only have to change text if set.
            FAbButtonCompo_SetText(data, data.ButtonText);
          }
        }
      }
    }
    else {
      UpdateCaption(data);
  
      // Set label.
      FAbButtonCompo_SetText(data, data.ButtonText);
    }
    break;

  } // switch data.ViewType

} // UpdateButton


// Update Control.
function UpdateCtl(data) {

  switch (data.CtlType) {
  case GMCT_LABEL:
  case GMCT_EDBOX:
  case GMCT_TGLMP:
  case GMCT_MULTI:
    if (data.DataType == CGRSTYPE_FIX) {
      return;
    }
    UpdateButton(data);
    break;

  case GMCT_TGBTN:
    UpdateButton(data);
    break;

  case GMCT_ATCHG:
    if (data.DataType == CGRSTYPE_FIX) {
      return;
    }
    // Change the page based on value.
    var val_num;
    if ((data.DataType == CGRSTYPE_STRREG) && (data.ValueText)) {
      // Assume string register holds a numeric value.
      val_num = Number(data.ValueText);
    }
    else {
      val_num = data.ValueNumber;
    }
    if (data.DataDefault) {
      if ((!data.Disable) && 
          (!data.IsEcho) &&
          (data.CmpValue != val_num)) {
        // Value actually changed so ok to change page.
        ChangePage(GetMultiString(data), data);
      }
    }
    else {
      // Must receive data first time to enable control.
      data.DataDefault = true;
    }
    // Save data to compare to determine if it changed.
    data.CmpValue = val_num;
    break;

  case GMCT_COMBOBOX:
    UpdateButton(data);
    break;

  case GMCT_MNCHG:
    UpdateButton(data);
    break;

  default:
    break;

  } // switch (data.CtlType)

} // UpdateCtl

// Update Value.
function UpdateValue(val_str, data) {

  // Get text value based on integer for combobox.
  if (data.CtlType == GMCT_COMBOBOX) {
    if (isNaN(data.ValueNumber) ||
        (!isNaN(data.ValueNumber) && (data.ValueNumber <= 0))) {
      data.ValueNumber = NaN;
    }
    if (data.DataType == CGRSTYPE_STRREG) {
      // Assume string register holds a numeric value.
      data.ValueNumber = parseInt(val_str);
    }
    switch (data.Type) {
    case COMBOBOX_LISTTYPE:
      break;

    case COMBOBOX_PROGRAMTYPE:
      break;

    case COMBOBOX_VARIABLETYPE:
      // Get text value selected from tpxensbv_krl_ext.
      var value = data.ValueNumber;
      if (data.BoolType === "1") {
        value++; // enum values start at 1
      }
      if ((!isNaN(data.ValueNumber)) && (!(data.ValueText === val_str))) {
        data.ValueText = UNINIT_STR_C;
        setTimeout(top.rpcmc_tpxensbv_krl_text(data.fDeviceId, data.varinfo[1].prognam,
                                  data.varinfo[1].progvar, data.ValueNumber,
                                  HandleVarCallback, data));
        // Return since ValueText is received asynchronously, don't update yet.
        return;
      }
      else {
        UpdateValueToController(-2, data.ValueText, data);
      }
      break;

    case COMBOBOX_DICTIONARYTYPE:
      // Get text value selected from tpxensub_ext.

      if ((!isNaN(data.ValueNumber)) && (!(data.ValueText === val_str))) {
        data.ValueText = UNINIT_STR_C;
        setTimeout(GetValueFromController(-2, data)); // Async, add to message queue
        // Return since ValueText is received asynchronously, don't udpate yet.
        return;
      }
      else {
        UpdateValueToController(-2, data.ValueText, data);
      }
      break;
    case COMBOBOX_STRREGTYPE:
      // Get text value selected from string register.
      if ((!isNaN(data.ValueNumber)) && (!(data.ValueText === val_str))) {
        data.ValueText = UNINIT_STR_C;
        setTimeout(GetValueFromController(-3, data)); // Async, add to message queue
        // Return since ValueText is received asynchronously, don't update yet.
        return;
      }
      else {
        UpdateValueToController(-2, data.ValueText, data);
      }
      break;

    } // switch (data.Type)
  }

  data.ValueText = val_str;

  if (data.DataType == CGRSTYPE_DICTELE) {
    return;
  }
  if (data.IOSim) {
    if (val_str.toLowerCase() == "sim") {
      data.ValueNumber = 1;
    }
    else {
      data.ValueNumber = 0;
    }
  }
  else if (val_str.toLowerCase() == "on") {
    data.ValueNumber = 1;
  }
  else if (val_str.toLowerCase() == "off") {
    data.ValueNumber = 0;
  }
  else if (val_str.toLowerCase() == "true") {
    data.ValueNumber = 1;
  }
  else if (val_str.toLowerCase() == "false") {
    data.ValueNumber = 0;
  }
  else {
    if (!isNaN(Number(val_str))) {
      if (data.Type != COMBOBOX_STRREGTYPE) {
        data.ValueNumber = Number(val_str);
      }
    }
  }

} // UpdateValue

// Replace Format Specifier.
function ReplaceFormat(data) {

} // ReplaceFormat

// Add form caption to menu string.
function AddFormCaption(num, PageCaption) {
  var str = '';
  if (PageCaption != "") {
    str = '{"lin":"' + num + '","buf":"' + PageCaption + '","spid":"0","scid":"0","dis":"0"},';
  }
  return str;

} // AddFormCaption

// Initialize or uninitialize pipe event.
function InitPipeEvent(init, data) { 
  // Notify PMON to start/stop monitoring our pipe
  if (init) {
    top.jQuery.filelis.bind('PipeEvent', data, HandlePipeEvent); // Attach handler for PipeEvent
    top.rpcmc_startPipeMonitor(data.Pipe, data.Interval); // Start PMON monitor for our pipe
  }
  else {
    top.rpcmc_stopPipeMonitor(data.Pipe); // Stop PMON monitor for our pipe
    top.jQuery.filelis.unbind('PipeEvent', HandlePipeEvent); // Detach handler for PipeEvent.
  }
  
} // InitPipeEvent

// Initialize or uninitialize events for each type.
function InitRobotEvents(init, data) {

  if (data.Interval < 0) {
    data.Interval = 250;
  }

  if ((data.Invisible) || (data.Disable)) {
    return;
  }

  if (data.fKey) {
    if (init) {
      top.jQuery.fkylis.bind('FKeyClickEvent', data, cgop_HandleFKeyClickEvent); // Attach handler for FKeyClickEvent.
    }
    else {
      top.jQuery.fkylis.unbind('FKeyClickEvent', cgop_HandleFKeyClickEvent); // Detach handler for FKeyClickEvent.
    }
  }

  // Verify Control is supposed to monitor data.
  switch (data.CtlType) {
  case GMCT_OSBTN:
  case GMCT_BTCHG:
  case GMCT_MNCHG:
    return;
  }

  switch (data.DataType) {
  case CGRSTYPE_FIX:
    // Add var listener for terminating dialog PREV case, but NO var monitor
    if (init) {
      if (!data.VarEvent) {
        top.jQuery.varlis.bind('VarEvent', data, cgop_HandleVarEvent); // Attach handler for VarEvent.
      }
      data.VarEvent++;
    }
    else {
      data.VarEvent--;
      if (!data.VarEvent) {
        top.jQuery.varlis.unbind('VarEvent', cgop_HandleVarEvent); // Detach handler for VarEvent.
      }
    }
    break;

  case CGRSTYPE_REG:
  case CGRSTYPE_STRREG:
  case CGRSTYPE_SYSVAR:
  case CGRSTYPE_KARELVAR:
  case CGRSTYPE_KARELPOS:
  case CGRSTYPE_POSREG:
  case CGRSTYPE_TFRAME:
  case CGRSTYPE_JFRAME:
  case CGRSTYPE_UFRAME:
    // Notify PMON to start/stop monitoring our DataType.
    if (init) {
      if (!data.VarEvent) {
        top.jQuery.varlis.bind('VarEvent', data, cgop_HandleVarEvent); // Attach handler for VarEvent.
      }
      if (!ContainsIndirectTag(data.varinfo[0].prognam, data.varinfo[0].progvar)) {
      data.VarEvent++;
      top.rpcmc_startVarMonitor(data.varinfo[0].prognam, data.varinfo[0].progvar,
                            data.Interval); // Start PMON monitors for vars.
    }
    }
    else {
      top.rpcmc_stopVarMonitor(data.varinfo[0].prognam, data.varinfo[0].progvar); // Stop PMON monitors for vars.
      data.VarEvent--;
      if (!data.VarEvent) {
        top.jQuery.varlis.unbind('VarEvent', cgop_HandleVarEvent); // Detach handler for VarEvent.
      }
    }
    break;

  case CGRSTYPE_DICTELE:
    if (init) {
      if (data.DictName != "") {
        top.rpcmc_dpread2(data.DictName, data.DictEle, HandleDictCallback, data);
      }
    }
    break;

  case CGRSTYPE_CURPOS:
    if (init) {
      top.jQuery.curposlis.bind('CurposEvent', data, cgop_HandleCurposEvent); // Attach handler for CurposEvent.
      if (!ContainsIndirectTag(data.MonType, data.DataIndex)) {
      top.rpcmc_startCurposMonitor(data.MonType, data.Interval); // Start PMON monitors for curpos.
    }
    }
    else {
      if (!ContainsIndirectTag(data.MonType, data.DataIndex)) {
      top.rpcmc_stopCurposMonitor(data.MonType); // Stop PMON monitors for curpos.
      }
      top.jQuery.curposlis.unbind('CurposEvent', cgop_HandleCurposEvent); // Detach handler for CurposEvent.
    }
    break;

  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_AI:
  case CGRSTYPE_AO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_GI:
  case CGRSTYPE_GO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_MIN:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:
    if (data.IOSim) {
      if (init) {
        // I/O Simulation status is not monitored
        // so must update value and control manually.
        // We will get unsolicited events when it changes.
        // This will get new value from controller.
        GetValueFromController(-1, data);

        top.jQuery.iolis.bind('IOSimEvent', data, cgop_HandleIOSimEvent); // Attach handler for IOSimEvent.
        top.jQuery.iolis.bind('IOSimAllEvent', data, cgop_HandleIOSimAllEvent); // Attach handler for IOSimAllEvent.
      }
      else {
        top.jQuery.iolis.unbind('IOSimEvent', cgop_HandleIOSimEvent); // Detach handler for IOSimEvent.
        top.jQuery.iolis.unbind('IOSimAllEvent', cgop_HandleIOSimAllEvent); // Detach handler for IOSimAllEvent.
      }
      break;
    }
    if (init) {
      top.jQuery.iolis.bind('IOBooleanEvent', data, cgop_HandleIOEvent); // Attach handler for IOBooleanEvent.
      if (!ContainsIndirectTag(data.DataType, data.DataIndex)) {
      top.rpcmc_startIOMonitor(data.DataType, data.DataIndex); // Start PMON monitor for the I/O port.
    }
    }
    else {
      top.rpcmc_stopIOMonitor(data.IOType, data.IOIndex); // Stop PMON monitor for the I/O port.
      top.jQuery.iolis.unbind('IOBooleanEvent', cgop_HandleIOEvent); // Detach handler for IOBooleanEvent.
    }
    break;
  } // switch(data.DataType)

} // InitRobotEvents

//  Initialize or uninitialize events for each indirection.
function InitIndirectEvents(init, data) {

  if ((data.Invisible) || (data.Disable)) {
    return;
  }

  for (var si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
    l_DataType = data.IDataType[si_ind];
    l_DataIndex = data.IDataIndex[si_ind];

    if (!l_DataType || !l_DataIndex) {
      continue;
    }
    // Prepare all the necessary data to access DataType/DataIndex from the controller.
    if (init) {
    PrepareCtl(l_DataType, l_DataIndex, data, PREP_NORMAL_C, si_ind, data.varinfo[0]);
    }

    switch (l_DataType) {
    case CGRSTYPE_REG:
    case CGRSTYPE_STRREG:
    case CGRSTYPE_SYSVAR:
    case CGRSTYPE_KARELVAR:
      // Notify PMON to start/stop monitoring our DataType.
      if (init) {
        if (!data.VarEvent) {
          top.jQuery.varlis.bind('VarEvent', data, cgop_HandleVarEvent); // Attach handler for VarEvent.
        }
        if (!ContainsIndirectTag(data.Istr1[si_ind], data.Istr2[si_ind])) {
        data.VarEvent++;
        top.rpcmc_startVarMonitor(data.Istr1[si_ind], data.Istr2[si_ind], data.Interval); // Start PMON monitors for vars.
      }
      }
      else {
        top.rpcmc_stopVarMonitor(data.Istr1[si_ind], data.Istr2[si_ind]); // Stop PMON monitors for vars.
        data.VarEvent--;
        if (!data.VarEvent) {
          top.jQuery.varlis.unbind('VarEvent', cgop_HandleVarEvent); // Detach handler for VarEvent.
        }
      }
      break;

    case CGRSTYPE_DI:
    case CGRSTYPE_DO:
    case CGRSTYPE_AI:
    case CGRSTYPE_AO:
    case CGRSTYPE_PI:
    case CGRSTYPE_PO:
    case CGRSTYPE_RI:
    case CGRSTYPE_RO:
    case CGRSTYPE_SI:
    case CGRSTYPE_SO:
    case CGRSTYPE_TPIN:
    case CGRSTYPE_TPOUT:
    case CGRSTYPE_WI:
    case CGRSTYPE_WO:
    case CGRSTYPE_GI:
    case CGRSTYPE_GO:
    case CGRSTYPE_UI:
    case CGRSTYPE_UO:
    case CGRSTYPE_LDI:
    case CGRSTYPE_LDO:
    case CGRSTYPE_LAI:
    case CGRSTYPE_LAO:
    case CGRSTYPE_WSI:
    case CGRSTYPE_WSO:
    case CGRSTYPE_MIB:
    case CGRSTYPE_MIN:
    case CGRSTYPE_FLAG:
    case CGRSTYPE_MARKER:
      if (init) {
        top.jQuery.iolis.bind('IOBooleanEvent', data, cgop_HandleIOEvent); // Attach handler for IOBooleanEvent.
        if (!ContainsIndirectTag(data.IDataType[si_ind], data.IDataIndex[si_ind])) {
        top.rpcmc_startIOMonitor(data.IDataType[si_ind], data.IDataIndex[si_ind]); // Start PMON monitor for the I/O port.
      }
      }
      else {
        top.rpcmc_stopIOMonitor(data.Istr1[si_ind], data.Istr2[si_ind]); // Stop PMON monitor for the I/O port.
        top.jQuery.iolis.unbind('IOBooleanEvent', cgop_HandleIOEvent); // Detach handler for IOBooleanEvent.
      }
      break;
    } // switch(data.DataType)
  } // for (var si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {

} // InitIndirectEvents

// Check if given control supports property of "Check Condition" tab.
function chkkey_supported(data) {
  if( (data.CtlType == GMCT_EDBOX) ||
      (data.CtlType == GMCT_OSBTN) || 
      (data.CtlType == GMCT_TGBTN) ||
      (data.CtlType == GMCT_MNCHG) || 
      (data.CtlType == GMCT_BTCHG) || 
      (data.CtlType == GMCT_COMBOBOX)) {
    return true;
  }
  else {
    return false;
  }
} // chkkey_supported

// Check if this is connection as iPendant.
// Buttons with chkkey is enabled can be pressed               
// only by iPendant connection                                 
function chkkey_cgtp(data) {
  if ((1==data.fTaskIdx) || (2==data.fTaskIdx) || (3==data.fTaskIdx)) {
    return true;
  }
  else {
    return false;
  }
} // chkkey_cgtp

// If specified check is not successful, string to display is returned.
// This function should be called only if one of chkkey property
//  (chkDeadman,chkTpEnable,chkShift or chkIO) is true.
function chkkey_check(data) {

  if (!chkkey_cgtp(data)) {
    // This is not display by iPendant only.
    chkkey_message(data, top.srvap_getString("CHKBTN_IPENDANTONLY_C"));
  }

  // Check for "TP Deadman"
  if (data.ChkDeadman != "0") {
    // Check TP ESTOP
    top.rpcmc_chkkey(tpin_type_c, estp_c, null, null, HandleChkkeyCallback, data);

    // Check TP Enable
    top.rpcmc_chkkey(tpin_type_c, tp_enbl_c, null, null, HandleChkkeyCallback, data);

    // Get $MOR.$SAFETY_STAT
    top.rpcmc_chkkey(tpin_type_c, mor_ss_c, SYSNAME_C, '$MOR.$SAFETY_STAT', HandleChkkeyCallback, data);

  }
  else if (data.ChkTpEnable != "0") {
    // TP Enable key
    top.rpcmc_chkkey(tpin_type_c, tp_enbl_c, null, null, HandleChkkeyCallback, data);
  }

  // SHIFT KEY
  if (data.ChkShift != "0") {
    top.rpcmc_chkkey(tpin_type_c, shft_c, null, null, HandleChkkeyCallback, data);
  }

  // Specified I/O
  if (data.ChkIo != "0") {
    top.rpcmc_chkkey(data.ChkIoType, data.ChkIoIdx, null, null, HandleChkkeyCallback, data);
  }
  return;
} // chkkey_check

// Replace indirect data.
function ProcessIndirect(data) {
  var si_ind;

  $.each(data, function( argn, argv ) { 
    if (typeof argv !== 'string') {
      return;
    }
    // Name start with IndirectData?
    if (argn.substring(0,8).toLowerCase() === "indirect") {
      for (si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
        if (Number(argn.substring(8,9)) == si_ind+1) {
          if (argn.toLowerCase().indexOf("datatype") >= 0) {
            data.IDataType[si_ind] = parseInt(argv);
          }
          else if (argn.toLowerCase().indexOf("dataindex") >= 0) {
            IndirectReplace(data);
            data.IDataIndex[si_ind] = data[argn];
          }
          break;
        }
      }
    }
  });
} // ProcessIndirect

// Update Parameters (result of pipe event)
function UpdateParams(data, cgopPipeData) {
  var idx = 0;
  var property_name = '';
  var property_val = 0;
  var scratch_var = 0; // volatile variable for paramter processing

  for (idx = 0; idx < cgopPipeData.length; idx++) {
    for (property_name in cgopPipeData[idx]) {

      property_val = cgopPipeData[idx][property_name];

      switch (property_name.toLowerCase()) {
      case ANZ_HEIGHT:
        if (data.height !== property_val) {
          data.height = property_val;
          data.repaint |= mCgop_paintResize;
        }
        break;

      case ANZ_LEFT:
      case ANZ_TOP:
        break;

      case ANZ_WIDTH:
        if (data.width !== property_val) {
          data.width = property_val;
          data.repaint |= mCgop_paintResize;
        }
        break;

      case ANZ_CAPTION:
        if (data.Caption !== property_val) {
          data.Caption = property_val;
          SetCaption(data);
          if (data.fKey) {
            data.repaint |= mCgop_paintRecreate;
          }
          data.repaint |= mCgop_paintUpdate;
        }
        break;

      case ANZ_TRUECAPTION:
        data.TrueCaption = property_val;
        data.repaint |= mCgop_paintUpdate;
        break;

      case ANZ_HALIGN:
        data.HAlign = property_val;
        SetAlignment(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_VALIGN:
        data.VAlign = property_val;
        SetAlignment(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FORECOLOR:
        if (!property_val) {
          data.ForeColor = data.TextColor;
        }
        else {
          if (isNaN(property_val)) {
            data.ForeColor = property_val;
          }
          else {
            data.ForeColor = translateColor(property_val);
          }
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_BACKCOLOR:
        if (!property_val) {
          data.BackColor = data.InvisibleColor;
        }
        else {
          if (isNaN(property_val)) {
            data.BackColor = property_val;
          }
          else {
            data.BackColor = translateColor(property_val);
          }
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_FONT_NAME:
        data.FontName = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_SIZE:
        data.FontSize = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_BOLD:
        data.FontBold = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_ITALIC:
        data.FontItalic = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_WEIGHT:
        data.FontWeight = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_UNDERLINE:
        data.FontUnderline = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_FONT_STRIKETHROUGH:
        data.FontStrikethrough = property_val;
        SetFont(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_TAB_INDEX:
      case ANZ_INDEX:
        break;

      case ANZ_CGDATATYPE:
        if (!(data.repaint & mCgop_paintReinit)) {
          // Uninitialize the current events first.
          InitRobotEvents(false, data);
          data.repaint |= mCgop_paintReinit;
        }
        data.DataType = parseInt(property_val);
        break;

      case ANZ_CGDATAINDEX:
        if (!(data.repaint & mCgop_paintReinit)) {
          if (data.DataIndex === property_val) {
            break; // no changes
          }
          // Uninitialize the current events first.
          InitRobotEvents(false, data);
          data.repaint |= mCgop_paintReinit;
        }
        data.DataIndex = property_val;
        break;

      case ANZ_CGBORDER:
        data.Border = property_val;
        cgop_SetBorder(data);
        data.repaint |= mCgop_paintButton;
        break;

      case ANZ_CGTYPE:
        if ((data.CtlType == GMCT_COMBOBOX) &&
            !(data.repaint & mCgop_paintReinit)) {
          // Changing COmbobox type!
          // Uninitialize the current events first.
          InitRobotEvents(false, data);
          data.repaint |= mCgop_paintReinit;
          data.repaint |= mCgop_paintDataSource;
          data.repaint |= mCgop_paintTextType;
        }
        data.Type = property_val;
        if (data.CtlType == GMCT_EXECUTION) {
          // Put type in globals so other controls can access it.
          gPaneExec[data.fDeviceId].fExecType = data.Type;
        }
        else if ((data.CtlType == GMCT_TGLMP) ||
                 (data.CtlType == GMCT_TGBTN)) {
          data.repaint |= mCgop_paintUpdate;
        }
        break;

      case ANZ_CGVIEWTYPE:
        if (!(data.repaint & mCgop_paintReinit)) {
          // Uninitialize the current events first.
          InitRobotEvents(false, data);
          data.repaint |= mCgop_paintReinit;
        }
        data.ViewType = property_val;
        data.repaint |= mCgop_paintRecreate;
        break;

      case ANZ_CGTRUEBMP:
      case ANZ_CGTRUEIMAGE:
        if (!property_val) {
          data.TrueImage = "";
        }
        else {
          data.TrueImage = property_val;
        }
        if (data.fKey) {
          data.TrueImage = avutil_getRelativeURL(data.FalseImage, data);
          data.repaint |= mCgop_paintRecreate;
        }
        data.repaint |= mCgop_paintImages;
        break;

      case ANZ_CGFALSEBMP:
      case ANZ_CGFALSEIMAGE:
        if (!property_val) {
          data.FalseImage = "";
        }
        else {
          data.FalseImage = property_val;
        }
        if (data.fKey) {
          data.FalseImage = avutil_getRelativeURL(data.FalseImage, data);
          data.repaint |= mCgop_paintRecreate;
        }
        data.repaint |= mCgop_paintImages;
        break;

      case ANZ_CGTRUECOLOR:
        if (!property_val) {
          data.TrueColor = data.InvisibleColor;
        }
        else {
          if (isNaN(property_val)) {
            data.TrueColor = property_val;
          }
          else {
            data.TrueColor = translateColor(property_val);
          }
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_CGFALSECOLOR:
        if (!property_val) {
          data.FalseColor = data.InvisibleColor;
        }
        else {
          if (isNaN(property_val)) {
            data.FalseColor = property_val;
          }
          else {
            data.FalseColor = translateColor(property_val);
          }
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_CGTRUESTRCOLOR:
        if (isNaN(property_val)) {
          data.TrueStrColor = property_val;
        }
        else {
          data.TrueStrColor = translateColor(property_val);
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_CGFALSESTRCOLOR:
        if (isNaN(property_val)) {
          data.FalseStrColor = property_val;
        }
        else {
          data.FalseStrColor = translateColor(property_val);
        }
        data.repaint |= mCgop_paintColors;
        break;

      case ANZ_CGTRUEVALUE:
        data.TrueValue = property_val;
        data.repaint |= mCgop_paintUpdate;
        break;

      case ANZ_CGFALSEVALUE:
        data.FalseValue = property_val;
        data.repaint |= mCgop_paintUpdate;
        break;

      case ANZ_CGSETVALUE:
        data.SetValue = property_str;
        // No need to repaint
        break;

      case ANZ_CGOTHERPHASE:
      case ANZ_CGCMPVALUE:
      case ANZ_CGCMPOPERATOR:
      case ANZ_CGFORMNAME:
      case ANZ_CGFRAME:
      case ANZ_CGFORMNAME01:
      case ANZ_CGFORMNAME02:
      case ANZ_CGFORMNAME03:
      case ANZ_CGFORMNAME04:
      case ANZ_CGFORMNAME05:
      case ANZ_CGFORMNAME06:
      case ANZ_CGFORMNAME07:
      case ANZ_CGFORMNAME08:
      case ANZ_CGFORMNAME09:
      case ANZ_CGFORMNAME10:
      case ANZ_CGFORMCAPTION01:
      case ANZ_CGFORMCAPTION02:
      case ANZ_CGFORMCAPTION03:
      case ANZ_CGFORMCAPTION04:
      case ANZ_CGFORMCAPTION05:
      case ANZ_CGFORMCAPTION06:
      case ANZ_CGFORMCAPTION07:
      case ANZ_CGFORMCAPTION08:
      case ANZ_CGFORMCAPTION09:
      case ANZ_CGFORMCAPTION10:
      case ANZ_CGDATADEFAULT:
      case ANZ_CGDATA01:
      case ANZ_CGDATA02:
      case ANZ_CGDATA03:
      case ANZ_CGDATA04:
      case ANZ_CGDATA05:
      case ANZ_CGDATA06:
      case ANZ_CGDATA07:
      case ANZ_CGDATA08:
      case ANZ_CGDATA09:
      case ANZ_CGDATA10:
      case ANZ_CGVALUEMIN01:
      case ANZ_CGVALUEMIN02:
      case ANZ_CGVALUEMIN03:
      case ANZ_CGVALUEMIN04:
      case ANZ_CGVALUEMIN05:
      case ANZ_CGVALUEMIN06:
      case ANZ_CGVALUEMIN07:
      case ANZ_CGVALUEMIN08:
      case ANZ_CGVALUEMIN09:
      case ANZ_CGVALUEMIN10:
      case ANZ_CGVALUEMAX01:
      case ANZ_CGVALUEMAX02:
      case ANZ_CGVALUEMAX03:
      case ANZ_CGVALUEMAX04:
      case ANZ_CGVALUEMAX05:
      case ANZ_CGVALUEMAX06:
      case ANZ_CGVALUEMAX07:
      case ANZ_CGVALUEMAX08:
      case ANZ_CGVALUEMAX09:
      case ANZ_CGVALUEMAX10:
        break;

      case ANZ_INTERVAL:
      case ANZ_PERIODIC:
      case ANZ_PULSE:
      case ANZ_PULSETIME:
      case ANZ_NAME:
      case ANZ_PARAMETER1:
      case ANZ_PARAMETER2:
      case ANZ_PARAMETER3:
      case ANZ_PARAMETER4:
      case ANZ_PARAMETER5:
      case ANZ_PARAMETER6:
      case ANZ_PARAMETER7:
      case ANZ_PARAMETER8:
      case ANZ_PROGTYPE:
        break;

      case ANZ_DATASOURCE:
        // DataSource only applies to ComboBox Control
        data.DataSource = property_val;
        if (data.CtlType == GMCT_COMBOBOX) {
          data.repaint |= mCgop_paintDataSource;
        }
        break;

      case ANZ_TEXTTYPE:
        // TextType only applies to ComboBox Control
        if (undefined != property_val) {
          data.TextType = parseInt(property_val);
          if (data.CtlType == GMCT_COMBOBOX) {
            data.repaint |= mCgop_paintTextType;
          }
        }
        break;

      case ANZ_TEXTINDEX:
        // TextIndex only applies to ComboBox Control
        data.TextIndex = property_val;
        if (data.CtlType == GMCT_COMBOBOX) {
          data.repaint |= mCgop_paintTextType;
        }
        break;

      case ANZ_CLICKEVENTS:
      case ANZ_CHANGEEVENTS:
      case ANZ_FOCUSEVENTS:
      case ANZ_IGNOREKEYSEVENTS:
      case ANZ_IOSIM:
      case ANZ_PIPE:
      case ANZ_READONLY:
        break;

      case ANZ_INVISIBLE:
      case ANZ_DISABLE:
        switch (property_val.toLowerCase()) {
        case "1":
        case "on":
        case "true":
          scratch_var = true;
          break;

        case "0":
        case "off":
        case "false":
          scratch_var = false;

        default:
          scratch_var = null;
          break;
        }

        if (scratch_var == null) {
          break; // Do nothing if not changed
        }

        if (property_name.toLowerCase() == ANZ_INVISIBLE) {
          if (scratch_var === data.Invisible) {
            break; // Do nothing if not changed
          }
  
          data.Invisible = scratch_var
        }
        else if (property_name.toLowerCase() == ANZ_DISABLE) {
          if (scratch_var === data.Disable) {
            break; // Do nothing if not changed
          }

          data.Disable = property_val;

          if (data.CtlType == GMCT_HELP) {
            SetHelp(data);
          }
        }

        if (!(data.repaint & mCgop_paintReinit)) {
          // Uninitialize the current events first.
          InitRobotEvents(false, data);
          data.repaint |= mCgop_paintReinit;
        }
  
        data.repaint |= mCgop_paintRecreate;
        break; // ANZ_INVISIBLE || ANZ_DISABLE

      case ANZ_POSALIGN:
        break;

      case ANZ_GROUPNUM:
        break;

      case ANZ_OVERRIDE_RO:
      case ANZ_RWACCESS:
        break;

      case ANZ_NOCONTENT:
        break;

      case ANZ_MOMENTARY:
        break;

      case ANZ_VALUESTR:
        break;

      case ANZ_TRUEVALUESTR:
        break;

      case ANZ_INCRVALUE:
      case ANZ_CONFIRM:
        break;

      case ANZ_INVERT:
        break;

      case ANZ_SETFOCUS:
      case ANZ_CLICKME:
        break;

      case ANZ_CHKDEADMAN:
        data.ChkDeadman = property_val;
        break;

      case ANZ_CHKTPENABLE:
        data.ChkTpEnable = property_val;
        break;

      case ANZ_CHKSHIFT:
        data.ChkShift = property_val;
        break;

      case ANZ_CHKIO:
        data.ChkIo = property_val;
        break;

      case ANZ_CHKIOTYPE:
        data.ChkIoType = property_val;
        break;

      case ANZ_CHKIOIDX:
        data.ChkIoIdx = property_val;
        break;

      case ANZ_TRUEFONT:
        data.TrueFont = property_val;
        SetFont(data);
        data.repaint |= mCgop_repaintButton;
        break;

      case ANZ_CAPTIONONIMG:
      case ANZ_NOSCALE:
        break;

      case ANZ_EXECCONNECTID:
      case ANZ_FEXECCONNECTID:
        // Put connect_id in globals so other controls can access it.
        top.gPaneExec[data.fDeviceId].fExecConnectId = property_val;
        break;

      default:
        break;
      }
    }
  }
  UpdateNewParams(data);
} // UpdateParams

// Update new parameters for the control (result of pipe event)
function UpdateNewParams(data) {

  if (data.repaint & mCgop_paintResize) {
    // Resize and recreate the control
    CreateButton(data);
  }
  else if (data.repaint & mCgop_paintRecreate) {
    // Recreate the control
    CreateButton(data);
  }

  if (data.repaint & mCgop_paintReinit) {
    if (data.DataIndex) {
      PrepareCtl(data.DataType, data.DataIndex, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[0]);
    }
    // Reinitialize the Robot Events
    InitRobotEvents(true, data);

    // Reset caption.
    SetCaption(data);
    data.repaint |= mCgop_paintUpdate;
  }

  if (data.repaint & mCgop_paintTextType) {
    if ((data.Type != COMBOBOX_PROGRAMTYPE) &&
        (data.Type != COMBOBOX_FILETYPE) &&
        (data.TextIndex) && (!(data.TextIndex === ""))) {
      PrepareCtl(data.TextType, data.TextIndex, data, PREP_NORMAL_C, DIRECT_C, data.varinfo[2]);
    }
  }

  if (data.repaint & mCgop_paintDataSource) {
    if ((data.Type == COMBOBOX_VARIABLETYPE) &&
        (data.DataSource)) {
      PrepareCtl(CGRSTYPE_KARELVAR, data.DataSource, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[1]);
    }
    else if ((data.Type == COMBOBOX_STRREGTYPE) &&
             (data.DataSource)) {
      PrepareCtl(CGRSTYPE_STRREG, data.DataSource, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[1]);
    }
    else if (data.Type == COMBOBOX_DICTIONARYTYPE) {
      // This will get new value from controller.
      UpdateValue("", data);
      data.repaint |= mCgop_paintUpdate;
    }
  }

  if (data.repaint & mCgop_paintImages) {
    // Reset the images.
    SetImages(data);
  }

  if (data.repaint & mCgop_paintColors) {
    // Reset the colors.
    cgop_SetColors(data);
    data.repaint |= mCgop_paintUpdate;
  }

  if (data.repaint & mCgop_paintUpdate) {
    UpdateCtl(data);
  }
  else if (data.repaint & mCgop_paintButton) {
    // noop in HTML5
  }
} // UpdateNewParams

// Process the parameters until a new Id is found.
function ProcessParams(data) {

  cgop_FixStrings(data);

  IndirectReplace(data);

  data.DataType = parseInt(data.DataType);
  if (undefined != data.TextType) {
    data.TextType = parseInt(data.TextType);
  }
  if (undefined != data.IOSim) {
    data.IOSim = parseInt(data.IOSim);
  }
  if (data.BackColor == "") {
    data.BackColor = data.InvisibleColor;
  }
  else {
    data.BackColor = translateColor(data.BackColor);
  }
  if (data.ForeColor == "") {
    data.ForeColor = data.TextColor;
  }
  else {
    data.ForeColor = translateColor(data.ForeColor);
  }
  if (undefined == data.TrueColor) {
    data.TrueColor = "";
  }
  else if (data.TrueColor != "") {
    data.TrueColor = translateColor(data.TrueColor);
  }
  if (undefined == data.FalseColor) {
    data.FalseColor = "";
  }
  else if (data.FalseColor != "") {
    data.FalseColor = translateColor(data.FalseColor);
  }
  if (undefined == data.TrueStrColor) {
    data.TrueStrColor = "";
  }
  else if (data.TrueStrColor != "") {
    data.TrueStrColor = translateColor(data.TrueStrColor);
  }
  if (undefined == data.FalseStrColor) {
    data.FalseStrColor = "";
  }
  else if (data.FalseStrColor != "") {
    data.FalseStrColor = translateColor(data.FalseStrColor);
  }
  if (data.fKey) {
    if (undefined == data.FalseImage) {
      data.FalseImage = "";
    }
    else if (data.FalseImage != "") {
      // jfunckeys will be loading so require full path
      data.FalseImage = avutil_getRelativeURL(data.FalseImage, data);
    }
    if (undefined == data.TrueImage) {
      data.TrueImage = "";
    }
    else if (data.TrueImage != "") {
      // jfunckeys will be loading so require full path
      data.TrueImage = avutil_getRelativeURL(data.TrueImage, data);
    }
  }
  if (data.CtlType == GMCT_EXECUTION) {
    // globals
    top.gPaneExec[data.fDeviceId].fTaskIdx = data.fTaskIdx;
    top.gPaneExec[data.fDeviceId].fExecType = data.Type;
    top.gPaneExec[data.fDeviceId].fExecName = data.Name;
    top.gPaneExec[data.fDeviceId].fExecConnectId = top.g_connect_id;

    if (parseInt(data.ClickEvents) != 0) {
      top.gPaneExec[data.fDeviceId].fFlags |= PANE_FLAG_CLICK_EVENTS_C;
    }
    else {
      top.gPaneExec[data.fDeviceId].fFlags &= (~PANE_FLAG_CLICK_EVENTS_C);
    }

    if (parseInt(data.ChangeEvents) != 0) {
      top.gPaneExec[data.fDeviceId].fFlags |= PANE_FLAG_CHANGE_EVENTS_C;
    }
    else {
      top.gPaneExec[data.fDeviceId].fFlags &= (~PANE_FLAG_CHANGE_EVENTS_C);
    }

    if (parseInt(data.FocusEvents) != 0) {
      top.gPaneExec[data.fDeviceId].fFlags |= PANE_FLAG_FOCUS_EVENTS_C;
    }
    else {
      top.gPaneExec[data.fDeviceId].fFlags &= (~PANE_FLAG_FOCUS_EVENTS_C);
    }

    if (parseInt(data.IgnoreKeysEvents) != 0) {
      top.gPaneExec[data.fDeviceId].fFlags |= PANE_FLAG_IGNORE_KEYS_EVENTS_C;
    }
    else {
      top.gPaneExec[data.fDeviceId].fFlags &= (~PANE_FLAG_IGNORE_KEYS_EVENTS_C);
    }
  }

  chkkey_init(data);

} // ProcessParams

// Handle Indirect events.
function ProcessIndirectEvents(data, str1, str2, str3, val_str, p_ind) {

  if ((p_ind < 0) || (p_ind >= MAX_INDIRECT_C)) {
    return;
  }
  if (data.IDataIndex[p_ind] == "") {
    return;
  }

  if ((data.Istr1[p_ind] != str1) || (data.Istr2[p_ind] != str2) || (data.Istr3[p_ind] != str3)) {
    return;
  }

  // Reprocess any parameters that use indirection including indirect parameters themselves!
  var l_pos = 0;
  var l_p_ind = p_ind;

  // Update indirect value
  switch (data.IDataType[p_ind]) {
  case CGRSTYPE_DI:
  case CGRSTYPE_DO:
  case CGRSTYPE_PI:
  case CGRSTYPE_PO:
  case CGRSTYPE_RI:
  case CGRSTYPE_RO:
  case CGRSTYPE_SI:
  case CGRSTYPE_SO:
  case CGRSTYPE_TPIN:
  case CGRSTYPE_TPOUT:
  case CGRSTYPE_WI:
  case CGRSTYPE_WO:
  case CGRSTYPE_UI:
  case CGRSTYPE_UO:
  case CGRSTYPE_LDI:
  case CGRSTYPE_LDO:
  case CGRSTYPE_LAI:
  case CGRSTYPE_LAO:
  case CGRSTYPE_WSI:
  case CGRSTYPE_WSO:
  case CGRSTYPE_MIB:
  case CGRSTYPE_FLAG:
  case CGRSTYPE_MARKER:
    if (data.Istr3[p_ind] === 1) {
      if (val_str == "0") {
        data.IValueStr[p_ind] = top.srvap_getString("OFF");
      }
      else {
        data.IValueStr[p_ind] = top.srvap_getString("ON");
      }
    }
    break;

  default:
    data.IValueStr[p_ind] = val_str;
    break;
  }

  $.each(data.Iargr, function(argIdx, argObject) {
    l_pos = argObject.toLowerCase().indexOf("!indirect" + (l_p_ind + 1)); // Find location of indirect

    if (l_pos >= 0) {
      if (data.hasOwnProperty(argIdx)) {

        ReplaceIndirectValues(data, argIdx, argObject);

        switch (argIdx.toLowerCase()) {
        case "caption":
          SetCaption(data);
          break;
  
        case "dataindex":
          UpdateIndirectMonitor(data, data.DataType, DIRECT_C, data.DataIndex);
          break;

        case "groupnum":
          if ((data.DataType == CGRSTYPE_CURPOS) ||
              (data.DataType == CGRSTYPE_POSREG) ||
              (data.DataType == CGRSTYPE_TFRAME) ||
              (data.DataType == CGRSTYPE_JFRAME) ||
              (data.DataType == CGRSTYPE_UFRAME)) {
            UpdateIndirectMonitor(data, data.DataType, DIRECT_C, data.GroupNum);
          }
          break;

        default:
          break;
        }
      }
    }
  });

  if ((data.CtlType == GMCT_COMBOBOX) &&
      (data.Type == COMBOBOX_STRREGTYPE) &&
      (data.IDataType[p_ind] == CGRSTYPE_STRREG)) {
    // !Indirect5 is used to monitor the StrReg.
    UpdateValue(val_str, data);
  }

  // Update control.
  UpdateCtl(data);
} // ProcessIndirectEvents

// Initialize Control Data.
function cgop_InitCtlData(data) {

  // Process parameters.
  ProcessIndirect(data);
  ProcessParams(data);

} // cgop_InitCtlData

// Initialize Control Data again using indirection.
function cgop_InitCtlData2(data) {
  if (data.CtlType == GMCT_EXECUTION) {
    SetParameter(1, data.Parameter1, data);
    SetParameter(2, data.Parameter2, data);
    SetParameter(3, data.Parameter3, data);
    SetParameter(4, data.Parameter4, data);
    SetParameter(5, data.Parameter5, data);
    SetParameter(6, data.Parameter6, data);
    SetParameter(7, data.Parameter7, data);
    SetParameter(8, data.Parameter8, data);
  }
} // cgop_InitCtlData2

// Initialize or uninitialize events for each type.
function cgop_InitRobotEvent(init, data) {
  // Start/stop the Pipe Event.
  if (data.Pipe != "") {
    InitPipeEvent(init, data);
  }

  // Start/stop the Robot Events.
  InitRobotEvents(init, data);

  // Start/stop the Indirect Events.
  InitIndirectEvents(init, data);

} // cgop_InitRobotEvent

// Create Control.
function cgop_CreateCtl(data) {

  if (data.DataIndex != "") {
    // Prepare all the necessary data to access DataType/DataIndex from the controller.
    PrepareCtl(data.DataType, data.DataIndex, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[0]);
  }

  switch (data.CtlType) {
  case GMCT_COMBOBOX:
    data.Type = parseInt(data.Type);
    if ((data.Type == COMBOBOX_VARIABLETYPE) &&
        (data.DataSource)) {
      // Prepare all the decessary data to access DataSource from the controller.
      // Store in varinfo2.
      PrepareCtl(CGRSTYPE_KARELVAR, data.DataSource, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[1]);
    }
    else if (data.Type == COMBOBOX_STRREGTYPE) {
      // DataSource will be filled in once we get the value.
      // Just use 0 for now.
      data.DataSource = "0";
      // Prepare all the necessary data to access DataSource from the
      // controller and store in data.varinfo[1].
      PrepareCtl(CGRSTYPE_STRREG, data.DataSource, data, PREP_VARNAME_C, DIRECT_C, data.varinfo[1]);
      // Set ~Indirect5 to monitor the StrReg.
      data.IDataType[4] = CGRSTYPE_STRREG;
      // data.IDataIndex[4] will be set once we know the index.
      data.IOIndex = 0;
    }
    if ((data.Type != COMBOBOX_PROGRAMTYPE) &&
        (data.Type != COMBOBOX_FILETYPE) &&
        (!(data.TextIndex === ""))) {
      // Prepare all the necessary data to access TextType/TextIndex from the
      // controller and store in varinfo3.
      PrepareCtl(data.TextType, data.TextIndex, data, PREP_NORMAL_C, DIRECT_C, data.varinfo[2]);
    }

  // fall through...
  case GMCT_LABEL:
  case GMCT_EDBOX:
    // Create a button for the label/editbox/combobox.
    CreateButton(data);

    // Set the caption.
    SetCaption(data);
    break;

  case GMCT_TGLMP:
  case GMCT_OSBTN:
  case GMCT_TGBTN:
  case GMCT_BTCHG:
  case GMCT_MNCHG:
    // Create a button for the toggle lamp/oneshot button/toggle button/button change/menu change.
    CreateButton(data);

    // Set the caption.
    SetCaption(data);
    break;

  case GMCT_MULTI:
    // Initialize data.
    if ((data.DataDefault != "") &&
        (data.ViewType != CGMULTI_BITMAP)) {
      data.ButtonText = data.DataDefault;
    }
    else {
      data.ButtonText = "";
    }

    // Create a button for the multi control.
    CreateButton(data);
    break;

  case GMCT_ATCHG:
    // Initialize DataDefault to FALSE.
    // We must receive data first time to enable control.
    data.DataDefault = false;
    break;

  case GMCT_EXECUTION:
    // Send Execution event when window loaded
    $(document).ready(function() {
      SendEventToController("0x430011", data);
    });
    break;

  default:
    break;
  } // switch (data.CtlType)
} // cgop_CreateCtl

function add_connectId(data) {
  var pos;	 
  if ((pos = data.indexOf("?")) >= 0) {
    var page_str = data.substr(pos+1);
    return '/SOFTPART/GENLINK?connect_id=' + top.g_connect_id + '&' + page_str;
  }	
} //add_connectId

// Handle Control events.
function cgop_CtlEvent(data) {

  var sb_state;

  if ((top.g_device_id == data.fDeviceId) &&
      (top.gPaneExec[data.fDeviceId].fFlags & PANE_FLAG_CLICK_EVENTS_C) &&
      (!data.fKey)) {
    SendEventToController("0x43001D", data);
  }

  if (((data.ReadOnly != "") && (data.ReadOnly != "0")) ||
      (data.IsEcho)) {
    return;
  }

  if (chkkey_supported(data) &&
      ((data.ChkDeadman != "0") || (data.ChkTpEnable != "0") ||
       (data.ChkShift != "0") || (data.ChkIo != "0"))) {
    if (!data.ChkDeadmanResult && !data.ChkTpEnableResult &&
        !data.ChkShiftResult && !data.ChkIoResult) {
      chkkey_check(data);
      return;
    }
    else {
      // Results validated in HandleChkkeyCallback.
      // Clear results prior to continuing
      chkkey_result_init(data);
    }
  } // chkkey_supported( ) && .....

  if ((data.CtlType == GMCT_TGBTN) || (data.CtlType == GMCT_OSBTN)) {
    if (((data.DataType == CGRSTYPE_DO) || (data.DataType == CGRSTYPE_FLAG)) &&
        (data.Pulse != "0") && (data.PulseTime != "0") && 
        (data.inPulse)) {
      // Ignore user key press and let timer finish.
      // Reset to proper value.
      if (data.CtlType == GMCT_OSBTN) {
        sb_state = !GetButtonState(data);
      }
      else {
        sb_state = GetButtonState(data);
      }
      FAbButtonCompo_SetValue(data, sb_state);
      return;
    }
  }

  switch (data.CtlType) {
  case GMCT_EDBOX:
    if (!data.MenuUp) {
      // Put up the input.
      var start_title = "";
      var start_guide = "";
      var start_str = "";
      if (data.Type == CGEDBOX_FULLKEY) {
        start_title = top.srvap_getString("Str_title");
        start_guide = top.srvap_getString("Str_guide");
        start_str = data.ValueText;
        top.dlg_input_open(start_title, start_guide, start_str);
      }
      else {
        top.dlg_input_real = false;
        top.dlg_input_min_val = null;
        top.dlg_input_max_val = null;
        start_title = top.srvap_getString("HK_title");
        start_guide = top.srvap_getString("HK_guide").replace("% g", data.ValueText);
        if ((undefined != data.ValueMin) && (data.ValueMin != VALUEMIN_C) && 
            (undefined != data.ValueMax) && (data.ValueMax != VALUEMAX_C)) {
          var res = top.srvap_getString("HK_guide2").split("% g");
          start_guide += "<br />" + res[0] + data.ValueMin + "<br />" + res[1] + data.ValueMax;
          top.dlg_input_min_val = data.ValueMin;
          top.dlg_input_max_val = data.ValueMax;
        }
        top.dlg_numeric_open(start_title, start_guide, start_str);
      }
      top.dlg_input_listener = data.$this;
      top.dlg_input_listener.bind('InputEvent', data, HandleInputEvent); // Attach handler for InputEvent.
      data.MenuUp = true;
      FAbButtonCompo_SetValue(data, true); // shows button pressed
    }
    else {
      // Take down the input.
      if (data.Type == CGEDBOX_FULLKEY) {
      top.dlg_input_close();
      }
      else {
        top.dlg_numeric_close();
      }
    // HandleInputEvent is called and will complete the close
    }
    break;

  case GMCT_OSBTN:
    data.ValueNumber = 1;
    FAbButtonCompo_SetValue(data, true); // shows button pressed
    SetImages(data);
    cgop_SetColors(data);

    if (data.Type == CGOSBTN_STRING) {
      UpdateValueToController(-1, data.ValueStr, data);
    }
    else {
      if (data.Type == CGOSBTN_BOOL) {
        if (data.SetValue != "0") {
          data.SetValue = "1";
        }
      }
      UpdateValueToController(-1, data.SetValue, data);
    }

    if (((data.DataType == CGRSTYPE_DO) || (data.DataType == CGRSTYPE_FLAG)) &&
        (data.Pulse != "0") && (data.PulseTime != "0")) {
      // Pulse DO
      data.inPulse = true;
      setTimeout(function() {
        // Set the reverse value.
        if (data.SetValue == "1") {
          val_str = "0";
        }
        else {
          val_str = "1";
        }
        if (UpdateValueToController(-1, val_str, data) == 0) {
          UpdateValue(val_str, data);
        }
        data.ValueNumber = 0;
        FAbButtonCompo_SetValue(data, false); // shows button released
        SetImages(data);
        cgop_SetColors(data);
        data.inPulse = false;
      }, Number(data.PulseTime) );
    }
    else {
      // Set button off since this is a oneshot button.
      setTimeout(function() {
        data.ValueNumber = 0;
        FAbButtonCompo_SetValue(data, false); // shows button released
        SetImages(data);
        cgop_SetColors(data);
      }, 100 );
    }
    if (data.dlgDismiss == "1") {
      // Dismiss dialog box.
      top.rpcmc_sendKeyCode(tk_cancel_c);
    }
    break;

  case GMCT_TGBTN:
    var sb_pulse = false;
    var val_str = "";
    var val_num = 0;

    sb_state = !GetButtonState(data);
    if (sb_state) {
      if (data.Type == CGTGBTN_STRING) {
        val_str = data.TrueValueStr;
      }
      else if (data.Type == CGTGBTN_BOOL) {
        val_num = (data.Invert != "0") ? 0 : 1;
        val_str = val_num.toString();
      }
      else {
        val_num = Number(data.TrueValue);
        val_str = data.TrueValue;
      }
    }
    else {
      if (data.Type == CGTGBTN_STRING) {
        val_str = data.ValueStr;
      }
      else if (data.Type == CGTGBTN_BOOL) {
        val_num = (data.Invert != "0") ? 1 : 0;
        val_str = val_num.toString();
      }
      else {
        val_str = data.FalseValue;
        val_num = Number(data.FalseValue);
      }
    }
    if (((data.DataType == CGRSTYPE_DO) || (data.DataType == CGRSTYPE_FLAG)) &&
        (data.Pulse != "0") && (data.PulseTime != "0")) {
      // Pulse DO
      sb_pulse = true;
    }
    if ((UpdateValueToController(-1, val_str, data) == 0) &&
        ((data.Interval > 200) || (sb_pulse))) {
      // Go ahead and set the data now
      // otherwise wait for controller event to set data.
      data.ValueNumber = val_num;
      UpdateValue(val_str, data);
    }

    // Update control.
    UpdateCtl(data);

    if (sb_pulse) {
      // Pulse DO
      data.ValueNumber = val_num;
      data.inPulse = true;
      setTimeout(function() {
        // Set the reverse value.
        if (data.ValueNumber) {
          data.ValueNumber = 0;
          val_str = top.srvap_getString("OFF");
        }
        else {
          data.ValueNumber = 1;
          val_str = top.srvap_getString("ON");
        }
        if (UpdateValueToController(-1, val_str, data) == 0) {
          UpdateValue(val_str, data);
        }
        data.inPulse = false;
        UpdateButton(data);
      }, Number(data.PulseTime) );
    }
    if (data.dlgDismiss == "1") {
      // Dismiss dialog box.
      top.rpcmc_sendKeyCode(tk_cancel_c);
    }
    break;

  case GMCT_BTCHG:
    data.ValueNumber = 1;
    FAbButtonCompo_SetValue(data, true); // shows button pressed
    SetImages(data);
    cgop_SetColors(data);
    if (data.PageName01 != "") {
      // Frame is stored in PageName01
      data.Framename = data.PageName01;
    }
    // Value contain /SOFTPART/GENLINK?
    if (data.PageName.indexOf("/SOFTPART/GENLINK?") >= 0) {
      data.PageName = add_connectId(data.PageName);
    }	
    ChangePage(data.PageName, data);

    // Set button off since this may actually perform
    // a command instead of changing the page.
    setTimeout(function() {
      data.ValueNumber = 0;
      FAbButtonCompo_SetValue(data, false); // shows button released
      SetImages(data);
      cgop_SetColors(data);
      if (top.gPaneExec[data.fDeviceId].fFlags & PANE_FLAG_CLICK_EVENTS_C) {
        SendEventToController("0x430013", data);
      }
    }, 100 );

    if (data.dlgDismiss == "1") {
      // Dismiss dialog box.
      top.rpcmc_sendKeyCode(tk_cancel_c);
    }
    break;

  case GMCT_MNCHG:
    if (!data.MenuUp) {

      // Put up the menu.
      top.dlg_listener = data.$this;
      top.dlg_listener.bind('MenuEvent', data, HandleMenuEvent); // Attach handler for MenuEvent.
      var str = AddFormCaption(1, data.PageCaption01);
      str += AddFormCaption(2, data.PageCaption02);
      str += AddFormCaption(3, data.PageCaption03);
      str += AddFormCaption(4, data.PageCaption04);
      str += AddFormCaption(5, data.PageCaption05);
      str += AddFormCaption(6, data.PageCaption06);
      str += AddFormCaption(7, data.PageCaption07);
      str += AddFormCaption(8, data.PageCaption08);
      str += AddFormCaption(9, data.PageCaption09);
      str += AddFormCaption(10, data.PageCaption10);
      if (str != "") {
        str = str.substring(0, str.length-1);
        str = '{"dlg":[' + str + ']}';
        top.dlg_items_per_page = 10;
        top.dlg_items = JSON.parse(str).dlg;
        if (top.dlg_items.length < top.dlg_items_per_page) {
          top.dlg_items_this_page = top.dlg_items.length;
        }
        else {
          top.dlg_items_this_page = top.dlg_items_per_page;
        }
        top.dlg_type = PMEV_TYPE_MENU_C;
        top.dlg_flyout = false;
        top.dlg_spage = 0;
        top.dlg_epage = ((top.dlg_items.length - 1) / (top.dlg_items_per_page-1)) >> 0; // integer division
        top.dlg_item = 1;
        top.dlg_notitle = false;
        top.dlg_populate();
        top.dlg_open("");
        data.MenuUp = true;
        FAbButtonCompo_SetValue(data, true); // shows button pressed
        SetImages(data);
        cgop_SetColors(data);
      }
    }
    else {
      // Take down the menu.
      top.dlg_close();
      // HandleMenuEvent is called and will complete the close
    }
    break;

  case GMCT_COMBOBOX:
    if (!data.MenuUp) {
      // Put up the menu.
      switch (data.Type) {
      case COMBOBOX_LISTTYPE:
        break;

      case COMBOBOX_PROGRAMTYPE:
        // RPC callback will open the menu
        // data.IOType is actually data.ProgType
        top.rpcmc_tpxprgsb_ext(data.fDeviceId, data.ProgType, '1', DialogOpen, data);
        break;

      case COMBOBOX_VARIABLETYPE:
        if (data.DataSource) {
          var sub_idx = 1;
          if (!isNaN(data.ValueNumber)) {
            sub_idx = data.ValueNumber;
            if (data.BoolType === "1") {
              sub_idx++; // enum values start at 1
            }
          }
          // RPC callback will open the menu
          top.rpcmc_tpxensbv_krl_ext(data.fDeviceId, data.varinfo[1].prognam,
                                 data.varinfo[1].progvar, sub_idx, DialogOpen,
                                 data);
        }
        break;

      case COMBOBOX_STRREGTYPE:
        if (data.DataSource) {
          var sub_idx = 1;
          if (isNaN(data.ValueNumber)) {
            sub_idx = 1;
          }
          else {
            sub_idx = parseInt(data.ValueNumber);
          }
          // RPC callback will open the menu
          top.rpcmc_tpxensub(data.fDeviceId, "TPDB_INPUT", tpdb_style_indx_sreg_c, sub_idx,
                         DialogOpen, data);
        }
        break;

      case COMBOBOX_DICTIONARYTYPE:
        if (data.DataSource) {
          var sub_idx = 1;
          if (isNaN(data.ValueNumber)) {
            // DictName may not be set yet because data is uninit
            if (data.DictName === "") {
              cgop_DictParam(data);
            } // if DictName not set yet
          }
          else {
            sub_idx = data.ValueNumber;
            if (data.BoolType === "1") {
              sub_idx++; // enum values start at 1
            }
          }
          // RPC callback will open the menu
          top.rpcmc_tpxensub(data.fDeviceId, data.DictName, data.DictEle, sub_idx,
                         DialogOpen, data);
        }
        break;

      case COMBOBOX_FILETYPE:
        // RPC callback will open the menu
        // data.IOType is actually data.ProgType
        top.rpcmc_tpxfilsb_ext(data.fDeviceId, data.DataSource, '1', DialogOpen, data);
        break;
      }
    } // Menu was not up
    else {
      // Take down the menu.
      top.dlg_close();
      // HandleMenuEvent is called and will complete the close

      if (data.dlgDismiss == "1") {
        // Dismiss dialog box.
        top.rpcmc_sendKeyCode(tk_cancel_c);
      }
    }
  } // switch (data.CtlType)
} // cgop_CtlEvent

// Delete Control Data.
function cgop_DeleteCtl(data) {
  if (data.fKey) {
    // Tell FunctionKey plug-in to stop our fkey (0 based).
    top.gPaneFKey[data.fDeviceId].FKeyListener[data.fKey-1] = null;
    top.gPaneFKey[data.fDeviceId].FKeyImgIcon[data.fKey-1] = null;

    // Trigger a DeviceEvent on the fky event listeners.
    top.jQuery.fkylis.trigger("DeviceEvent", ["CGOP", data.fKey]);
  }
  // Issue Disconnect for Execution Control
  if (data.CtlType == GMCT_EXECUTION) {
    if ((!top.g_echo) && (top.g_connect_id)) {
      if ((top.gPaneExec[data.fDeviceId].fTaskIdx) &&
          (top.gPaneExec[data.fDeviceId].fExecConnectId)) {
        top.rpcmc_tplink_disconnect(data.fDeviceId);
      }
    }
    top.gPaneExec[data.fDeviceId].fTaskIdx = null;
    top.gPaneExec[data.fDeviceId].fFlags = null;
    top.gPaneExec[data.fDeviceId].fExecType = null;
    top.gPaneExec[data.fDeviceId].fExecName = '';
    top.gPaneExec[data.fDeviceId].fExecConnectId = null;
  }

} // cgop_DeleteCtl

// Private functions

function cgop_HandleFKeyClickEvent(event, device, cgop_index) {
  var data = event.data || event;

  if (cgop_index == data.fKey - 1) {
    // We have a match.
    HandleClickEvent(event);
  }
} // cgop_HandleFKeyClickEvent


function cgop_HandleIOEvent(event, io_type, io_index, io_value) {
  var data = event.data || event;

  // Process indirect events first.
  for (var si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
    ProcessIndirectEvents(data, io_type, io_index, data.Istr3[si_ind], io_value, si_ind);
  }

  if ((io_type == data.IOType) && (io_index == data.IOIndex) && (!data.IOSim)) {
    // We have a match.
    if (data.IOBoolean) {
      if (io_value == "0") {
        io_value = top.srvap_getString("OFF");
      }
      else {
        io_value = top.srvap_getString("ON");
      }
    }
    UpdateValue(io_value, data);
    UpdateCtl(data);
  }
} // cgop_HandleIOEvent

function HandlePipeEvent(event, file, buffer) {
  event.preventDefault();
  var data = event.data || event;

  var keyVal = {};
  var cgopPipeData = [];
  var lineData = null;
  var tempLineData = null;
  var subData = null;
  var ctrlId = '';
  var idx = 0;

  // Build pipe data object for cgop control to update against
  if ((file.toLowerCase() == data.Pipe.toLowerCase()) && (buffer.length > 0)) {
    // Pipe has new data.
    lineData = FixPipe(buffer);
    if (lineData) {
      data.repaint = mCgop_paintNone;
      // Process indirection first.
      for (idx = 0; idx < lineData.length; idx++) {
        keyVal = {};
        if (lineData[idx]) {
          tempLineData = lineData[idx].trim();
          if (tempLineData.indexOf('=') > 0) {
            subData = tempLineData.split('=');

            // We have our key:value pipe data format
            if (subData[0].toLowerCase() == "id") {
              // skip Id parameter
            }
            else {
              if (hasOwnProperty_noCase(data, subData[0])) {
                subData[1] = subData[1].replace(/^"(.*)"$/g, '$1'); // Trim double quotes, if any
                keyVal[subData[0]] = subData[1];
                cgopPipeData.push(keyVal);
              }
            }
          }
        }
      }
    }
  }
  if (cgopPipeData.length > 0) {
    UpdateParams(data, cgopPipeData);
  }
  return true;
} // HandlePipeEvent

function HandleIOCallback(io_type, io_index, io_value, p_ind, data) {

  if (p_ind > 0) {
    // Indirect parameter
    // Process indirect events
    ProcessIndirectEvents(data, io_type, io_index, data.Istr3[p_ind], io_value, p_ind);
    return;
  }
  if ((io_type == data.IOType) && (io_index == data.IOIndex) && (!data.IOSim)) {
    // We have a match.
    if (data.IOBoolean) {
      if (io_value == "0") {
        io_value = top.srvap_getString("OFF");
      }
      else {
        io_value = top.srvap_getString("ON");
      }
    }
    UpdateValue(io_value, data);
    UpdateCtl(data);
  }
} // HandleIOCallback

function cgop_HandleIOSimEvent(event, io_type, io_index, io_value) {
  var data = event.data || event;
  if ((io_type == data.IOType) && (io_index == data.IOIndex) && (data.IOSim)) {
    // We have a match.
    if (!io_value) {
      io_value = top.srvap_getString("UNSIM");
    }
    else {
      io_value = top.srvap_getString("SIM");
    }
    UpdateValue(io_value, data);
    UpdateCtl(data);
  }
} // cgop_HandleIOSimEvent

function cgop_HandleIOSimAllEvent(event, io_type, io_index, io_value) {
  var data = event.data || event;
  if (((io_type == "0") || // all types 
      (io_type == data.IOType)) && 
      (data.IOSim)) {
    // We have a match.
    if (!io_value) {
      io_value = top.srvap_getString("UNSIM");
    }
    else {
      io_value = top.srvap_getString("SIM");
    }
    UpdateValue(io_value, data);
    UpdateCtl(data);
  }
} // cgop_HandleIOSimAllEvent

function HandleIOSimCallback(io_type, io_index, io_value, p_ind, data) {
  if ((io_type == data.IOType) && (io_index == data.IOIndex) && (data.IOSim)) {
    // We have a match.
    if (!(io_value === "1")) {
      io_value = top.srvap_getString("UNSIM");
    }
    else {
      io_value = top.srvap_getString("SIM");
    }
    UpdateValue(io_value, data);
    UpdateCtl(data);
  }

} // HandleIOSimCallback

function cgop_HandleVarEvent(event, prog_name, var_name, type_code, val_str) {

  var data = event.data || event;

  // NOTE: pr52764 Uninit strings will be "\r\n"
  if (val_str != "") {
    if ((val_str.toLowerCase() == "uninitialized") ||
        (val_str.match(/^\s*$/g))) { // check if entire string contains whitespace character class
      val_str = UNINIT_STR_C;
    }
  }

  // Process indirect events first.
  for (var si_ind = 0; si_ind < MAX_INDIRECT_C; si_ind++) {
    ProcessIndirectEvents(data, prog_name, var_name, data.Istr3[si_ind], val_str, si_ind);
  }

  // Check `prog_name` && `var_name` is equivalent to check `px_event->fpc_var`
    if ((prog_name == data.varinfo[0].prognam) && (var_name == data.varinfo[0].progvar)) {
      // We have a match.
      data.ValueNumber = parseInt(val_str);
      UpdateValue(val_str, data);

      if ((data.Type == COMBOBOX_STRREGTYPE) &&
          (data.IDataType[4] == CGRSTYPE_STRREG)) {
        // Change !Indirect5 to use the value as the index into StrReg.
        if (data.IOIndex != data.ValueNumber) {
          InitIndirectEvents(false, data); // stop existing strreg mon
          data.IDataIndex[4] = data.IOIndex = data.ValueNumber;
          InitIndirectEvents(true, data); // start new strreg idx mon
        }
      }
    UpdateCtl(data);
  }

  if ((data.CtlType == GMCT_COMBOBOX) && (data.MenuUp)) {
    switch (data.Type) {
    case COMBOBOX_PROGRAMTYPE:
      if (prog_name != "*TPXPRGSB*") {
        break;
      }
      if (var_name == "ENTER") {
        // New program was selected from tpxprgsb_ext.
        // Add rpc request to message queue.
        setTimeout(UpdateValueToController(-1, val_str, data));
        if (data.Interval > 200) {
          // Go ahead and set the data now
          // Otherwise wait for controller event to set data
          UpdateValue(val_str, data);
        }
        UpdateCtl(data);
      }
      DialogOpen(DIALOG_CLOSE_C, data);
      break;

    case COMBOBOX_VARIABLETYPE:
      if (prog_name != "*TPXENSBV*") {
        break;
      }
      if (var_name == "ENTER") {
        // New integer value selected from tpxensbv_krl_ext.
        // Add rpc request to message queue.
        setTimeout(UpdateValueToController(-1, data.ValueNumber, data));
        if (data.Interval > 200) {
          // Go ahead and set the data now
          // otherwise wait for controller event to set data
          data.ValueNumber = parseInt(val_str);
          if (data.BoolType === "1") { // enum values start at 1
            if (data.ValueNumber == 1) {
              data.ValueNumber = 0;
            }
            else {
              data.ValueNumber = 1;
            }
          }
          UpdateValue(val_str, data);
        }
        UpdateCtl(data);
      }
      DialogOpen(DIALOG_CLOSE_C, data);
      break;

    case COMBOBOX_DICTIONARYTYPE:
    case COMBOBOX_STRREGTYPE:
      if (prog_name != "*TPXENSUB*") {
        break;
      }
      if (var_name == "ENTER") {
        // New integer value selected from tpxensub_ext.
        // Add rpc request to message queue.
        setTimeout(UpdateValueToController(-1, data.ValueNumber, data));
        if (data.Interval > 200) {
          data.ValueNumber = val_str;
          if (data.BoolType === "1") { // enum values start at 1
            if (data.ValueNumber == 1) {
              data.ValueNumber = 0;
            }
            else {
              data.ValueNumber = 1;
            }
          }
          UpdateValue(val_str, data);
        }

        if ((data.CGType == COMBOBOX_STRREGTYPE) &&
            (data.IDataType[4] == CGRSTYPE_STRREG)) {
          // Change !Indirect5 to use the value as the index into StrReg.
          if (data.IOIndex != data.ValueNumber) {
            InitIndirectEvents(false, data); // stop existing strreg mon
            data.IDataIndex = data.IOIndex = data.ValueNumber;
            InitIndirectEvents(true, data); // start new strreg idx mon
          }
        }
        UpdateCtl(data);
      }
      DialogOpen(DIALOG_CLOSE_C, data);
      break;

    case COMBOBOX_FILETYPE:
      if (prog_name != "*TPXFILSB*") {
        break;
      }
      if (var_name == "ENTER") {
        // New file was selected from tpxfilsb_ext.
        // Add rpc request to message queue.
        setTimeout(UpdateValueToController(-1, val_str, data));
        if (data.Interval > 200) {
          // Go ahead and set the data now
          // otherwise wait for controller event to set data
          UpdateValue(val_str, data);
        }
        UpdateCtl(data);
      }
      DialogOpen(DIALOG_CLOSE_C, data);
      break;
    }

    if (data.dlgDismiss == "1") {
      // Dismiss dialog box.
      top.rpcmc_sendKeyCode(tk_cancel_c);
    }
  }
} // cgop_HandleVarEvent

function HandleVarCallback(prog_name, var_name, type_code, val_str, p_ind, data) {

  var l_status = "";

  if (val_str == "Uninitialized") {
    val_str = UNINIT_STR_C;
  }

  if (p_ind >= 0) {
    // Process indirect events
    HandleVarCallbackErrors(data.IDataType[p_ind], val_str, (data.Id + ": " + data.IDataIndex[p_ind]), l_status);
    if (l_status === "") {
    ProcessIndirectEvents(data, prog_name, var_name, data.Istr3[p_ind], val_str, p_ind);
    }
    return;
  }

  if (typeof(type_code) === "string") {
    // Try to convert it
    type_code = parseInt(type_code);
  }

  if (!isNaN(type_code)) {
    switch (type_code) {
    case MSTRREG:
    case TPXENSBV_KRL_TEXT:
      if ((prog_name == data.varinfo[1].prognam) && (var_name == data.varinfo[1].progvar)) {
        // We have a match.
        HandleVarCallbackErrors(CGRSTYPE_DICTELE, val_str, (data.Id + ": " + data.DataIndex), l_status);
        if (l_status === "") {
        data.ValueText = val_str;
        UpdateValue(val_str, data);
        UpdateCtl(data);
      }
      }
      break;

    default:
      if ((prog_name == data.varinfo[0].prognam) && (var_name == data.varinfo[0].progvar)) {
        // We have a match.
        HandleVarCallbackErrors(data.DataType, val_str, (data.Id + ": " + data.DataIndex), l_status);
        if (l_status === "") {
        UpdateValue(val_str, data);
        UpdateCtl(data);
        }
      }
      break;
    }
  }
} // HandleVarCallback

function HandleDictCallback(dict_name, ele_no, val_str, data) {
  if ((dict_name == data.DictName) && (ele_no == data.DictEle)) {
    // We have a match.
    UpdateValue(val_str, data);
    UpdateCtl(data);
  }
} // HandleDictCallback


function HandleDictCallback2(dict_name, ele_no, val_str, p_ind, data) {

  if (p_ind >= 0) {
    // Indirect parameter
    // Process indirect events
    ProcessIndirectEvents(data, dict_name, ele_no, data.Istr3[p_ind], val_str, p_ind);
    return;
  }

  var sub_idx = 1;
  if (p_ind == -2) {
    if (!isNaN(data.ValueNumber)) {
      if (data.BoolType === "1") { // enum values start at 1
        sub_idx = (data.DictEle + data.ValueNumber);
      }
      else {
        sub_idx = (data.DictEle + (data.ValueNumber - 1));
      }
    }
  }
  else {
    sub_idx = data.DictEle;
  }

  if ((dict_name == data.DictName) && (ele_no == sub_idx)) {
    // We have a match.
    data.ValueText = val_str;
    UpdateValue(val_str, data);
    UpdateCtl(data);
  }
} // HandleDictCallback2

function cgop_HandleCurposEvent(event, monType, groupNum, title, newlines) {
  var data = event.data || event;

  if ((monType == data.MonType) && (groupNum == data.GroupNum)) {
    PopulateCurpos(data, title, newlines);
  }
} // cgop_HandleCurposEvent

function HandleMenuEvent(event, val_num) {
  var data = event.data || event;
  top.dlg_listener.unbind('MenuEvent',HandleMenuEvent); // Detach handler for MenuEvent.
  top.dlg_listener = null;

  switch (data.CtlType) {
  case GMCT_MNCHG:
    data.MenuUp = false;
    FAbButtonCompo_SetValue(data, false); // shows button released
    SetImages(data);
    cgop_SetColors(data);

    switch (val_num+1) {
    case 1:
      ChangePage(data.PageName01, data);
      break;
    case 2:
      ChangePage(data.PageName02, data);
      break;
    case 3:
      ChangePage(data.PageName03, data);
      break;
    case 4:
      ChangePage(data.PageName04, data);
      break;
    case 5:
      ChangePage(data.PageName05, data);
      break;
    case 6:
      ChangePage(data.PageName06, data);
      break;
    case 7:
      ChangePage(data.PageName07, data);
      break;
    case 8:
      ChangePage(data.PageName08, data);
      break;
    case 9:
      ChangePage(data.PageName09, data);
      break;
    case 10:
      ChangePage(data.PageName10, data);
      break;
    } // switch (val_num)

    if (val_num >= 0) {
      if (data.dlgDismiss == "1") {
        // Dismiss dialog box.
        top.rpcmc_sendKeyCode(tk_cancel_c);
      }
    }
    break;

  case GMCT_COMBOBOX:
    // Combobox can use dialog, or popup type sub menu
    if (((top.dlg_type == PMEV_SUB_MENU_C) && !top.dlg_popup_closed) ||
        (top.dlg_dialog_inuse)) {
      if (val_num >= 0) {
        data.ValueNumber = val_num + 1;
        top.dlg_select(val_num);
	// The VarEvent will set data.MenuUp false
      }
      else {
        // Send close request.
        if ((top.dlg_type == PMEV_SUB_MENU_C) ||
            ((top.dlg_dialog_inuse) && !top.dlg_dialog_closed)) {
          top.rpcmc_sendKeyCode(prev_c);
        }
        DialogOpen(DIALOG_CLOSE_C, data);
      }
    }
    break;
  } // switch (data.CtlType)
} // HandleMenuEvent

function HandleInputEvent(event, val_str) {
  var data = event.data || event;
  data.MenuUp = false;
  FAbButtonCompo_SetValue(data, false); // shows button released
  top.dlg_input_listener.unbind('InputEvent',HandleInputEvent); // Detach handler for InputEvent.
  top.dlg_input_listener = null;

  if (val_str != null) {
    if ((UpdateValueToController(-1, val_str, data) == 0) &&
        (data.Interval > 200)) {
      // Go ahead and set the data now
      // otherwise wait for controller event to set data.
      UpdateValue(val_str, data);
    }

    // Update control.
    UpdateCtl(data);

    if (data.dlgDismiss == "1") {
      // Dismiss dialog box after input is removed.
      setTimeout(function() { top.rpcmc_sendKeyCode(tk_cancel_c); }, 250);
    }
  }
} // HandleInputEvent

function HandleClickEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  if ((data.CtlType == GMCT_TGLMP) ||
      (data.CtlType == GMCT_LABEL) ||
      (data.CtlType == GMCT_MULTI) ||
      (data.Disable)) {
    // Button cannot be clicked.
    return false;
  }
  else if ((data.CtlType == GMCT_COMBOBOX) && data.MenuUp) {
    // Combobox busy
    return false;
  }
  else {

    chkkey_init(data);

    var bstyle = data.$this.css("border-left-style");
    if (bstyle == "outset") {
      data.$this.css("border-style" , "inset");
    }
    else if (bstyle == "inset") {
      data.$this.css("border-style" , "outset");
    }
    cgop_CtlEvent(data);
    return true;
  }
} // HandleClickEvent

function HandleMouseDownEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  // Only support turning the button on.
  if (GetButtonState(data)) {
    return false;
  }
  else {

    chkkey_init(data);

    var bstyle = data.$this.css("border-left-style");
    if (bstyle == "outset") {
      data.$this.css("border-style" , "inset");
    }
    cgop_CtlEvent(data);
    return true;
  }
} // HandleMouseDownEvent

function HandleMouseUpEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  // Only support turning the button off.
  if (!GetButtonState(data)) {
    return false;
  }
  else {
    var bstyle = data.$this.css("border-left-style");
    if (bstyle == "inset") {
      data.$this.css("border-style" , "outset");
    }
    cgop_CtlEvent(data);
    return true;
  }
} // HandleMouseUpEvent

function HandleFocusEvent(event) {
  event.preventDefault();
  var data = event.data || event;
  SendEventToController("0x430015", data);
} // HandleFocusEvent

function HandleChkkeyCallback(io_type, io_index, io_value, data, chk_status) {

  var sb_state = false;
  var check_result = false;
  var deadman_okay = false;
  var tpenable_okay = false;
  var shift_okay = false;
  var io_okay = false;
  var msg_string = "";

  if (chk_status != 0) {
    sb_state = GetButtonState(data);
    setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
    chkkey_message(data, top.srvap_getString("IOERR_C"));
  }
  else {
    // Process received value
    if (io_value && (io_value == "1")) {
      check_result = true;
    }

    // Same TP check occurs for multiple cases: TP and Deadman
    if ((io_type == tpin_type_c) && (io_index == tp_enbl_c) &&
        ((data.ChkDeadman != "0") || (data.ChkTpEnable != "0"))) {
      if (check_result) {
        data.ChkTpEnableResult = true;
      }
      else {
        // TP is disabled.
        sb_state = GetButtonState(data);
        setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
        chkkey_message(data, top.srvap_getString("CHKBTN_TPDIS_C"));
      }
    }

    // Compare result with matching check
    if ((data.ChkDeadman != "0") && (io_type == tpin_type_c)) {
      switch (io_index) {
        case estp_c:
          if (!check_result) {
            data.ChkDeadmanEstpResult = true;
          }
          else {
            // TP Estop is pressed
            sb_state = GetButtonState(data);
            setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
            chkkey_message(data, top.srvap_getString("CHKBTN_ESTOP_C"));
          }
          break;

        case tp_enbl_c:
          // Already done
          break;

        case mor_ss_c:
          // This is same as piSafetySignals.c 0 means proper status.
          if (0 == (parseInt(io_value) & MFS_DEADMAN)) {
            // Properly pressed
            data.ChkDeadmanMorssResult = true;
          }
          else {
            // Deadman is not in proper status
            setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
            chkkey_message(data, top.srvap_getString("CHKBTN_DEADMAN_C"));
          }
          break;

        default:
          // no match
          break;
      }

      // Logically AND the required signals into the summary bit
      data.ChkDeadmanResult = (data.ChkDeadmanEstpResult &
                               data.ChkTpEnableResult &
                               data.ChkDeadmanMorssResult);
    }

    if ((data.ChkShift != "0") && (io_type == tpin_type_c) && (io_index == shft_c)) {
      if (check_result) {
        data.ChkShiftResult = true;
      }
      else {
        // Shift key is not pressed.
        sb_state = GetButtonState(data);
        setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
        chkkey_message(data, top.srvap_getString("CHKBTN_SHIFT_C"));
        return;
      }
    }

    if ((data.ChkIo != "0") && (io_type == data.ChkIoType) && (io_index == data.ChkIoIdx)) {
      if (check_result) {
        data.ChkIoResult = true;
      }
      else {
        // Specified I/O is OFF.
        sb_state = GetButtonState(data);
        setTimeout(FAbButtonCompo_SetValue, 100, data, sb_state);
        chkkey_message(data, top.srvap_getString("CHKBTN_IO_C"));
        return;
      }
    }

    // Only call cgop_CtlEvent again when all pending checks have passed
    deadman_okay = ((data.ChkDeadman == "0") || ((data.ChkDeadman != "0") && (data.ChkDeadmanResult == true)));
    tpenable_okay = ((data.ChkTpEnable == "0") || ((data.ChkTpEnable != "0") && (data.ChkTpEnableResult == true)));
    shift_okay = ((data.ChkShift == "0") || ((data.ChkShift != "0") && (data.ChkShiftResult == true)));
    io_okay = ((data.ChkIo == "0") || ((data.ChkIo != "0") && (data.ChkIoResult == true)));

    if (deadman_okay && tpenable_okay && shift_okay && io_okay) {
      cgop_CtlEvent(data);
    }
  }
} // HandleChkkeyCallback

function cgop_KclRecord_HandleCallback(result_stat, data) {

  var err_title_str = (data.Id + ": Error recording position");
  var err_msg_str = "";

  // Primitive error handling
  if (result_stat != 0) {
    switch (data.DataType) {
    case CGRSTYPE_KARELPOS:
      err_msg_str = "Could not record KARELPOS " + data.DataIndex + "\nStatus:" + result_stat;
      break;

    case CGRSTYPE_POSREG:
      err_msg_str = "Could not record PR[" + data.GroupNum + "," + data.DataIndex + "]\nStatus:" + result_stat;
      break;

    default:
      break;
    }
    top.dlg_msgbox_open(err_title_str, err_msg_str);
  }
} // cgop_KclRecord_HandleCallback
