/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2011
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: jquery.sta.js
 *  
 *  Description:
 *    Status plugin
 *
 *  Author: Judy Evans
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  01-JAN-2011 EVANSJA Initial version.
 *  04-JAN-2017 EVANSJA Set the font family.
 *  19-JAN-2018 GRASSCM pr50667 - Add support for notification icon.
 *  12-OCT-2018 GRASSCM pr51104 - Update statbar CSS to prevent label wrapping.
 *  12-OCT-2018 EVANSJA pr51145 - Support Lock HMI menu for Software Operator Panel.
 *  -
*/

;(function($) {

//  Usage
//  $('#sta1').sta({ name: 'STAT' });
//
$.fn.sta = function(settings) {
    $(this).bind('sta', settings, function() { });
  };

  var sta = $.event.special.sta = {
    // Status plugin defaults
    name: 'STAT',
    device_observer: null,
    device_newlines: new Array(),
    device_tpprog_flg: null,
    warn_timer_id: null,
    warn_icon_active: false,
    warn_icon_showing: false,
    warn_url: null,
    $this: null,

    // Called each time an element does bind
    add: function(handleObj) {
      handleObj.data = $.extend({
        name: sta.name,
        device_observer: sta.device_observer,
        device_newlines: sta.device_newlines,
        device_tpprog_flg: sta.device_tpprog_flg,
        warn_timer_id: null,
        warn_icon_active: sta.warn_icon_active,
        warn_icon_showing: sta.warn_icon_showing,
        warn_url: sta.warn_url,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      data.$this.css("font-family", top.getDefFont());

      // Create labels inside the element.
      var out = '<img id="warn_icon" src=""/>';
      out = out + '<div class="statdiv">';
      out = out + '<label id="l1" class="statlabel" />';
      out = out + '<br />';
      out = out + '<label id="l2" class="statlabel" />';
      out = out + '</div>';
      data.$this.html(out);

      if (top.jQuery.devlis) {
        if (!top.g_cgtpmain) {
          // Send "cgtpmain is loaded" external request to GUID task.
          // This is only done once per connection.
          top.g_cgtpmain = true;
          top.rpcmc_tpextreq(0, tp_cgtpmain_done_c);
        }
        else {
          // vt220 window size change
          top.rpcmc_tpextreq2(0, tp_reconfig_c);
        }

        // Attach handler for DeviceEvent.
        top.jQuery.devlis.bind('DeviceEvent', data, HandleDeviceEvent);

        top.rpcmc_startDeviceMonitor(data.name);

        // Attach handler for SetLangEvent.
        top.jQuery.dictlis.bind('SetLangEvent', data, HandleDictEvent);

        // Attach handler for WarnEvent.
        top.jQuery.devlis.bind('WarnEvent', data, HandleWarnEvent);

        // Attach handler for mouseup event.
        data.$this.mouseup(function(event) {
          var params_idx;

          // Check if mouseup on alarm, else warning
          if((!data.warn_icon_active) || (event.pageX <
             $(this).width() - $("#warn_icon").outerWidth())) {
            // Send DIAG key to controller.
            top.rpcmc_sendKeyCode(diag_c);
          }
          else {
            // Get notification softpart screen
            if((!top.g_echo) && (!top.g_lock_hmi)) {
              params_idx = data.warn_url.indexOf('?');
              if(params_idx >= 0) {
                $.get(data.warn_url.slice(0, (params_idx + 1)) +
                      'connect_id=' + top.g_connect_id + '&' +
                      data.warn_url.slice((params_idx + 1)), function() {
                });
              }
            }
          }
        });
      }
    }, // add

    // Called each time an element removes bind
    remove: function(handleObj) {
      handleObj.data = $.extend({
        name: sta.name,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      if (top.jQuery.devlis) {
        // Stop PMON monitor for the device.
        top.rpcmc_stopDeviceMonitor(data.name);

        // Detach handler for DeviceEvent.
        top.jQuery.devlis.unbind('DeviceEvent', HandleDeviceEvent);

        // Detach handler for SetLangEvent.
        top.jQuery.dictlis.unbind('SetLangEvent', HandleDictEvent);

        // Detach handler for WarnEvent.
        top.jQuery.devlis.unbind('WarnEvent', HandleWarnEvent);
      }
    } // remove
  };

  // Private functions

  function DrawStatbar(data) {
    var $dmyl2 = $('<label>', {id: 'l2_dmy', class: 'statlabel'});
    $dmyl2.css('visibility', 'hidden');
    $('.statdiv').append($dmyl2);
    $('#l2_dmy').html(data.device_newlines[1].buffer);

    var dispstr = data.device_newlines[1].buffer;
    while($('#stat1').width() < $("#l2_dmy").width()){
      var divpos = 0;
      if (data.device_tpprog_flg == 1)  {  // tp program
        // targets characters up to whitespace
        var omitchar = '';
        var searchchar = '&gt'; // '>'
        var delcnt = 1; // num of delete char
        if (dispstr.indexOf(searchchar) < 0) {
          omitchar = searchchar;
          searchchar = ' '; // whitespace
          delcnt = 2;
        }
        divpos = dispstr.indexOf(searchchar);
        if (divpos == 0) {
          // no targets
          break;
        }
        var tpprgstr = dispstr.slice(0, divpos);
        var jogstr = dispstr.slice(divpos);
        tpprgstr = tpprgstr.slice(0, tpprgstr.length - delcnt) + omitchar;
        dispstr = tpprgstr + jogstr;
      } else if (data.device_tpprog_flg == 0) { // alarm message
        // targets characters up to the <span>
        divpos = dispstr.indexOf('<span');
        if (divpos == 0) {
          // no targets
          break;
        }
        dispstr = dispstr.slice(0, divpos - 1) + dispstr.slice(divpos);
      } else {
        break;
      }
      $('#l2_dmy').html(dispstr);
    } // while

    $('#l1').html(data.device_newlines[0].buffer);
    $('#l2').html(dispstr);
    // remove dummy elem
    $dmyl2.remove();
  } // DrawStatbar

  // If device matches, populates the lines with the buffer.
  function HandleDeviceEvent(event, device, newlines, tpprog_flg) {
    var data = event.data || event;
    if ((device == data.name) && (newlines.length >= 2)) {
      data.device_newlines = newlines.concat();
      data.device_tpprog_flg = tpprog_flg;
      DrawStatbar(data);
      if (data.device_observer == null) {
        data.device_observer = new MutationObserver(function(mutations) {
          DrawStatbar(data);
        });
        var filter = ['style'];
        var options = {
          attributes: true,
          attributeFilter: filter
        };
        data.device_observer.observe($('#stat1')[0], options);
      }
    }
  } // HandleDeviceEvent

  function HandleDictEvent(event) {
    var data = event.data || event;
    // Change the font-family
    data.$this.css("font-family", top.getDefFont());

    if (top.g_logout) {
      $('#logout').html('<button id="logout">' + top.srvap_getString("LOGOUT_C") + '</button>');
      $('button, #logout').css('font-family', top.getDefFont());
    }
  } // HandleDictEvent

  function ToggleWarnIcon(data) {
    if(data.warn_icon_showing) {
      $('#warn_icon', data.$this).hide();
      data.warn_icon_showing = false;
    }
    else {
      $('#warn_icon', data.$this).show();
      data.warn_icon_showing = true;
    }
  } // ToggleWarnIcon

  function ClearWarnTimer(data) {
    if(data.warn_timer_id != null) {
      clearInterval(data.warn_timer_id);
      data.warn_timer_id = null;
    }
  } // ClearWarnTimer

  function HandleWarnEvent(event, img_src, warn_type, warn_url, interval) {
    var data = event.data || event;
    if(warn_type == PMEV_WARN_MENU_C) {
      $('#warn_icon', data.$this).attr("src", img_src);
      data.warn_icon_active = true;
      data.warn_url = warn_url;
      ClearWarnTimer(data);
      if(interval > 0) {
        data.warn_timer_id = setInterval(function(){
          ToggleWarnIcon(data);
        }, interval);
      }
      else {
        $('#warn_icon', data.$this).show();
        data.warn_icon_showing = true;
      }
    }
    else {
      $('#warn_icon', data.$this).hide();
      data.warn_icon_active = false;
      data.warn_url = null;
      data.warn_icon_showing = false;
      ClearWarnTimer(data);
    }
  } // HandleWarnEvent

})(jQuery);
