/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2011
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: jquery.res.js
 *  
 *  Description:
 *    Resource plugin
 *
 *  Author: Judy Evans
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  01-JAN-2011 EVANSJA Initial version.
 *  -
*/

;(function($) {
  $.res = {};

  /** Map holding bundle keys (if mode: 'map') */
  $.res.map = {};

  /**
  * Load and parse message bundle files (.res),
  * making bundles keys available as javascript variables.
  * 
  * res files are named <name>.res, or <name>_<language>.res or <name>_<language>_<country>.res
  * Where:
  *      The <language> argument is a valid ISO Language Code. These codes are the lower-case, 
  *      two-letter codes as defined by ISO-639. You can find a full list of these codes at a 
  *      number of sites, such as: http://www.loc.gov/standards/iso639-2/englangn.html
  *      The <country> argument is a valid ISO Country Code. These codes are the upper-case,
  *      two-letter codes as defined by ISO-3166. You can find a full list of these codes at a
  *      number of sites, such as: http://www.iso.ch/iso/en/prods-services/iso3166ma/02iso-3166-code-lists/list-en1.html
  * 
  * Sample usage for a bundles/Messages.res bundle:
  * $.fn.res({
  *      name:      'Messages', 
  *      language:  'en_US',
  *      path:      'bundles'
  * });
  * @param  name			(string/string[], required) names of file to load (eg, 'Messages' or ['Msg1','Msg2'])
  * @param  language		(string, optional) language/country code (eg, 'en', 'en_US', 'pt_PT'). if not specified, language reported by the browser will be used instead.
  * @param  path			(string, optional) path of directory that contains file to load
  * @param  mode			(string, optional) whether bundles keys are available as JavaScript variables/functions or as a map (eg, 'vars' or 'map')
  * @param  callback     (function, optional) callback function to be called after script is terminated
  */
  $.fn.res = function(settings) {
    // set up settings
    var defaults = {
      name: 'Messages',
      language: '',
      path: '',
      mode: 'vars',
      callback: function() { }
    };
    settings = $.extend(defaults, settings);
    //if (settings.language === null || settings.language == '') {
    //  settings.language = normaliseLanguageCode(navigator.language /* Mozilla */ || navigator.userLanguage /* IE */);
    // }
    if (settings.language === null) { settings.language = ''; }

    // load and parse bundle files
    var files = getFiles(settings.name);
    for (i = 0; i < files.length; i++) {

      // 2. with language code (eg, cgtp_ja.res)
      if (settings.language.length >= 2) {
        loadAndParseFile(settings.path + files[i] + '_' + settings.language.substring(0, 2) + '.res', settings.language, settings.mode);
      } else {
        // 1. load base (eg, cgtp.res)
        loadAndParseFile(settings.path + files[i] + '.res', settings.language, settings.mode);
      }
    }

    // call callback
    if (settings.callback) { settings.callback(); }
  };


  /**
  * When configured with mode: 'map', allows access to bundle values by specifying its key.
  * Eg, jQuery.res.text('com.company.bundles.menu_add')
  */
  $.res.text = function(key, placeHolderValues) {
    var value = $.res.map[key];
    if (value === null) { return key; }
    if (!placeHolderValues) {
      //if(key == 'spv.lbl.modified') {alert(value);}
      return value;
    } else {
      for (var i = 0; i < placeHolderValues.length; i++) {
        var regexp = new RegExp('\\{(' + i + ')\\}', "g");
        value = value.replace(regexp, placeHolderValues[i]);
      }
      return value;
    }
  };


  /** Load and parse .res files */
  function loadAndParseFile(filename, language, mode) {
    $.ajax({
      url: filename,
      async: false,
      contentType: "text/plain;charset=UTF-8",
      dataType: 'text',
      success: function(data, status) {
        var parsed = '';
        var parameters = data.split(/\n/);
        var regPlaceHolder = /(\{\d+\})/g;
        var regRepPlaceHolder = /\{(\d+)\}/g;
        var unicodeRE = /(\\u.{4})/ig;
        for (var i = 0; i < parameters.length; i++) {
          parameters[i] = parameters[i].replace(/^\s\s*/, '').replace(/\s\s*$/, ''); // trim
          if ((parameters[i].length > 0) &&
              (parameters[i].match("^#") != "#") && // skip comments
              (parameters[i] != "/*") &&
              (parameters[i] != "*/")) {
            var pair = parameters[i].split(',');
            if (pair.length > 0) {
              /** Process key & value */
              var name = unescape(pair[0]).replace(/^\s\s*/, '').replace(/\s\s*$/, ''); // trim
              name = name.replace(/"/g, ''); // remove quotation marks (")
            }
            if ((pair.length > 0) && (name != "")) {
              var value = pair.length == 1 ? "" : pair[1];
              //value = value.replace(/"/g, '\\"'); // escape quotation mark (")
              value = value.replace(/"/g, ""); // remove quotation marks (")
              value = value.replace(/^\s\s*/, '').replace(/\s\s*$/, ''); // trim  

              /** Mode: bundle keys in a map */
              if (mode == 'map' || mode == 'both') {
                // handle unicode chars possibly left out
                var unicodeMatches = value.match(unicodeRE);
                if (unicodeMatches) {
                  for (var u = 0; u < unicodeMatches.length; u++) {
                    value = value.replace(unicodeMatches[u], unescapeUnicode(unicodeMatches[u]));
                  }
                }
                // add to map
                $.res.map[name] = value;
              }

              /** Mode: bundle keys as vars/functions */
              if (mode == 'vars' || mode == 'both') {
                // make sure namespaced key exists (eg, 'some.key') 
                checkKeyNamespace(name);

                // value with variable substitutions
                if (regPlaceHolder.test(value)) {
                  var parts = value.split(regPlaceHolder);
                  // process function args
                  var first = true;
                  var fnArgs = '';
                  var usedArgs = [];
                  for (var p = 0; p < parts.length; p++) {
                    if (regPlaceHolder.test(parts[p]) && usedArgs.indexOf(parts[p]) == -1) {
                      if (!first) { fnArgs += ','; }
                      fnArgs += parts[p].replace(regRepPlaceHolder, 'v$1');
                      usedArgs.push(parts[p]);
                      first = false;
                    }
                  }
                  parsed += name + '=function(' + fnArgs + '){';
                  // process function body
                  var fnExpr = '"' + value.replace(regRepPlaceHolder, '"+v$1+"') + '"';
                  parsed += 'return ' + fnExpr + ';' + '};';

                  // simple value
                } else {
                  parsed += name + '="' + value + '";';
                }
              }
            }
          }
        }
        if (mode == 'vars' || mode == 'both') {
          eval(parsed);
        }
      }
    });
  }

  /** Make sure namespace exists (for keys with dots in name) */
  function checkKeyNamespace(key) {
    var regDot = /\./g;
    if (regDot.test(key)) {
      var fullname = '';
      var names = key.split(/\./);
      for (var i = 0; i < names.length; i++) {
        if (i > 0) { fullname += '.'; }
        fullname += names[i];
        if (eval('typeof ' + fullname + ' == "undefined"')) {
          eval(fullname + '={};');
        }
      }
    }
  }

  /** Make sure filename is an array */
  function getFiles(names) {
    return (names && names.constructor == Array) ? names : [names];
  }

  /** Ensure language code is in the format aa_AA. */
  function normaliseLanguageCode(lang) {
    lang = lang.toLowerCase();
    if (lang.length > 3) {
      lang = lang.substring(0, 3) + lang.substring(3).toUpperCase();
    }
    return lang;
  }

  /** Unescape unicode chars ('\u00e3') */
  function unescapeUnicode(str) {
    // unescape unicode codes
    var codes = [];
    var code = parseInt(str.substr(2), 16);
    if (code >= 0 && code < Math.pow(2, 16)) {
      codes.push(code);
    }
    // convert codes to text
    var unescaped = '';
    for (var i = 0; i < codes.length; ++i) {
      unescaped += String.fromCharCode(codes[i]);
    }
    return unescaped;
  }

})(jQuery);
