/*
*  This material is the joint property of FANUC Robotics America  and
*  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics
*  America or FANUC LTD Japan immediately upon request.  This material  and
*  the  information  illustrated or contained herein may not be reproduced,
*  copied, used, or transmitted in whole or in part in any way without  the
*  prior written consent of both FANUC Robotics America and FANUC LTD
*  Japan.
*
*           All Rights Reserved
*           Copyright (C)   2011
*           FANUC Robotics America
*           FANUC LTD Japan
*
*  +
*  Module: jquery.foc.js
*
*  Description:
*    FocusBar plugin
*
*  Author: Judy Evans
*          FANUC Robotics America
*          3900 W. Hamlin Road
*          Rochester Hills, Michigan    48309-3253
*
*  Modification history:
*  01-JAN-2011 EVANSJA     Initial version.
*  23-JUN-2016 Y.Amagata   Change icons for title bar
*  13-APR-2017 EVANSJA     pr50070 - Use DEF_FONT_C from RES files and change when language changes
*  02-MAY-2018 EVANSJA     pr50891 - Support for multiple panes
*  05-JUL-2018 EVANSJA     pr50997 - Use g_all_menus to determine if related_view and max_restore buttons are supported
*  12-JUL-2018 EVANSJA     pr50989 - Writing to #title1 needs to use .html instead of .text to avoid &gt;
*  25-JUL-2018 EVANSJA     pr51059 - Do not add or remove function keys when help/diag is being displayed.
*  12-OCT-2018 EVANSJA     pr51145 - Support Force HMI menu for Software Operator Panel.
*  14-JAN-2019 EVANSJA     pr51303a- Add MENU (if not iRProgrammer), DISP, FCTN, and change Related View icon.
*  05-MAR-2019 EVANSJA     pr51398 - Add menu_search_c to call tpmenus_list to search for menus.
*  18-MAR-2019 EVANSJA     pr51438 - Do not add or remove function keys when menu help is being displayed.
*  15-SEP-2020 Y.Kurihara  Hide all focus button for Tablet TP.
*  05-FEB-2021 FSI         Remove function keys when Force Control screen is being displayed.
*  11-MAR-2024 K.Teramoto  Remove function keys when laser sensor schedule screen is being displayed.
*  18-APR-2024 T.Furuta    Remove function keys when the Touch Sensing screens are being displayed.
*  16-JUL-2024 S.Kobayashi Remove function keys when the Paint Instruction Icons screens are being displayed.
*  27-NOV-2024 T.Furuta    Remove function keys when the Arc Teaching Setup screen is being displayed.
*  -
*/

;(function($) {

  //	Use these properties to customize settings...
  //	$.event.special.foc.name = 'TP';
  //	$.event.special.foc.device_id = 0;
  //
  //  No callback function, using defaults
  //  $('#foc1').foc();
  //
  //  No callback functions, specify parameters
  //  $('#foc1').foc({ name: 'TP', device_id: 0 });
  //
  //  Callback function, using defaults
  //  $('#foc1').bind('foc', function(event, device, newlines) {});
  //
  //  Callback functions, specify parameters
  //  $('#foc1').bind('foc', { name: 'TP2', device_id: 1 },
  //                             function(event, device, newlines) {});
  //
  $.fn.foc = function(settings) {
    $(this).bind('foc', settings,
      function(event, device, line, buffer) { });
  };

  $.ajaxSetup({
    // Disable caching of AJAX responses */
    cache: false
  });

  var TREEVIEW_DEV = 4;

  var foc = $.event.special.foc = {
    // FocusBar plugin defaults
    name: 'TP',
    device_id: 0,
    set_focus: 0,
    max_restore: ' ',
    zoom: '0',
    related_view: 0,
    help_view: 0,
    doc_title: false,
    doc_title_timeout: null,
    focusbar_area: 0,
    $this: null,

    // Called each time an element does bind
    add: function(handleObj) {
      handleObj.data = $.extend({
        name: foc.name,
        device_id: foc.device_id,
        set_focus: foc.set_focus,
        max_restore: foc.max_restore,
        zoom: foc.zoom,
        related_view: foc.related_view,
        help_view: foc.help_view,
        doc_title_timeout: null,
        focusbar_area: 0,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      data.$this.css("font-family", top.getDefFont());

      // Create 3 image buttons for Zoom, i, Maximize/Restore
      // Add spaces so they are easier to click with finger
      var h = $(window).height();
      var title_offset = 4;
      var but_offset = 2;
      if (h > 30) {
        title_offset += (h - 30) / 2;
        but_offset += (h - 30) / 2;
      }
      var out = '<div id="title1" style="position:absolute; top:' + title_offset + 'px;width:90%"/>';
      if (data.device_id == TREEVIEW_DEV) {
        if (top.g_all_menus) {
          // Create SEARCH button for menu search
          out += '<div style="position:absolute; right:40px; top:' + but_offset + 'px;">';
          out += '<a class="focusbutton" href="javascript:search_click(' + data.device_id + ');">';
          out += '&nbsp;<img id="tpdisp" class="focusimg" src="search.png" />&nbsp;</a>';
          out += '</div>';
        }
        // Create indicators to open/close tree
        if (undefined != parent.window.treemod_open) {
          out += '<div style="position:absolute; right:6px; top:' + but_offset + 'px;">';
          out += '<span id="closetree" class="indicator" onclick="tree_close()"><<</span>',
          out += '<span id="opentree" class="indicator" onclick="tree_open()">>></span>',
          out += '</div>';
        }
      }
      else {
        out += '<div style="position:absolute; right:0px; top:' + but_offset + 'px;">';
        if (top.g_all_menus) {
          if (!top.g_irprog) {
            // Create MENU button except for iRProgrammer since menus are in Tree.
            out += '<a class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 4);">';
            out += '&nbsp;<img id="tpdisp" class="focusimg" src="/frh/fcgtp/menu.png" />&nbsp;</a>';
          }
          if (data.device_id == 0) {
            // Create DISP and FCTN buttons for PRIM pane only
            out += '<a class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 5);">';
            out += '&nbsp;<img id="tpdisp" class="focusimg" src="disp.png" />&nbsp;</a>';
            out += '<a class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 6);">';
            out += '&nbsp;<img id="tpfctn" class="focusimg" src="fctn.png" />&nbsp;</a>';
          }
          // Create related_view button
          out += '<a class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 1);">';
          out += '&nbsp;<img id="tprelv" class="focusimg" src="relv.png" />&nbsp;</a>';
        }
        // Create zoom button
        out += '<a id="tpi_a" class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 2);">';
        out += '&nbsp;<img id="tpzm_gif" class="focusimg" src="/frh/cgtp/zoomin.png" />&nbsp;</a>';
        if (top.g_all_menus) {
          // Create max_restore button
          out += '<a class="focusbutton" href="javascript:focusimg_click(' + data.device_id + ', 3);">';
          out += '&nbsp;<img id="tpminmax" class="focusimg" src="/frh/cgtp/winrestr.png" />&nbsp;</a>';
        }
        out += '</div>';
      }
      data.$this.html(out);

      if (data.device_id == TREEVIEW_DEV) {
        // Hide open tree button
        $('#opentree').hide();
        return;
      }
      if (top.g_all_menus) {
        // Hide related view by default
        $("#tprelv", data.$this).hide();
      }
      if (top.g_tabtp && top.g_irprog) {
        // Hide all focusbutton for Tablet TP
        $('.focusbutton').css('display', 'none');
      }

      if (h <= 30) {
        $('.focusbar_focus').css('font-size', 'small');
        $('.focusbar_nofocus').css('font-size', 'small');
      }
      ChangeFocus(data);

      if (top.g_all_menus) {
        // Focusbar area is left of the first image
        data.focusbar_area = $('#tpi_a').offset().left - $('#tpi_a').width();
      }

      if (top.jQuery.devlis2) {
        if (data.device_id == 0) {
          // Send "configuration done" external request to GUID task.
          top.rpcmc_tpextreq(0, tp_config_done_c);
        }

        // Attach handler for DevEvent2.
        top.jQuery.devlis2.bind('DevEvent2', data, HandleDevEvent2);

        // Attach handler for ChangeFocusEvent.
        top.jQuery.fkylis.bind('ChangeFocusEvent', data, HandleChangeFocusEvent);

        // Start PMON monitor for the device.
        top.rpcmc_startDeviceMonitor(data.name);

        // Attach handler for SetLangEvent.
        top.jQuery.dictlis.bind('SetLangEvent', data, HandleDictEvent);

        // Attach handler for mousedown event.
        $('#title1', data.$this).mousedown(function(event) {
          if (event.pageX < data.focusbar_area) {
            if (data.device_id != top.g_device_id) {
              // Send "set focus" external request to GUID task.
              top.rpcmc_tpextreq2(data.device_id, tp_websetfocus_c, data.device_id);
              data.set_focus = 1;
            }
            else {
              data.set_focus = 0;
            }
          }
        });
      }
    }, // add

    // Called each time an element removes bind
    remove: function(handleObj) {
      handleObj.data = $.extend({
        name: foc.name,
        device_id: foc.device_id,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      if (data.device_id == TREEVIEW_DEV) {
        return;
      }
      if (top.jQuery.devlis2) {
        // Stop PMON monitor for the device.
        top.rpcmc_stopDeviceMonitor(data.name);

        // Detach handler for DevEvent2.
        top.jQuery.devlis2.unbind('DevEvent2', HandleDevEvent2);

        // Detach handler for SetLangEvent.
        top.jQuery.dictlis.unbind('SetLangEvent', HandleDictEvent);

        // Detach handler for ChangeFocusEvent.
        top.jQuery.fkylis.unbind('ChangeFocusEvent', HandleChangeFocusEvent);
      }
    } // remove
  };

  // Private functions

  // Change the CSS class based on focus.
  // RemoveClass and AddClass is not working in Google Chrome so just change the css directly
  function ChangeFocus(data) {
    if (data.device_id == top.g_device_id) {
      if (data.help_view) {
        $(data.$this).css('background-color', BGHelp_Color);
        $(data.$this).parent().css('background-color', BGHelp_Color);
        $(data.$this).css('color', 'black');
      }
      else {
        if (undefined != top.g_focusbg) {
          $(data.$this).css('background-color', top.g_focusbg);
          $(data.$this).parent().css('background-color', top.g_focusbg);
        }
        else {
          $(data.$this).css('background-color', DARK_GRAY);
          $(data.$this).parent().css('background-color', DARK_GRAY);
        }
        $(data.$this).css('color', 'white');
      }
    }
    else {
      if (data.related_view == 'l') {
        $(data.$this).css('background-color', TITLE_DELINK_BG_COLOR);
        $(data.$this).parent().css('background-color', TITLE_DELINK_BG_COLOR);
      }
      else {
        $(data.$this).css('background-color', LIGHT_GRAY);
        $(data.$this).parent().css('background-color', LIGHT_GRAY);
      }
      $(data.$this).css('color', DARK_GRAY);
    }
  } // ChangeFocus

  // Populates the text with the title from the document.
  function setTitleBar(data) {
    data.doc_title_timeout = null;
    var titlestr = "";
    try {
      switch (data.device_id) {
        case 0: titlestr = parent.prim.document.title;
          break;
        case 1: titlestr = parent.dual.document.title;
          break;
        case 2: titlestr = parent.third.document.title;
          break;
      }
    } catch (err) { /* nop */ };
    $('#title1', data.$this).html(top.g_hostname + '&nbsp;' + titlestr);
  }

  // If device matches, populates the text with the buffer.
  function HandleDevEvent2(event, device, newlines) {
    var data = event.data || event;
    if ((device == data.name) && (newlines[0].lin == 0)) {
      //top.rpcmc_rprintf("title " + newlines[0].buf);
      var buffer = newlines[0].buf;
      var buf_len = buffer.length;
      var help_view = data.help_view;
      if (buffer == "*U*") {
        if ((data.doc_title) && (data.doc_title_timeout == null)) {
          // Special URL event to refresh title if shown
          data.doc_title_timeout = setTimeout(function() { setTitleBar(data); }, 100);
        }
        // no more processing
        return;
      }
      data.help_view = false;
      if (((buffer.charAt(1) == 'B') || (buffer.charAt(1) == 'H')) && (buffer.charAt(2) == '*')) {
        data.doc_title = true;
        if (data.doc_title_timeout == null) {
          data.doc_title_timeout = setTimeout(function() { setTitleBar(data); }, 100);
        }
        if (buffer.charAt(1) == 'H') {
          data.help_view = true;
        }
      }
      else {
        data.doc_title = false;
        if (data.doc_title_timeout != null) {
          clearTimeout(data.doc_title_timeout);
          data.doc_title_timeout = null;
        }
        $('#title1', data.$this).html(top.g_hostname + '&nbsp;' + buffer.substring(0, buf_len - 3));
      }
      if (help_view != data.help_view) {
        ChangeFocus(data);
      }
      if (buf_len >= 3) {
        // Get the Zoom character.
        var org_zoom = data.zoom;
        data.zoom = buffer.charAt(buf_len - 3);
        if (org_zoom != data.zoom) {
          if (data.zoom == ' ') {
            $("#tpzm_gif", data.$this).hide();
          }
          else {
            // '0', '1', '2'
            if (data.zoom == '2') {
              $("#tpzm_gif", data.$this).attr("src", "/frh/cgtp/zoomout.png");
            }
            else {
              $("#tpzm_gif", data.$this).attr("src", "/frh/cgtp/zoomin.png");
            }
            $("#tpzm_gif", data.$this).show();
          }
        }

        // Get the Related View character.
        var org_view = data.related_view;
        if (buffer.charAt(buf_len - 2) == ' ') {
          data.related_view = 0;
        }
        else {
          // 'l' for linked, 'i' for related views available
          data.related_view = buffer.charAt(buf_len - 2);
        }
        if (org_view != data.related_view) {
          if ((data.device_id == 0) && (data.related_view)) {
            $("#tprelv", data.$this).show();
          }
          else {
            $("#tprelv", data.$this).hide();
          }
          ChangeFocus(data);
        }

        // Get the Maximize/Restore character.
        var org_max = data.max_restore;
        // ' ' for None, '0' for Maximize, '1' for Minimize
        data.max_restore = buffer.charAt(buf_len - 1);
        if (org_max != data.max_restore) {
          if ((data.max_restore == '1') || (data.max_restore == ' ')) {
            $("#tpminmax").attr("src", "/frh/cgtp/winrestr.png");
          }
          else {
            $("#tpminmax").attr("src", "/frh/cgtp/winmaxim.png");
          }
        }
      }
      // Trigger a callback for the element.
      data.$this.triggerHandler('foc', [device, newlines]);
    }
  } // HandleDevEvent2

  function HandleDictEvent(event) {
    var data = event.data || event;
    // Change the font-family
    data.$this.css("font-family", top.getDefFont());
  } // HandleDictEvent

  function HandleChangeFocusEvent(event) {
    ChangeFocus(event.data);
  } // HandleChangeFocusEvent

})(jQuery);

// Common routines for *mod.htm files

var first_time = 1;
var prim_height = 0; // without keys
var prim_width = 0;
var dual_width = 0;
var third_width = 0;
var prim_visible = true;
var dual_visible = true;
var third_visible = true;

function focusimg_click(device_id, value) {
  if (device_id != top.g_device_id) {
    // Send "set focus" external request to GUID task.
    top.rpcmc_tpextreq2(device_id, tp_websetfocus_c, device_id);
  }
  else {
    switch (value) {
    case 1:
      // Send Related View external request.
      top.rpcmc_tpextreq2(device_id, tp_thispanel_c, fctn_i_c);
      break;
    case 2:
      // Send Zoom external request.
      top.rpcmc_tpextreq(device_id, tp_zoom_c);
      break;
    case 3:
      // Send Maximize/Restore external request.
      top.rpcmc_tpextreq(device_id, tp_maxrestore_c);
      break;
    case 4:
      // Send MENU key to controller.
      top.rpcmc_sendKeyCode(menu_sel_c);
      break;
    case 5:
      // Send DISP key to controller.
      top.rpcmc_sendKeyCode(disp_s_c);
      break;
    case 6:
      // Send FUNC key to controller.
      top.rpcmc_sendKeyCode(cmn_func_c);
      break;
    }  // switch (value)
  }
}

function url_send_to_controller(url) {
  if ((url.toLowerCase().indexOf("/softpart/") >= 0) ||
      (url.toLowerCase().indexOf("/karelcmd/") >= 0) ||
      (url.toLowerCase().indexOf("/kcldo/") >= 0) ||
      (url.toLowerCase().indexOf("/smondo/") >= 0) ||
      (url.toLowerCase().indexOf("/frh/cgtp/blank.htm") >= 0) ||
      (url.toLowerCase().indexOf("javascript") >= 0)) {
    return false;
  }
  return true;
}

function prim_loaded() {
  // This will set $panestate to UI_LOADED
  top.rpcmc_tpextreq2(0, tp_webonload_c, tk_ui_loaded_c);

  if (top.g_force_hmi) {
    // Force HMI menu
    top.g_force_hmi = false;
    $.get('/softpart/genlink?connect_id=' + top.g_connect_id +
        '&config=single&prim=menupage,' + top.g_force_spid + ',' + top.g_force_scid, function() {
    });
  }
}

var primLoadChecker;
var primEmptyChecker;
function prim_connected() {
  try {
    var url = prim.location.href;
    clearTimeout(primLoadChecker);
    clearTimeout(primEmptyChecker);
    if (url_send_to_controller(url)) {
      // This will set $panestate to UI_CONNECTED
      top.rpcmc_tplink_new_url(0, url, 'prim');
      setTimeout(prim_loaded, LOADED_TIMEOUT);

      // Trigger a DevEvent2 on the dev event2 listeners.
      // *U* is special event to refresh title from document if shown
      top.jQuery.devlis2.trigger("DevEvent2", ["TP", [{ "lin": "0", "buf": "*U*"}]]);

      if (top.g_irprog && top.isFirstPrimLoad) {
        if (isFlexdevEmpty(top.mainfrm.prim) || isFlexuifEmpty()) {
          primEmptyChecker = setTimeout (function() {
            // If the page is still empty after several seconds, it displays default page.
            if (isFlexdevEmpty(top.mainfrm.prim) || isFlexuifEmpty()) {
              console.log("[iRProg]:(prim) The main content has been initialized because it was empty.");
              initMainCont("prim");
            }
            top.isFirstPrimLoad = false;
          },5000);
        } else {
          top.isFirstPrimLoad = false;
        }
      }
    } else if (top.g_irprog && top.isFirstPrimLoad) {
      primLoadChecker = setTimeout (function() {
        if (!url_send_to_controller(prim.location.href)) {
          console.log("[iRProg]:(prim) The main content has been initialized because it was blank.");
          initMainCont("prim");
        }
        top.isFirstPrimLoad = false;
      },5000);
    }

  } catch (err) { /* nop */ };

  if (top.g_irprog) {
    if (typeof check_keys !== "undefined") {
      var stat = need_funckeys(url);
      if (stat > -1) {
        if (stat == 0) {
          prim_visible = false;
        }
        else {
          prim_visible = true;
        }
        check_keys();
      }
    }
  }
}

function dual_loaded() {
  // This will set $panestate to UI_LOADED
  top.rpcmc_tpextreq2(1, tp_webonload_c, tk_ui_loaded_c);
}

var dualLoadChecker;
var dualEmptyChecker;
function dual_connected() {
  try {
    var url = dual.location.href;
    clearTimeout(dualLoadChecker);
    clearTimeout(dualEmptyChecker);
    if (url_send_to_controller(url)) {
      // This will set $panestate to UI_CONNECTED
      top.rpcmc_tplink_new_url(1, url, 'dual');
      setTimeout(dual_loaded, LOADED_TIMEOUT);

      // Trigger a DevEvent2 on the dev event2 listeners.
      // *U* is special event to refresh title from document if shown
      top.jQuery.devlis2.trigger("DevEvent2", ["TP2", [{ "lin": "0", "buf": "*U*"}]]);

      if(top.g_irprog) {
        if (isFlexdevEmpty(top.mainfrm.dual)) {
          dualEmptyChecker = setTimeout (function() {
            // If the page is still empty after several seconds, it displays default page.
            if (isFlexdevEmpty(top.mainfrm.dual)) {
              console.log("[iRProg]:(dual) The main content has been initialized because it was empty.");
              initMainCont("dual");
            }
          },5000);
        }
      }
    } else if (top.g_irprog) {
      dualLoadChecker = setTimeout (function() {
        if (!url_send_to_controller(dual.location.href)) {
          console.log("[iRProg]:(dual) The main content has been initialized because it was blank.");
          initMainCont("dual");
        }
      },5000);
    }
  } catch (err) { /* nop */ };

  if (top.g_irprog) {
    if (typeof check_keys !== "undefined") {
      var stat = need_funckeys(url);
      if (stat > -1) {
        if (stat == 0) {
          dual_visible = false;
        }
        else {
          dual_visible = true;
        }
        check_keys();
      }
    }
  }
}

function third_loaded() {
  // This will set $panestate to UI_LOADED
  top.rpcmc_tpextreq2(2, tp_webonload_c, tk_ui_loaded_c);
}

var thirdLoadChecker;
var thirdEmptyChecker;
function third_connected() {
  try {
    var url = third.location.href;
    clearTimeout(thirdLoadChecker);
    clearTimeout(thirdEmptyChecker);
    if (url_send_to_controller(url)) {
      // This will set $panestate to UI_CONNECTED
      top.rpcmc_tplink_new_url(2, url, 'third');
      setTimeout(third_loaded, LOADED_TIMEOUT);

      // Trigger a DevEvent2 on the dev event2 listeners.
      // *U* is special event to refresh title from document if shown
      top.jQuery.devlis2.trigger("DevEvent2", ["TP3", [{ "lin": "0", "buf": "*U*"}]]);

      if(top.g_irprog) {
        if (isFlexdevEmpty(top.mainfrm.third)) {
          thirdEmptyChecker = setTimeout (function() {
            // If the page is still empty after several seconds, it displays default page.
            if (isFlexdevEmpty(top.mainfrm.third)) {
              console.log("[iRProg]:(triple) The main content has been initialized because it was empty.");
              initMainCont("triple");
            }
          },5000);
        }
      }
    } else if (top.g_irprog) {
      thirdLoadChecker = setTimeout (function() {
        if (!url_send_to_controller(third.location.href)) {
          console.log("[iRProg]:(triple) The main content has been initialized because it was blank.");
          initMainCont("triple");
        }
      },5000);
    }
  } catch (err) { /* nop */ };

  if (top.g_irprog) {
    if (typeof check_keys !== "undefined") {
      var stat = need_funckeys(url);
      if (stat > -1) {
        if (stat == 0) {
          third_visible = false;
        }
        else {
          third_visible = true;
        }
        check_keys();
      }
    }
  }
}

// Checks if "FLEXUIF" is blank
// *FLEXUIF can be shown on only "prim", so dual/triple don't be check target
function isFlexuifEmpty() {
  if (top.mainfrm.prim.location.href.toLowerCase().match(/flexuif.htm/) == null) return false;

  // Blank check
  if (!url_send_to_controller(top.mainfrm.prim.ifrmPrim.location.href.toLowerCase())) {
    console.log("[iRProg]:(prim) The FLEXUIF is blank.");
    return true;
  }
  return false;
}

// Checks if "FLEXDEV" is blank/empty
function isFlexdevEmpty(mainFrame) {
  if (mainFrame.location.href.match(/flexdev.stm/) == null) return;

  // Empty check
  var isEmpty = true;
  $($(mainFrame.document).find('.lines')).each(function() {
    if (!$.trim($(this).text()) == "") {
      isEmpty = false;
      return false; // = break; (be careful)
    }
  })
  return isEmpty;
}

// Returns -1 for no change, 0 for no function keys, 1 for function keys
function need_funckeys(url_param) {
  if (undefined == url_param) {
    return -1;
  }
  var url = url_param.toLowerCase();
  // These are the web pages that do not change
  if ((url.indexOf("/frh/diag") >= 0) ||
      (url.indexOf("/frh/help") >= 0) ||
      (url.indexOf("/md/tpmenu") >= 0)) {
    return -1;
  }
  // These are the web pages that do not require function keys
  if ((url.indexOf("/irprog")                >= 0) ||
      (url.indexOf("/plugin")                >= 0) ||
      (url.indexOf("/jitpky2")               >= 0) ||
      (url.indexOf("/pmcedit")               >= 0) ||
      (url.indexOf("/karel/com")             >= 0) ||
      (url.indexOf("/forcecontrol/treemenu") >= 0) ||
      (url.indexOf("/pallettool")            >= 0) ||
      (url.indexOf("/arcweld")               >= 0) ||
      (url.indexOf("/laser_sensor")          >= 0) ||
      (url.indexOf("/touch_sensing")         >= 0) ||
      (url.indexOf("/painticons")            >= 0) ||
      (url.indexOf("/inst/arctch")           >= 0)) {
    return 0;
  }
  if ((url.indexOf("vsvtrn.stm") >= 0) ||
      (url.indexOf("vsset.stm") >= 0) ||
      (url.indexOf("vlfrmn.stm") >= 0) ||
      (url.indexOf("vimreg.stm") >= 0) ||
      (url.indexOf("vtuthead.stm") >= 0)) {
    return 0;
  }

  if ((!top.g_all_menus) && (url.indexOf("/touchj") >= 0)) {
    // The functions keys do not work properly and are not required.
    // However, if $IRPROG_CFG.$ALL_MENUS is TRUE then we require
    // the function keys.  Otherwise iHMI will get stuck in this
    // Guide and we cannot go back.
    return 0;
  }
  return 1;
} // need_funckeys


function tree_open() {
  $('#opentree').hide();
  $('#closetree').show();
  parent.window.treemod_open();
  $('#title1').text(top.g_hostname);
} // tree_open

function tree_close() {
  $('#opentree').show();
  $('#closetree').hide();
  parent.window.treemod_close();
  $('#title1').text('');
} // tree_close

function search_click(device_id) {
  // Send "menu search" external request to GUID task.
  top.rpcmc_tpextreq2(0, tp_focuspanel_c, menu_search_c);
}

function getHintUrl(trgtFrm) {
  var frmTyp = trgtFrm != undefined ? (top.mainfrm.location.href.match(/p\/.*\.htm/))[0].substring(2,10) : "default";
  var menupage = '=menupage,153,1';  // hintpage
  var frmTypObj = {
    "wholemod": '&config=single&',
    "doublmod": '&config=double&',
    "doubhmod": '&config=doubleh&',
    "triplmod": '&config=triple&',
    "triphmod": '&config=tripleh&',
    "default": 'current',
  };
  return '/softpart/genlink?' + frmTypObj[frmTyp] + (trgtFrm == null ? "" : trgtFrm) + menupage;
}
function initMainCont(trgtFrm) {
  var initUrl = getHintUrl(trgtFrm);
  if (typeof initUrl === 'string') {
    if (top.g_connect_id) {
      if (initUrl.indexOf('/softpart/genlink?') == 0) {
        var urlStr;
        urlStr = initUrl.substring(0, 18) + 'connect_id=' + top.g_connect_id + '&' + initUrl.substring(18);
        $.get(urlStr, function() { });
      }
      else {
        $.get(initUrl, function() { });
      }
    }
    if ((top.g_startmd != CTRLSTART) && (initUrl != "")) {
      var currentFrameId = top.getMyFrameId();
      var urlStr;
      urlStr = initUrl.replace(/&/g, '^');
      if (top.treemod.primScreenId != null) {
        if (trgtFrm == "prim") {
          if (urlStr != top.IHMI_EDITOR_ADDRESS.replace(/&/g, '^')) {
            top.chgProgCB = undefined; // If it is called by other than the CRX editor, an error occurs.
          }
          top.treemod.setUrlList(urlStr);
          if (currentFrameId != "prim" && typeof top.statbar.hideBackBtn != 'undefined') {
            top.statbar.hideBackBtn();
          }
        }
      }
      if (top.treemod.primScreenId != null) {
        if (trgtFrm == "prim") {
          top.treemod.primScreenId = urlStr;
        }
        else if (trgtFrm == "dual") {
          top.treemod.dualScreenId = urlStr;
        }
        else {
          top.treemod.thirdScreenId = urlStr;
        }
      }
      else {
        top.treemod.primScreenId = urlStr;
      }
      if (trgtFrm == "prim") {
        top.rpcmc_setVar(SYSNAME_C, "$IRPROG_CFG.$CURR_URL", urlStr);
      }
    }
    if (trgtFrm == "prim") {
      top.treemod.update_tree_url(getHintUrl());
    }
  }
}
