/*
*  This material is the joint property of FANUC Robotics America  and
*  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics
*  America or FANUC LTD Japan immediately upon request.  This material  and
*  the  information  illustrated or contained herein may not be reproduced,
*  copied, used, or transmitted in whole or in part in any way without  the
*  prior written consent of both FANUC Robotics America and FANUC LTD
*  Japan.
*
*           All Rights Reserved
*           Copyright (C)   2011
*           FANUC Robotics America
*           FANUC LTD Japan
*
*  +
*  Module: jquery.fky.js
*
*  Description:
*    Function Key plugin
*
*  Author: Judy Evans
*          FANUC Robotics America
*          3900 W. Hamlin Road
*          Rochester Hills, Michigan    48309-3253
*
*  Modification history:
*  01-JAN-2011 EVANSJA Initial version.
*  23-NOV-2015 EVANSJA pr48789 - Support color on the function keys.
*  04-JAN-2017 EVANSJA Update for V9.0.
*  16-JAN-2018 EVANSJA pr50656 - Support for function key viewtype.
*  09-APR-2018 EVANSJA pr50839 - IRPROG: File Menu needs F1 key for DELETE and DO_COPY
*  18-JUN-2018 EVANSJA pr50988 - IRPROG: Add EXIT HELP to F5 key
*  05-JUL-2018 EVANSJA pr50997 - Use g_all_menus to determine if F1 [ TYPE ] is supported
*  -
*/

; (function($) {

  //  Usage
  //  $('#f4').fky({ fkyindex: 4, key_code: f4_c,
  //    tptext0: '<!--#echo dict=ipkey ele=ip_refresh_c -->',
  //    tpimg0:  '/frh/cgtp/brwsrefr.png',
  //    tptext1: '<!--#echo dict=ipkey ele=ip_stop_c -->',
  //    tpimg1:  '/frh/cgtp/brwsstop.png',
  //    helptext0: '<!--#echo dict=ipkey ele=ip_refresh_c -->' });
  //    helptext1: '<!--#echo dict=ipkey ele=ip_stop_c -->' });
  //  fkyindex is the key index (0 - 6)
  //  key_code is from wdkcodes.hc
  //  tptext0 is the first page of BROWSER keys
  //  tpimg0 is the first page of BROWSER images
  //  tptext1 is the next page of BROWSER keys
  //  tpimg1 is the second page of BROWSER keys
  //  helptext0 is the first page of HELP_BROWSER keys
  //  helptext1 is the next page of HELP_BROWSER keys
  //  key_class is the class for the button
  //
  $.fn.fky = function(settings) {
    $(this).bind('fky', settings, function() { });
  };

  // Constants
  var TPFK_NAME = 'TPFK';
  var DEVICE = 0;
  var BROWSER = 1;
  var HELP_BROWSER = 2;
  var SYSVAR = 3;

  // Variables
  var fkey_state = DEVICE;
  var next_page = 0;

  var fky = $.event.special.fky = {
    // fky plugin defaults
    fkyindex: 0,
    key_code: 0,
    tptext0: '',
    tpimg0: '',
    tptext1: '',
    tpimg1: '',
    helptext0: '',
    helptext1: '',
    key_class: 'fky_button',
    $this: null,

    // Called each time an element does bind
    add: function(handleObj) {
      handleObj.data = $.extend({
        fkyindex: fky.fkyindex,
        key_code: fky.key_code,
        tptext0: fky.tptext0,
        tpimg0: fky.tpimg0,
        tptext1: fky.tptext1,
        tpimg1: fky.tpimg1,
        helptext0: fky.helptext0,
        helptext1: fky.helptext1,
        key_class: fky.key_class,
        $this: $(this)
      }, handleObj.data || {});

      // Create button inside the element.
      var data = handleObj.data;
      data.$this.html('<button type="button" class="' + data.key_class + '"></button>');

      if (top.jQuery.fkylis) {
        // Attach handler for DeviceEvent.
        // Only the first function key needs to get the event
        if (data.fkyindex == 1) {
          top.jQuery.fkylis.bind('DeviceEvent', data, HandleDeviceEvent);

          // Start PMON monitor for the device.
          top.rpcmc_startDeviceMonitor(TPFK_NAME);

          // Attach handler for KeyPressedEvent.
          top.jQuery.keylis.bind('KeyPressedEvent', HandleKeyEvent);
        }

        // Attach handler for DevDrawEvent.
        $(data.$this).bind('DevDrawEvent', data, HandleDevDrawEvent);

        // Attach handler for DrawEvent.
        $(data.$this).bind('DrawEvent', data, HandleDrawEvent);

        // Attach handler to button for click event.
        $('button', data.$this).click(function(event) {
          try {
            event.preventDefault();
            if ((data.fkyindex >= 1) && (data.fkyindex <= 5)) {
              // Check for cgop control
              var cgop_index = data.fkyindex - 1;
              if (next_page == 1) {
                cgop_index += 5;
              }
              if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
                // Send event to cgop control
                top.jQuery.fkylis.trigger("FKeyClickEvent", ["CGOP", cgop_index]);
                return;
              }
            }
            if (data.key_code === flexuif_home_c) {
              if (top.g_all_menus) {
                // Support 2 byte key codes
                top.rpcmc_tpextreq2(0, tp_focuspanel_c, data.key_code);
              }
            }
            else if (fkey_state === DEVICE) {
              if (data.key_code > 0) {
                if ((!top.g_all_menus) && (data.key_code === f1_c) && ($('button', data.$this).html() == "")) {
                  // Do not allow F1 [ TYPE ]
                }
                else {
                  // Send key to controller.
                  top.rpcmc_sendKeyCode(data.key_code);
                }
              }
            }
            else if ((next_page == 0) && (fkey_state === HELP_BROWSER) && (data.key_code == f5_c)) {
              // Exit help - Send prev to controller.
              top.rpcmc_sendKeyCode(prev_c);
            }
            else if ((data.key_code === f1_c) || (data.key_code === prev_c) || (data.key_code === next_c)) {
              // Handled by controller
              top.rpcmc_sendKeyCode(data.key_code);
            }
            else if (data.key_code === f2_c) {
              if (next_page === 0) {
                // BACK
                if (top.g_device_id === 1) {
                  parent.mainfrm.dual.window.history.back();
                }
                else if (top.g_device_id === 2) {
                  parent.mainfrm.third.window.history.back();
                }
                else {
                  parent.mainfrm.prim.window.history.back();
                }
              }
              else if ((next_page == 1) && (fkey_state == BROWSER)) {
                // ZOOM - TBD
                //top.rpcmc_tpextreq(top.g_device_id, tp_zoom_c);
              }
            }
            else if (data.key_code === f3_c) {
              if (next_page === 0) {
                // FORWARD
                if (top.g_device_id === 1) {
                  parent.mainfrm.dual.window.history.forward();
                }
                else if (top.g_device_id === 2) {
                  parent.mainfrm.third.window.history.forward();
                }
                else {
                  parent.mainfrm.prim.window.history.forward();
                }
              }
              else if ((next_page == 1) && (fkey_state == BROWSER)) {
                // CLR CACHE - May not be possible
              }
            }
            else if (data.key_code === f4_c) {
              if (next_page === 0) {
                // REFRESH
                if (top.g_device_id === 1) {
                  parent.mainfrm.dual.location.reload(true);
                }
                else if (top.g_device_id === 2) {
                  parent.mainfrm.third.location.reload(true);
                }
                else {
                  parent.mainfrm.prim.location.reload(true);
                }
              }
              else if (next_page === 1) {
                // STOP
                if (top.g_device_id === 1) {
                  if (top.g_msie) {
                    parent.mainfrm.dual.document.execCommand('Stop');
                  }
                  else {
                    parent.mainfrm.dual.window.stop();
                  }
                }
                else if (top.g_device_id === 2) {
                  if (top.g_msie) {
                    parent.mainfrm.third.document.execCommand('Stop');
                  }
                  else {
                    parent.mainfrm.third.window.stop();
                  }
                }
                else {
                  if (top.g_msie) {
                    parent.mainfrm.prim.document.execCommand('Stop');
                  }
                  else {
                    parent.mainfrm.prim.window.stop();
                  }
                }
              }
            }
            else if ((data.key_code === f5_c) && (fkey_state == BROWSER)) {
              if (next_page === 0) {
                // HOME
                if (top.g_device_id === 1) {
                  parent.mainfrm.dual.location.href = "../../";
                }
                else if (top.g_device_id === 2) {
                  parent.mainfrm.third.location.href = "../../";
                }
                else {
                  parent.mainfrm.prim.location.href = "../../";
                }
              }
              else if (next_page === 1) {
                // FAVORITES
                if (top.g_device_id === 1) {
                  parent.mainfrm.dual.location.href = "../../karel/peeritp";
                }
                else if (top.g_device_id === 2) {
                  parent.mainfrm.third.location.href = "../../karel/peeritp";
                }
                else {
                  parent.mainfrm.prim.location.href = "../../karel/peeritp";
                }
              }
            }
          } catch (err) { /* nop */ };

          // do not retain focus
          $('button', data.$this).blur();

        });
      }
    }, // add

    // Called each time an element removes bind
    remove: function(handleObj) {
      handleObj.data = $.extend({
        fkyindex: fky.fkyindex,
        key_code: fky.key_code,
        tptext0: fky.tptext0,
        tpimg0: fky.tpimg0,
        tptext1: fky.tptext1,
        tpimg1: fky.tpimg1,
        helptext0: fky.helptext0,
        helptext1: fky.helptext1,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      if (top.jQuery.fkylis) {
        // Only the first function key handles the events
        if (data.fkyindex == 1) {
          // Stop PMON monitor for the device.
          top.rpcmc_stopDeviceMonitor(TPFK_NAME);

          // Detach handler for DeviceEvent.
          top.jQuery.fkylis.unbind('DeviceEvent', HandleDeviceEvent);

          // Detach handler for KeyPressedEvent.
          top.jQuery.keylis.unbind('KeyPressedEvent', HandleKeyEvent);
        }
      }
    } // remove
  };

  // Private functions

  // If device matches, populates the button text with the buffer.
  function HandleDeviceEvent(event, device, newlines) {
    var buffer;
    var cgop_index;
    var data = event.data || event;
    // Function key labels are split by COMET into newlines.
    if ((device == TPFK_NAME) && (newlines.length >= 6)) {
      // check first function key label
      buffer = newlines[0].buffer;
      if ((buffer.charAt(1) == 'B') && (buffer.charAt(2) == '*')) {
        DisplayBrowserKeys();
        return;
      }
      if ((buffer.charAt(1) == 'H') && (buffer.charAt(2) == '*')) {
        DisplayHelpKeys();
        return;
      }
      fkey_state = DEVICE;
      $('#home').trigger("DrawEvent");
      $('#f1').trigger("DevDrawEvent", buffer);
      $('#f2').trigger("DevDrawEvent", newlines[1].buffer);
      $('#f3').trigger("DevDrawEvent", newlines[2].buffer);
      $('#f4').trigger("DevDrawEvent", newlines[3].buffer);
      $('#f5').trigger("DevDrawEvent", newlines[4].buffer);

      // Save next indicator on line 6
      top.gPaneFKey[top.g_device_id].FKeyDevNext = newlines[5].buffer;
    }
    else if (device == "CGOP") {
      if (((next_page == 0) && (newlines >= 2) && (newlines <= 5)) ||
          ((next_page == 1) && (newlines >= 6) && (newlines <= 10))) {
        cgop_index = newlines - 1;
        if ((newlines >= 6) && (newlines <= 10)) {
          newlines -= 5;
        }
        if ((top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) || (fkey_state != DEVICE)) {
          $('#f' + newlines).trigger("DrawEvent");
        }
        else {
          // FkeyListener has been removed from DEVICE state
          $('#f' + newlines).trigger("DevDrawEvent", top.gPaneFKey[top.g_device_id].FKeyDevText[cgop_index]);
        }
      }
    }
    else {
      return;
    }
    // check next indicator
    if (next_page == 1) {
      $('#next').trigger("DrawEvent");
    }
    else {
      if (top.gPaneFKey[top.g_device_id].FKeyDevNext == '>') {
        $('#next').trigger("DrawEvent");
      }
      else {
        // Look for FKeyListener on 2nd page
        for (cgop_index = 6; cgop_index <= 10; cgop_index++) {
          if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
            $('#next').trigger("DrawEvent");
            break;
          }
        }
        if (cgop_index > 10) {
          // No FKeyListener found, use original buffer (FKeyDevNext)
          $('#next').trigger("DevDrawEvent", top.gPaneFKey[top.g_device_id].FKeyDevNext);
        }
      }
    }
  } // HandleDeviceEvent

  // Populates the button text with the buffer.
  function HandleDevDrawEvent(event, buffer) {
    var data = event.data || event;
    if ((!top.g_all_menus) && (data.key_code === f1_c) &&
        ((buffer.indexOf('[') >= 0) || (buffer.indexOf('|') >= 0))) {
      // Do not allow F1 [ TYPE ]
      $('button', data.$this).html("");
    }
    else {
      if ((data.fkyindex >= 1) && (data.fkyindex <= 5)) {
        // Check for cgop control
        var cgop_index = data.fkyindex - 1;
        if (next_page == 1) {
          cgop_index += 5;
        }
        if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
          if (top.gPaneFKey[top.g_device_id].FKeyImgIcon[cgop_index] != null) {
            $('button', data.$this).html('<img src=' + top.gPaneFKey[top.g_device_id].FKeyImgIcon[cgop_index] + ' /><br>' + top.gPaneFKey[top.g_device_id].FKeyText[cgop_index]);
          }
          else {
            $('button', data.$this).text(top.gPaneFKey[top.g_device_id].FKeyText[cgop_index]);
          }
          if (top.gPaneFKey[top.g_device_id].FKeyBackColor[cgop_index] != null) {
            $('button', data.$this).css('background-color', top.gPaneFKey[top.g_device_id].FKeyBackColor[cgop_index]);
          }
          if (top.gPaneFKey[top.g_device_id].FKeyTextColor[cgop_index] != null) {
            $('button', data.$this).css('color', top.gPaneFKey[top.g_device_id].FKeyTextColor[cgop_index]);
          }
          // Save the buffer for later
          top.gPaneFKey[top.g_device_id].FKeyDevText[cgop_index] = buffer;
          return;
        }
      }
      $('button', data.$this).html(buffer);
      top.gPaneFKey[top.g_device_id].FKeyDevText[cgop_index] = "";
    }

    // Set the button background color to the span background color if defined
    var bgcolor =  $('span', data.$this).css('background-color');
    if (undefined != bgcolor) {
      $('button', data.$this).css('background-color', bgcolor);
    }
    else {
      /* Make the button black if there is no content */
      if (buffer.trim().length == 0) {
        $('button', data.$this).css('background-color', BRIGHT_GRAY);
      }
      else {
        $('button', data.$this).css('background-color', MEDIUM_GRAY);
      }
    }
    $('button', data.$this).css('color', 'white');
  }  // HandleDevDrawEvent

  // Populates the button text with localized Browser or Help text or cgop control
  function HandleDrawEvent(event) {
    var data = event.data || event;
    if ((data.fkyindex >= 1) && (data.fkyindex <= 5)) {
      // Check for cgop control
      var cgop_index = data.fkyindex - 1;
      if (next_page == 1) {
        cgop_index += 5;
      }
      if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
        if (top.gPaneFKey[top.g_device_id].FKeyImgIcon[cgop_index] != null) {
          $('button', data.$this).html('<img src=' + top.gPaneFKey[top.g_device_id].FKeyImgIcon[cgop_index] + ' /><br>' + top.gPaneFKey[top.g_device_id].FKeyText[cgop_index]);
        }
        else {
          $('button', data.$this).text(top.gPaneFKey[top.g_device_id].FKeyText[cgop_index]);
        }
        if (top.gPaneFKey[top.g_device_id].FKeyBackColor[cgop_index] != null) {
          $('button', data.$this).css('background-color', top.gPaneFKey[top.g_device_id].FKeyBackColor[cgop_index]);
        }
        if (top.gPaneFKey[top.g_device_id].FKeyTextColor[cgop_index] != null) {
          $('button', data.$this).css('color', top.gPaneFKey[top.g_device_id].FKeyTextColor[cgop_index]);
        }
        return;
      }
    }
    if (fkey_state == BROWSER) {
      if (next_page == 0) {
        if (data.tpimg0 != '') {
          $('button', data.$this).html('<img src=' + data.tpimg0 + ' /><br>' + data.tptext0);
        }
        else {
          $('button', data.$this).text(data.tptext0);
        }
      }
      else {
        if (data.tpimg1 != '') {
          $('button', data.$this).html('<img src=' + data.tpimg1 + ' /><br>' + data.tptext1);
        }
        else {
          $('button', data.$this).text(data.tptext1);
        }
      }
    }
    else {
      if (next_page == 0) {
        if ((data.tptext0 == data.helptext0) && (data.tpimg0 != '')) {
          $('button', data.$this).html('<img src=' + data.tpimg0 + ' /><br>' + data.tptext0);
        }
        else if ((fkey_state == HELP_BROWSER) && (data.key_code == f5_c)) {
            // Exit help
          $('button', data.$this).html('<img src="/frh/cgtp/help.png" /><br>' + data.helptext0);
        }
        else {
          $('button', data.$this).text(data.helptext0);
        }
      }
      else {
        if ((data.tptext1 == data.helptext1) && (data.tpimg1 != '')) {
          $('button', data.$this).html('<img src=' + data.tpimg1 + ' /><br>' + data.tptext1);
        }
        else {
          $('button', data.$this).text(data.helptext1);
        }
      }
    }
    $('button', data.$this).css('background-color', MEDIUM_GRAY);
    $('button', data.$this).css('color', 'white');
  }  // HandleDrawEvent

  // Display Browser keys
  function DisplayBrowserKeys() {
    fkey_state = BROWSER;
    $('#home').trigger("DrawEvent");
    $('#f1').trigger("DrawEvent");
    $('#f2').trigger("DrawEvent");
    $('#f3').trigger("DrawEvent");
    $('#f4').trigger("DrawEvent");
    $('#f5').trigger("DrawEvent");
    $('#next').trigger("DrawEvent");
    top.gPaneFKey[top.g_device_id].FKeyDevNext = '>';
  } // DisplayBrowserKeys

  // Display Help keys
  function DisplayHelpKeys(event) {
    fkey_state = HELP_BROWSER;
    $('#home').trigger("DrawEvent");
    $('#f1').trigger("DrawEvent");
    $('#f2').trigger("DrawEvent");
    $('#f3').trigger("DrawEvent");
    $('#f4').trigger("DrawEvent");
    $('#f5').trigger("DrawEvent");
    $('#next').trigger("DrawEvent");
    top.gPaneFKey[top.g_device_id].FKeyDevNext = '>';
  } // DisplayHelpKeys

  // If the given hardkey matches next, then handle the Next event
  function HandleKeyEvent(event, key) {
    if ((key == next_c) || (key == next_s_c)) {
      if ((fkey_state == BROWSER) || (fkey_state == HELP_BROWSER)) {
        if (next_page == 0) {
          next_page = 1;
        }
        else {
          next_page = 0;
        }
        if (fkey_state == BROWSER) {
          DisplayBrowserKeys();
        }
        else {
          DisplayHelpKeys();
        }
        return;
      }
      // fkey_state == DEVICE
      // Normally controller will send the keys to display unless they are cgop controls
      var save_next_page = next_page;
      if (next_page == 1) {
        next_page = 0;
      }
      else {
        // Check for cgop control
        var cgop_index;
        for (cgop_index = 6; cgop_index <= 10; cgop_index++) {
          if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
            if (next_page == 0) {
              next_page = 1;
            }
            else {
              next_page = 0;
            }
            break;
          }
        }
      }
      if (save_next_page != next_page) {
        if (next_page == 1) {
          $('#f1').trigger("DrawEvent");
        }
        $('#f2').trigger("DrawEvent");
        $('#f3').trigger("DrawEvent");
        $('#f4').trigger("DrawEvent");
        $('#f5').trigger("DrawEvent");
      }
    }
    else {
      var index = 0;
      switch (parseInt(key)) {
      case f1_c:
      case f1_s_c:
        index = 1;
        break;
      case f2_c:
      case f2_s_c:
        index = 2;
        break;
      case f3_c:
      case f3_s_c:
        index = 3;
        break;
      case f4_c:
      case f4_s_c:
        index = 4;
        break;
      case f5_c:
      case f5_s_c:
        index = 5;
        break;
      }
      if (index > 0) {
        // Check for cgop control
        var cgop_index = index - 1;
        if (next_page == 1) {
          cgop_index += 5;
        }
        if (top.gPaneFKey[top.g_device_id].FKeyListener[cgop_index]) {
          // Send event to cgop control
          top.jQuery.fkylis.trigger("FKeyClickEvent", ["CGOP", cgop_index]);
          return;
        }
      }
      if (((fkey_state == BROWSER) || (fkey_state == HELP_BROWSER)) && (index > 1)) {
        // Trigger a click event on the button
        $('button', '#f' + index).trigger('click');
      }
    }
  } // HandleKeyEvent

})(jQuery);
