/*
 *  This material is the joint property of FANUC Robotics America  and
 *  FANUC  LTD  Japan,  and  must be returned to either FANUC Robotics 
 *  America or FANUC LTD Japan immediately upon request.  This material  and
 *  the  information  illustrated or contained herein may not be reproduced,
 *  copied, used, or transmitted in whole or in part in any way without  the
 *  prior written consent of both FANUC Robotics America and FANUC LTD
 *  Japan.
 *  
 *           All Rights Reserved
 *           Copyright (C)   2011
 *           FANUC Robotics America
 *           FANUC LTD Japan
 * 
 *  +
 *  Module: jquery.dev.js
 *  
 *  Description:
 *    Device plugin
 *
 *  Author: Judy Evans
 *          FANUC Robotics America
 *          3900 W. Hamlin Road
 *          Rochester Hills, Michigan    48309-3253
 *  
 *  Modification history:
 *  01-JAN-2011 EVANSJA Initial version.
 *  06-APR-2017 EVANSJA pr50047 - Change the font size to fit at least 40 chars
 *  13-APR-2017 EVANSJA pr50070 - Use DEF_FONT_C from RES files and change when language changes
 *  26-OCT-2017 DOWMW   pr50492 - Update <DIR> so it properly displays
 *  27-OCT-2017 DOWMW   pr50492 - Moved logic to cmt_pmev.c to work with all < and >
 *  02-MAY-2018 EVANSJA pr50891 - No need to check for resize_dev since mode_resize will do it
 *  08-JAN-2019 EVANSJA pr51304 - Support resizing the font for User menu.
 *  25-JUN-2019 EVANSJA pr51594 - Prevent page up/down from skipping 2 pages. 
 *  02-OCT-2019 EVANSJA pr51895 - Recheck the row/col limits after changing the font size.
 *  -
*/

;(function($) {

  //	Use these properties to customize settings...
  //	$.event.special.dev.name = 'TP';
  //	$.event.special.dev.device_id = 0;
  // 
  //  No callback function, using defaults
  //  $('#dev1').dev();
  //
  //  No callback functions, specify parameters
  //  $('#dev1').dev({ name: 'TP', device_id: 0 });
  //
  //  Callback function, using defaults
  //  $('#dev1').bind('dev', function(event, device, newlines) {});
  //
  //  Callback functions, specify parameters
  //  $('#dev1').bind('dev', { name: 'TP2', device_id: 1 },
  //                           function(event, device, newlines) {});
  //
  $.fn.dev = function(settings) {
    $(this).bind('dev', settings,
      function(event, device, newlines) { });
  };

  var dev = $.event.special.dev = {
    // Device plugin defaults
    name: 'TP',
    device_id: 0,
    lines: 0,
    columns: 0,
    charh: 0,
    charw: 0,
    winh: 0,
    winw: 0,
    resize_font: false,
    resize_rows: 0,
    resize_cols: 0,
    $this: null,

    // Called each time an element does bind
    add: function(handleObj) {
      handleObj.data = $.extend({
        name: dev.name,
        device_id: dev.device_id,
        lines: dev.lines,
        columns: dev.columns,
        charh: dev.charh,
        charw: dev.charw,
        winh: dev.winh,
        winw: dev.winw,
        resize_font: dev.resize_font,
        resize_rows: dev.resize_rows,
        resize_cols: dev.resize_cols,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      data.$this.css("font-family", top.getFixFont());

      // resize_dev will create lines inside the element.
      resize_dev(data);

      if (top.jQuery.devlis2) {
        // Attach handler for DevEvent2.
        top.jQuery.devlis2.bind('DevEvent2', data, HandleDevEvent2);

        // Start PMON monitor for the device.
        top.rpcmc_startDeviceMonitor(data.name);

        // Attach handler for SetLangEvent.
        top.jQuery.dictlis.bind('SetLangEvent', data, HandleDictEvent);

        // Attach handler for mousedown event.
        data.$this.mousedown(function(event) {
          if (data.device_id != top.g_device_id) {
            // Send "set focus" external request to GUID task.
            top.rpcmc_tpextreq2(data.device_id, tp_websetfocus_c, data.device_id);
          }
        });
      }
    }, // add

    // Called each time an element removes bind
    remove: function(handleObj) {
      handleObj.data = $.extend({
        name: dev.name,
        device_id: dev.device_id,
        lines: dev.lines,
        columns: dev.columns,
        charh: dev.charh,
        charw: dev.charw,
        winh: dev.winh,
        winw: dev.winw,
        $this: $(this)
      }, handleObj.data || {});

      var data = handleObj.data;
      if (top.jQuery.devlis2) {
        // Stop PMON monitor for the device.
        top.rpcmc_stopDeviceMonitor(data.name);

        // Detach handler for DevEvent2.
        top.jQuery.devlis2.unbind('DevEvent2', HandleDevEvent2);

        // Detach handler for SetLangEvent.
        top.jQuery.dictlis.unbind('SetLangEvent', HandleDictEvent);
      }
    } // remove
  };

  // Private functions

  // If device matches, populates the lines with the buffer.
  function HandleDevEvent2(event, device, newlines) {
    var data = event.data || event;
    var idx;
    var line;
    var n_lines;
    var buffer;
    var charh;
    var max_charh;

    if ((device == data.name) && (top.getDevH(data.device_id))) {
      if ((data.winh != top.getDevH(data.device_id)) || 
          (data.winw != top.getDevW(data.device_id))) {
        // Wait for resize_dev to be called
        return;
      }
      n_lines = newlines.length;
      charh = 0;
      max_charh = 0;
      for (idx = 0; idx < n_lines; idx++) {
        line = newlines[idx].lin;
        // skip first line which is the title
        if (line > 0 && line <= data.lines) {
          // Some browsers do not support white-space: pre
          buffer = newlines[idx].buf;
          buffer = buffer.replace(/ /g, '&nbsp;');
          buffer = buffer.replace(/span&nbsp;/g, 'span ');
          $('#l' + line, data.$this).html(buffer);
        }
      }
      // Trigger a callback for the element.
      data.$this.triggerHandler('dev', [device, newlines]);

    }
  } // HandleDevEvent2

  function HandleDictEvent(event) {
    var data = event.data || event;
    if (top.g_irprog) {
      location.reload();
      return;
    }
    // Change the font-family
    data.$this.css("font-family", top.getFixFont());
  } // HandleDictEvent

  // Creates the device based on the size of the window.
  function resize_dev(data) {
    var out;
    var resize_font_enabled;
    if (!top.getDevH(data.device_id)) {
      // Wait until iframe_size is called in *mod*.htm
      //top.rpcmc_rprintf("[jquery.dev.js] resize_dev height not set for " + data.device_id);
      setTimeout(function() { resize_dev(data); }, 0);
      return;
    }
    resize_font_enabled = false;
    if ((data.winh != top.getDevH(data.device_id)) ||
        (data.winw != top.getDevW(data.device_id))) {
      data.winh = top.getDevH(data.device_id);
      data.winw = top.getDevW(data.device_id);

      // Get the height of a line and the width of a space 
      out = '<div class="lines" id="l1"><span id="s1">&nbsp;</span></div>';
      if (top.g_android && ((top.g_lang_suffix == 'ja') || (top.g_lang_suffix == 'ch'))) {
        out = '<div class="lines" id="l1"><span id="s1">&ensp;</span></div>';
      }
      out = out + '<div class="lines" id="l2"><span id="s2">&nbsp;</span></div>';
      data.$this.html(out);
      data.charh = $('#l2', data.$this).position().top - $('#l1', data.$this).position().top;
      data.charw = $('#s1', data.$this).outerWidth();
      resize_font_enabled = true; // ok to resize font later
    }
    var rows = data.winh / data.charh;
    var cols = data.winw / data.charw;
    rows = Math.ceil(rows) - 4; // avoid vert scroll bars
    if (top.g_msie6) {
      rows--; // for PanelView
    }
    cols = Math.ceil(cols) - 6; // avoid horz scroll bars

    // Check the limits
    if (rows < 5) {
      rows = 5;
    }
    else if (rows > 49) { // wdlimits.hc is 50, subtract 1 for prompt
      rows = 49;
    }
    if ((cols % 2) > 0) {
      cols--;
    }
    if (cols < 40) {
      // Change the font size to fit at least 40 chars
      // font_size of 32 (zoom) is expected to have horizontal scroll bars
      // but font_size of 24 should not
      var font_size = parseInt(data.$this.css('font-size'));
      if (font_size == 24) {
        // normally this is bold
        data.$this.css('font-weight', 'normal');
        if (resize_font_enabled) {
          data.charw = $('#s1', data.$this).outerWidth();
          cols = Math.ceil(data.winw / data.charw) - 6;
          while ((cols > 0) && (cols < 40)) {
            // try resizing the font
            font_size--;
            data.$this.css('font-size', font_size + 'px');
            data.charw = $('#s1', data.$this).outerWidth();
            cols = Math.ceil(data.winw / data.charw) - 6;
          }
          data.charh = $('#l2', data.$this).position().top - $('#l1', data.$this).position().top;
          rows = Math.ceil(data.winh / data.charh) - 4;
          if (top.g_msie6) {
            rows--; // for PanelView
          }
          // Check the limits
          if (rows < 5) {
            rows = 5;
          }
          else if (rows > 49) { // wdlimits.hc is 50, subtract 1 for prompt
            rows = 49;
          }
        }
      }
      cols = 40;
    }
    else if (cols > 80) {
      cols = 80;
    }
    if ((resize_font_enabled) && (data.resize_font) && ((rows != data.resize_rows) || (cols != data.resize_cols))) {
      // Change the font size to fit the rows and cols
      var font_size = parseInt(data.$this.css('font-size'));
      if ((font_size == 24) && ((rows < data.resize_rows) || (cols < data.resize_cols))) {
        // normally this is bold
        data.$this.css('font-weight', 'normal');
        data.charh = $('#l2', data.$this).position().top - $('#l1', data.$this).position().top;
        rows = Math.ceil(data.winh / data.charh) - 4;
        data.charw = $('#s1', data.$this).outerWidth();
        cols = Math.ceil(data.winw / data.charw) - 6;
      }
      while ((rows > data.resize_rows) && (cols > data.resize_cols)) {
        // try increasing the font for better reading
        font_size++;
        data.$this.css('font-size', font_size + 'px');
        data.charh = $('#l2', data.$this).position().top - $('#l1', data.$this).position().top;
        rows = Math.ceil(data.winh / data.charh) - 4;
        data.charw = $('#s1', data.$this).outerWidth();
        cols = Math.ceil(data.winw / data.charw) - 6;
      }
      while ((rows < data.resize_rows) && (cols < data.resize_cols)) {
        // try decreasing the font until it fits
        font_size--;
        data.$this.css('font-size', font_size + 'px');
        data.charh = $('#l2', data.$this).position().top - $('#l1', data.$this).position().top;
        rows = Math.ceil(data.winh / data.charh) - 4;
        data.charw = $('#s1', data.$this).outerWidth();
        cols = Math.ceil(data.winw / data.charw) - 6;
      }
      rows = data.resize_rows;
      cols = data.resize_cols;

      // Check the limits again
      if (rows < 5) {
        rows = 5;
      }
      else if (rows > 49) { // wdlimits.hc is 50, subtract 1 for prompt
        rows = 49;
      }
      if ((cols % 2) > 0) {
        cols--;
      }
      if (cols < 40) {
        cols = 40;
      }
      else if (cols > 80) {
        cols = 80;
      }
    }
    if ((data.lines != rows) || (data.columns != cols)) {
      if (top.g_connect_id) {
        if (top.g_android && ((top.g_lang_suffix == 'ja') || (top.g_lang_suffix == 'ch')) && (cols < 46)) {
          cols = 40;
        }
        // Send "window configuration" external request to GUID task.
        top.rpcmc_tpextreq3(data.device_id, tp_winsize_c, rows, cols);
      }

      // Create or remove lines inside the device.
      out = '&nbsp;'; // for some whitespace
      if (rows > data.lines) {
        for (var idx = 1; idx <= rows; idx++) {
          out = out + '<div class="lines" id="l' + idx + '" />';
        }
        data.$this.html(out);

        // Attach handler to lines class for mouseup event.
        $('.lines', data.$this).mouseup(function(event) {
          // Send device onMouseUp event to controller.
          var col = Math.ceil(event.pageX / data.charw) - 1;
          var idstr = $(this).attr('id');
          row = parseInt(idstr.substr(1));
          top.rpcmc_tpextreq2(data.device_id, tp_devonmouseup_c, ((MOUSE_LEFT << 16) | (row << 8) | col));
        });

        // Attach handler to lines class for dblclick event.
        $('.lines', data.$this).dblclick(function(event) {
          // Send device onMouseUp event to controller.
          var col = Math.ceil(event.pageX / data.charw) - 1;
          var idstr = $(this).attr('id');
          row = parseInt(idstr.substr(1));
          top.rpcmc_tpextreq2(data.device_id, tp_devonmouseup_c, (((MOUSE_LEFT + MOUSE_DBLCLK) << 16) | (row << 8) | col));
        });

        // Attach handler for mouseup event.
        $(document).mouseup(function(event) {
          // Send device onMouseUp event to controller.
          var row = Math.ceil(event.pageY / data.charh) - 1;
          var col = Math.ceil(event.pageX / data.charw) - 1;
          if (row < 1) {
            row = 1; // page up, row = 0 doesn't work
            top.rpcmc_tpextreq2(data.device_id, tp_devonmouseup_c, ((MOUSE_LEFT << 16) | (row << 8) | col));
          }
          else if (row > data.lines) {
            row = data.lines + 1; // page down
            top.rpcmc_tpextreq2(data.device_id, tp_devonmouseup_c, ((MOUSE_LEFT << 16) | (row << 8) | col));
          }
        });
      }
      else {
        for (var idx = rows + 1; idx <= data.lines; idx++) {
          $('#l' + idx, data.$this).remove();
        }
      }
      data.lines = rows;
      data.columns = cols;
    }
    if ((resize_font_enabled) && (data.resize_font)) {
      // One time only
      data.resize_font = false;
    }

  } // resize_dev

})(jQuery);
