/**
 * @fileoverview JavaScript Library: This file is used to resize window.
 */
/*jsl:import ihmi_cookie.js*/

/**
 * Resize a page based on cookie or use the default size.
 * Should only be called once on onload event.
 *
 * The cookie looks something like this:
 *  format is width,height,x,y
 * @requires ihmi_cookie.js
 */
function initPageSize(frame, name) {
  var timerId = null;
  var defX = 0;
  var defY = 0;
  var defW = 0;
  var defH = 0;
  var deltaW = 0;
  var deltaH = 0;
  var deltaX = 0;
  var deltaY = 0;
  var pageName;
  var ieVersion;
  var isModernBrowser = true; /* since ie10 */
  
  function getIEVersion (userAgent) {
    var result = userAgent.match(/MSIE +([\d.]+);/);
    if (result === null) {
      result = userAgent.match(/rv:([\d.]+)/);
    }
    if (result !== null) {
      return parseFloat(result[1]);
    }
    return -1;
  }

  function addEventHandler (target, event, handler) {
    if (target.attachEvent) {
      target.attachEvent('on' + event, handler);
    }
    else {
      target.addEventListener(event, handler);
    }
  }
  
  ieVersion = getIEVersion(window.navigator.userAgent);
  
  if (frame == null) return;
  
  var isTabret = (navigator.userAgent.indexOf("Android") >= 0) ||
                 (navigator.userAgent.indexOf("iPad") >= 0) ||
                 (navigator.userAgent.indexOf("iPhone") >= 0) ||
                 (navigator.userAgent.indexOf("iPod") >= 0);
                 
  if (isTabret == true) {
    //not worked
    return;
  }

  // In case of IE version is 10 or more, "document.hidden","document.outerWidth","document.outerHeight" can be used.
  if ((0 < ieVersion) && (ieVersion < 10)) {
    isModernBrowser = false;
  }
  
  pageName = (name !== undefined) ? name : frame.webpage;

  // onmove event does not work so call onmove every second. */
  frame.setInterval(onmove, 1000);
  addEventHandler(frame, "resize", onresize);

  var windowValues = [];
  var found = getSizeFromCookie(windowValues);
  if (found) {
    defW = windowValues[0];
    defH = windowValues[1];
    defX = windowValues[2];
    defY = windowValues[3];
  }
  else {
    if (isModernBrowser) {
      defW = frame.outerWidth;
      defH = frame.outerHeight;
      defX = frame.screenX;
      defY = frame.screenY;
    }
    else {
      defW = frame.document.documentElement.clientWidth;
      defH = frame.document.documentElement.clientHeight;
      defX = frame.screenLeft;
      defY = frame.screenTop;
    }
  }
  
  if (defW < 0) defW = 1028;
  if (defH < 0) defH = 746;
  if (defX < 0) defX = 0;
  if (defY < 0) defY = 0;

  try {
    frame.moveTo(defX, defY);
    frame.resizeTo(defW, defH);
    frame.moveTo(defX, defY);
  }
  catch(e) {
    return;
  }
  
  if (isModernBrowser === true) {
    // Save for onmove().
    defX = frame.screenX;
    defY = frame.screenY;
  }
  else {
    // Get deltas so window size can remain consistent.
    deltaW = defW - frame.document.documentElement.clientWidth;
    deltaH = defH - frame.document.documentElement.clientHeight;
    deltaX = defX - frame.screenLeft;
    deltaY = defY - frame.screenTop;
    
    // Save for onmove().
    defX = frame.screenLeft;
    defY = frame.screenTop;
  }
  function getSizeFromCookie(windowValues) {
    var str = Cookie.get(pageName + ':winsize');
    var dims, dimIndex;

    if (str != null) {
      dims = str.split(',');
      if (dims.length == 4) {
        for (dimIndex = 0; dimIndex < 4; dimIndex++) {
          windowValues[dimIndex] = parseInt(dims[dimIndex], 10);
        }
        return true;
      }
    }
    return false;
  }

  function saveSizeToCookie() {
    var newW;
    var newH;
    var newX;
    var newY;
    // Use an expiration a year from now.
    var expires = new Date();
    expires.setFullYear(expires.getFullYear() + 1);
    
    if (isModernBrowser === true) {
      if(frame.document.hidden == true){  // minimize a window.
        return true;
      }
      defX = frame.screenX;
      defY = frame.screenY;
      newW = frame.outerWidth;
      newH = frame.outerHeight;
      newX = defX;
      newY = defY;
    }
    else {
      defX = frame.screenLeft;
      defY = frame.screenTop;
      newW = frame.document.documentElement.clientWidth + deltaW;
      newH = frame.document.documentElement.clientHeight + deltaH;
      newX = defX + deltaX;
      newY = defY + deltaY;
    }
    var newstr = [newW, newH, newX, newY].join(',');
    Cookie.store(pageName + ':winsize', newstr, expires);
    return true;
  }

  /**
   * Fired by the onresize event. Save the resize after 1/2 second.
   */
  function onresize() {
    frame.clearTimeout(timerId);
    timerId = frame.setTimeout(saveSizeToCookie, 500);
  }

  /**
   * Since onmove event is not working, call onmove instead to determine
   * if window has moved.
   */
  function onmove() {

    if (isModernBrowser === true) {
      if ((frame.screenX != defX) || (frame.screenY != defY)) {
        saveSizeToCookie();
      }
    }
    else {
      if ((frame.screenLeft != defX) || (frame.screenTop != defY)) {
        saveSizeToCookie();
      }
    }
  }
}
